use web3::types::{H160, U256, U64};
use serde::Deserialize;

use super::{BscChainApi, Response};
use crate::{error::CustomErrors};

pub struct Blocks;

#[derive(Debug, Deserialize, Clone)]
pub struct BlockReward {
    #[serde(rename = "blockNumber")]
    block_no: String,
    #[serde(rename = "timeStamp")]
    timestamp: String,
    #[serde(rename = "blockMiner")]
    block_miner: H160,
    #[serde(rename = "blockReward")]
    block_reward: String,
    uncles: Vec<U64>,
    #[serde(rename = "uncleInclusionReward")]
    uncle_inclusive_reward: String,
}

impl BlockReward {
    pub fn block_no(&self) -> U64 {
        U64::from_dec_str(&self.block_no).unwrap()
    }

    pub fn timestamp(&self) -> U256 {
        U256::from_dec_str(&self.timestamp).unwrap()
    }

    pub fn block_miner(&self) -> H160 {
        self.block_miner
    }

    pub fn block_reward(&self) -> U256 {
        U256::from_dec_str(&self.block_reward).unwrap()
    }

    pub fn uncles(&self) -> Vec<U64> {
        self.uncles.clone()
    }

    pub fn uncle_inclusive_reward(&self) -> U256 {
        U256::from_dec_str(&self.uncle_inclusive_reward).unwrap()
    }
}

impl Response<BlockReward> {
    pub async fn parse_str(response: String) -> Result<BlockReward, CustomErrors> {
        Ok(serde_json::from_str::<Response<BlockReward>>(&response)
            .unwrap()
            .result()?)
    }
}

#[derive(Debug, Deserialize, Clone)]
pub struct BlockData {
    #[serde(rename = "CurrentBlock")]
    current_block: String,
    #[serde(rename = "CountdownBlock")]
    countdown_block: String,
    #[serde(rename = "RemainingBlock")]
    remaining_block: String,
    #[serde(rename = "EstimateTimeInSec")]
    estimate_time: String,
}

impl BlockData {
    pub fn current_block(&self) -> U64 {
        U64::from_dec_str(&self.current_block).unwrap()
    }

    pub fn countdown_block(&self) -> U64 {
        U64::from_dec_str(&self.countdown_block).unwrap()
    }

    pub fn remaining_block(&self) -> U64 {
        U64::from_dec_str(&self.remaining_block).unwrap()
    }

    pub fn estimate_time_in_secs(&self) -> f64 {
        self.estimate_time.parse::<f64>().unwrap()
    }
}

impl Response<BlockData> {
    pub async fn parse_str(response: String) -> Result<BlockData, CustomErrors> {
        Ok(serde_json::from_str::<Response<BlockData>>(&response)
            .unwrap()
            .result()?)
    }
}

impl Response<U256> {
    pub async fn get_timestamp(response: String) -> Result<U256,CustomErrors> {
        let resp = serde_json::from_str::<Response<String>>(&response)
            .unwrap()
            .result();
        Ok(U256::from_dec_str(&resp?).unwrap())
    }
}

impl Blocks {
    pub async fn get_block_by_timestamp(
        &self,
        api: &mut BscChainApi,
        timestamp: i64,
        before: bool,
    ) -> Result<U256, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "block");
        api.query.add_params("action", "getblocknobytime");
        api.query.add_params("timestamp", &timestamp.to_string());
        api.query
            .add_params("closest", if before { "before" } else { "after" });

        Ok(Response::<U256>::get_timestamp(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?)
    }

    pub async fn get_block_and_uncle_rewards(
        &self,
        api: &mut BscChainApi,
        block_no: u32,
    ) -> Result<BlockReward, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "block");
        api.query.add_params("action", "getblockreward");
        api.query.add_params("blockno", &block_no.to_string());

        Ok(Response::<BlockReward>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?)
    }

    pub async fn get_estimated_block_time(
        &self,
        api: &mut BscChainApi,
        block_no: u32,
    ) -> Result<BlockData, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "block");
        api.query.add_params("action", "getblockcountdown");
        api.query.add_params("blockno", &block_no.to_string());

        Ok(Response::<BlockData>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?)
    }
}

#[cfg(test)]
mod test {
    use super::*;

    fn create_success() -> BscChainApi {
        BscChainApi::new("YOUR_API_KEY_HERE")
    }

    #[actix_rt::test]
    async fn get_blockno_by_timestamp() {
        let mut api = create_success();
        assert_eq!(
            Blocks.get_block_by_timestamp(&mut api, 1605122780, true).await.unwrap(),
            U256::from(2150000)
        );
    }

    #[actix_rt::test]
    async fn get_block_reward() {
        let mut api = create_success();
        assert_eq!(
            Blocks.get_block_and_uncle_rewards(&mut api,
                7302556,)
                .await
                .unwrap()
                .timestamp(),
            U256::from(1620670493u128)
        );
    }

    #[actix_rt::test]
    async fn get_clock_estimate() {
        let mut api = create_success();
        assert_eq!(
            Blocks.get_estimated_block_time(&mut api, 8000000)
                .await
                .unwrap()
                .countdown_block(),
            U64::from(8000000)
        );
    }
}
