use serde::Deserialize;
use serde_json;
use web3::types::{H160, H256, U256, U64};

use super::{BscChainApi, Response};
use crate::error::CustomErrors;

#[derive(Debug, Deserialize, Clone)]
pub struct Balance {
    account: H160,
    balance: String,
}

pub struct Account;

impl Balance {
    pub fn get_balace(&self) -> U256 {
        U256::from_dec_str(&self.balance).unwrap()
    }

    pub fn account(&self) -> H160 {
        self.account
    }
}

impl Response<U256> {
    pub async fn parse_str(response: String) -> Result<U256, CustomErrors> {
        let resp = serde_json::from_str::<Response<String>>(&response).unwrap();
        Ok(U256::from_dec_str(&resp.result()?).unwrap())
    }
}

impl Response<Vec<Balance>> {
    pub async fn parse_str(response: String) -> Response<Vec<Balance>> {
        serde_json::from_str::<Response<Vec<Balance>>>(&response).unwrap()
    }
}

#[derive(Debug, Deserialize, Clone)]
pub struct TokenTx {
    #[serde(rename = "blockNumber")]
    block_no: String,
    #[serde(rename = "timeStamp")]
    timestamp: String,
    hash: H256,
    nonce: String,
    #[serde(rename = "blockHash")]
    block_hash: H256,
    from: H160,
    to: H160,
    #[serde(default)]
    value: String,
    #[serde(rename = "contractAddress")]
    contract_address: H160,
    #[serde(default)]
    #[serde(rename = "tokenID")]
    token_id: String,
    #[serde(rename = "tokenName")]
    token_name: String,
    #[serde(rename = "tokenSymbol")]
    token_symbol: String,
    #[serde(rename = "tokenDecimal")]
    token_decimal: String,
    gas: String,
    #[serde(rename = "gasPrice")]
    gas_price: String,
    #[serde(default)]
    #[serde(rename = "transactionIndex")]
    transaction_index: String,
    #[serde(rename = "cumulativeGasUsed")]
    cumulative_gas_used: String,
    #[serde(rename = "gasUsed")]
    gas_used: String,
    confirmations: String,
}

impl TokenTx {
    pub fn block_no(&self) -> U64 {
        U64::from_dec_str(&self.block_no).unwrap_or_default()
    }

    pub fn timestamp(&self) -> U256 {
        U256::from_dec_str(&self.block_no).unwrap_or_default()
    }

    pub fn hash(&self) -> H256 {
        self.hash
    }

    pub fn nonce(&self) -> U256 {
        U256::from_dec_str(&self.block_no).unwrap_or_default()
    }

    pub fn block_hash(&self) -> H256 {
        self.block_hash
    }

    pub fn from(&self) -> H160 {
        self.from
    }

    pub fn to(&self) -> H160 {
        self.to
    }

    pub fn value(&self) -> Option<U256> {
        if self.value.is_empty() {
            None
        } else {
            Some(U256::from_dec_str(&self.value).unwrap_or_default())
        }
    }

    pub fn contract_address(&self) -> H160 {
        self.contract_address
    }

    pub fn token_id(&self) -> String {
        self.token_id.clone()
    }

    pub fn token_symbol(&self) -> String {
        self.token_symbol.clone()
    }

    pub fn token_decimal(&self) -> u16 {
        self.token_decimal.parse::<u16>().unwrap_or(0)
    }

    pub fn token_name(&self) -> String {
        self.token_name.clone()
    }

    pub fn gas(&self) -> U256 {
        U256::from_dec_str(&self.gas).unwrap_or_default()
    }

    pub fn gas_price(&self) -> U256 {
        U256::from_dec_str(&self.gas_price).unwrap_or_default()
    }

    pub fn gas_used(&self) -> U256 {
        U256::from_dec_str(&self.gas_used).unwrap_or_default()
    }

    pub fn transaction_index(&self) -> u16 {
        if self.transaction_index.is_empty() {
            0
        } else {
            self.transaction_index.parse::<u16>().unwrap()
        }
    }

    pub fn cumulative_gas_used(&self) -> U256 {
        U256::from_dec_str(&self.cumulative_gas_used).unwrap_or_default()
    }

    pub fn confirmations(&self) -> usize {
        self.confirmations.parse::<usize>().unwrap_or(0)
    }
}

impl Response<Vec<TokenTx>> {
    pub async fn parse_str(response: String) -> Response<Vec<TokenTx>> {
        serde_json::from_str::<Response<Vec<TokenTx>>>(&response).unwrap()
    }
}

impl Account {
    pub async fn balance(
        &self,
        api: &mut BscChainApi,
        address: &str,
    ) -> Result<U256, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "balance");
        api.query.add_params("tag", "latest");
        api.query.add_params("address", address);

        Ok(Response::<U256>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?)
    }

    pub async fn multi_balance(
        &self,
        api: &mut BscChainApi,
        accounts: Vec<&str>,
    ) -> Result<Response<Vec<Balance>>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "balancemulti");
        api.query.add_params("tag", "latest");
        api.query.multi_params("address", accounts);

        Ok(Response::<Vec<Balance>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await)
    }

    

    pub async fn bnb_balace_by_block_no(
        &self,
        api: &mut BscChainApi,
        address: &str,
        block_no: u32,
    ) -> Result<U256, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "balancehistory");
        api.query.add_params("address", address);
        api.query.add_params("blockno", &block_no.to_string());

        Ok(Response::<U256>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await?)
    }

    pub async fn bep_20_transactions(
        &self,
        api: &mut BscChainApi,
        address: &str,
        sort: bool,
    ) -> Result<Vec<TokenTx>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "tokentx");
        api.query.add_params("address", address);
        api.query.add_params("startblock", "0");
        api.query.add_params("endblock", "latest");
        api.query
            .add_params("sort", if sort { "asc" } else { "dsc" });

        Ok(Response::<Vec<TokenTx>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await
        .result()?)
    }

    pub async fn bep_20_transactions_pagination(
        &self,
        api: &mut BscChainApi,
        address: &str,
        sort: bool,
        page: u16,
        offest: u32,
    ) -> Result<Vec<TokenTx>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "tokentx");
        api.query.add_params("address", address);
        api.query.add_params("page", &page.to_string());
        api.query.add_params("offset", &offest.to_string());
        api.query
            .add_params("sort", if sort { "asc" } else { "dsc" });

        Ok(Response::<Vec<TokenTx>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await
        .result()?)
    }

    pub async fn bep_20_token_transaction_pagination(
        &self,
        api: &mut BscChainApi,
        address: &str,
        contract_address: &str,
        sort: bool,
        page: u16,
        offest: u32,
    ) -> Result<Vec<TokenTx>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "tokentx");
        api.query.add_params("address", address);
        api.query.add_params("contractaddress", contract_address);

        api.query.add_params("page", &page.to_string());
        api.query.add_params("offset", &offest.to_string());
        api.query
            .add_params("sort", if sort { "asc" } else { "dsc" });

        Ok(Response::<Vec<TokenTx>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await
        .result()?)
    }

    pub async fn erc_721_transactions_pagination(
        &self,
        api: &mut BscChainApi,
        address: &str,
        sort: bool,
        page: u16,
        offest: u32,
    ) -> Result<Vec<TokenTx>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "tokennfttx");
        api.query.add_params("address", address);

        api.query.add_params("page", &page.to_string());
        api.query.add_params("offset", &offest.to_string());
        api.query
            .add_params("sort", if sort { "asc" } else { "dsc" });

        Ok(Response::<Vec<TokenTx>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await
        .result()?)
    }

    pub async fn erc_721_transactions(
        self,
        api: &mut BscChainApi,
        address: &str,
        sort: bool,
    ) -> Result<Vec<TokenTx>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "tokennfttx");
        api.query.add_params("address", address);

        api.query.add_params("startblock", "0");
        api.query.add_params("endblock", "latest");
        api.query
            .add_params("sort", if sort { "asc" } else { "dsc" });

        Ok(Response::<Vec<TokenTx>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await
        .result()?)
    }

    pub async fn erc_721_token_transactions_pagination(
        &self,
        api: &mut BscChainApi,
        address: &str,
        contract_address: &str,
        sort: bool,
        page: u16,
        offest: u32,
    ) -> Result<Vec<TokenTx>, CustomErrors> {
        api.query.add_params("apikey", &api.api_key);
        api.query.add_params("module", "account");
        api.query.add_params("action", "tokennfttx");
        api.query.add_params("address", address);
        api.query.add_params("contractaddress", contract_address);

        api.query.add_params("page", &page.to_string());
        api.query.add_params("offset", &offest.to_string());
        api.query
            .add_params("sort", if sort { "asc" } else { "dsc" });

        Ok(Response::<Vec<TokenTx>>::parse_str(
            api.client
                .get(&api.query.build_url())
                .send()
                .await?
                .text()
                .await?,
        )
        .await
        .result()?)
    }
}

#[cfg(test)]
mod test {
    use super::*;

    fn create_success() -> BscChainApi {
        BscChainApi::new("YOUR_API_KEY_HERE")
    }

    #[actix_rt::test]
    async fn test_balance() {
        let mut api = create_success();
        let m = Account
            .balance(&mut api, "0xdB13FCA9C10805c39683FF8b2642648C573d1437")
            .await;
        assert_eq!(m.unwrap(), U256::from(186272507868818958u128));
    }

    

    // Could not test as it is a pro features
    #[actix_rt::test]
    async fn bnb_by_block() {
        let mut api = create_success();
        let m = Account
            .bnb_balace_by_block_no(
                &mut api,
                "0x0000000000000000000000000000000000001004",
                99999,
            )
            .await;
        assert_eq!(m.unwrap(), U256::from(00u128));
    }

    #[actix_rt::test]
    async fn bep_20_transactions() {
        let mut api = create_success();
        let m = Account
            .bep_20_transactions(&mut api, "0x7bb89460599dbf32ee3aa50798bbceae2a5f7f6a", true)
            .await
            .unwrap();
        assert_eq!(m[0].token_symbol(), "BUNNY")
    }

    #[actix_rt::test]
    async fn bep20_transactions_pag() {
        let mut api = create_success();
        let m = Account
            .bep_20_transactions_pagination(
                &mut api,
                "0x7bb89460599dbf32ee3aa50798bbceae2a5f7f6a",
                true,
                1,
                10,
            )
            .await
            .unwrap();
        assert_eq!(m[0].token_symbol(), "BUNNY")
    }

    #[actix_rt::test]
    async fn bep20_transactions_token() {
        let mut api = create_success();
        let m = Account
            .bep_20_token_transaction_pagination(
                &mut api,
                "0x7bb89460599dbf32ee3aa50798bbceae2a5f7f6a",
                "0xc9849e6fdb743d08faee3e34dd2d1bc69ea11a51",
                true,
                1,
                10,
            )
            .await
            .unwrap();
        assert_eq!(m[0].transaction_index(), 2);
    }

    #[actix_rt::test]
    async fn erc_721_transactions_pagination() {
        let mut api = create_success();
        let m = Account
            .erc_721_transactions_pagination(
                &mut api,
                "0xcd4ee0a77e09afa8d5a6518f7cf8539bef684e6c",
                true,
                1,
                10,
            )
            .await
            .unwrap();
        assert_eq!(m[0].token_symbol(), "PLT")
    }

    #[actix_rt::test]
    async fn erc_721_specific_token() {
        let mut api = create_success();
        let m = Account
            .erc_721_token_transactions_pagination(
                &mut api,
                "0xcd4ee0a77e09afa8d5a6518f7cf8539bef684e6c",
                "0x5e74094cd416f55179dbd0e45b1a8ed030e396a1",
                true,
                1,
                10,
            )
            .await
            .unwrap();
        assert_eq!(m[0].token_symbol(), "PLT")
    }
}
