//! This module contains all window related functionality.

mod builder;

use super::prelude::*;
use super::event::Event;

use browser_window_core::prelude::*;



pub use builder::WindowBuilder;



pub type StandardWindowEvent = Event<'static, WindowHandle>;

/// A handle that exposes all windowing functionality.
#[derive(Clone, Copy)]
pub struct WindowHandle {
	pub(in super) inner: WindowImpl
}

#[derive(Default)]
pub(in crate) struct WindowEvents {
	pub on_close: StandardWindowEvent,
	pub on_destroy: StandardWindowEvent,
	pub on_resize: Event<'static, WindowResizeEventArgs>
}

pub struct WindowResizeEventArgs {
	handle: WindowHandle,
	new_size: Dims2D
}

pub trait OwnedWindow {
	fn window_handle( &self ) -> WindowHandle;
}



impl WindowHandle {
	impl_prop!{ pub content_dimensions: ContentDimensions }
	impl_prop!{ pub opacity: Opacity }
	impl_prop!{ pub position: Position }
	impl_prop!{ pub title: Title }
	impl_prop!{ pub window_dimensions: WindowDimensions }

	/// Destroys the window.
	pub fn close( self ) {
		self.hide();
		// The window will be dropped because ownership of `self` is taken.
	}

	/// Make the window invisible to the user.
	pub fn hide( &self ) {
		self.inner.hide()
	}

	pub(in super) fn new( inner: WindowImpl ) -> Self {
		Self {
			inner
		}
	}

	/// Make the window visible to the user.
	pub fn show( &self ) {
		self.inner.show()
	}
}



prop! { /// Gets or sets the width and height of the content of the window.
        /// Setting this property only works on Windows.
        /// This will be fixed once CEF works with GTK3 correctly.
	ContentDimensions<Dims2D>( this: WindowHandle ) {
		get => this.inner.get_content_dimensions().into(),
		set(val) => this.inner.set_content_dimensions( val.into() )
	}
}

prop! { /// Gets or sets the opacity of the window.
        /// An opacity of 255 means the window is invisible.
        /// An opacity of 0 means the window is completely visible.
        /// Anything in between makes the window transparent.
        /// 
        /// This feature only works on Windows since CEF currently does not work with GTK3 correctly.
	Opacity<u8>( this: WindowHandle ) {
		get => this.inner.get_opacity(),
		set(val) => this.inner.set_opacity( val )
	}
}

prop! { /// Gets or sets the current position of the window.
        /// 
        /// This feature only works on Windows since CEF currently does not work with GTK3 correctly.
        /// On non-Windows platforms the getting the position still works, but relates to the position of the browser inside the window, instead of the window itself.
	Position<Pos2D>( this: WindowHandle ) {
		get => this.inner.get_position(),
		set(val) => this.inner.set_position( val )
	}
}

prop!{ /// Gets or sets the title of the window.
	pub Title<String, &str>( this: WindowHandle ) {
		get => this.inner.get_title(),
		set(val) => this.inner.set_title( val ).into()
	}
}

prop! { /// Gets or sets the current window size including its border and titlebar.
        /// 
        /// This feature only works on Windows since CEF currently does not work with GTK3 correctly.
	WindowDimensions<Dims2D>( this: WindowHandle ) {
		get => this.inner.get_window_dimensions().into(),
		set(val) => this.inner.set_window_dimensions( val.into() )
	}
}
