use crate::OutputFormat;
use anyhow::Result;
use brevdash_data as data;
use std::path::Path;
use structopt::StructOpt;

#[derive(Debug, StructOpt)]
pub struct Args {
    /// The entity which should be shown
    #[structopt(subcommand)]
    command: Command,

    /// The output format (default: toml)
    #[structopt(long = "format")]
    output_format: Option<OutputFormat>,
}

impl Args {
    pub fn process(&self, working_directory: &Path) -> Result<()> {
        match &self.command {
            Command::Project(arg) => {
                arg.process(working_directory, self.output_format)
            }
        }
    }
}

#[derive(Debug, StructOpt)]
pub enum Command {
    /// Show a project
    Project(ProjectArgs),
}

#[derive(Debug, StructOpt)]
pub struct ProjectArgs {
    project_id: String,
}

impl ProjectArgs {
    pub fn process(
        &self,
        working_directory: &Path,
        output_format: Option<OutputFormat>,
    ) -> Result<()> {
        let repo = data::Repository::open(working_directory)?;

        let description =
            repo.load_project_description(&self.project_id)?;
        match output_format.unwrap_or_default() {
            OutputFormat::Toml => {
                println!("{}", toml::to_string_pretty(&description)?)
            }
            OutputFormat::Json => {
                println!(
                    "{}",
                    serde_json::to_string_pretty(&description)?
                );
            }
        }

        Ok(())
    }
}
