#![deny(
    missing_debug_implementations,
    missing_copy_implementations,
    trivial_casts,
    trivial_numeric_casts,
    unsafe_code,
    unstable_features,
    unused_import_braces,
    unused_qualifications
)]

use anyhow::{bail, Result};
use brevdash_data as data;
use chrono::naive::NaiveDate;
use log::debug;
use serde::{Deserialize, Serialize};
use std::collections::BTreeMap;
use std::path::{Path, PathBuf};
use structopt::StructOpt;

mod add;
mod init;
mod list;
mod report;
mod show;

const PROGRAM_VERSION: &str = env!("CARGO_PKG_VERSION");
const PROGRAM_NAME: &str = env!("CARGO_PKG_NAME");
const PROGRAM_HOMEPAGE: &str = env!("CARGO_PKG_HOMEPAGE");

#[derive(Serialize, Deserialize, PartialEq, Eq, Clone, Copy, Debug)]
pub enum OutputFormat {
    Toml,
    Json,
}

impl Default for OutputFormat {
    fn default() -> Self {
        Self::Toml
    }
}

impl std::str::FromStr for OutputFormat {
    type Err = anyhow::Error;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s {
            "toml" => Ok(OutputFormat::Toml),
            "json" => Ok(OutputFormat::Json),
            _ => bail!("Couldn't parse output format {:?}", s),
        }
    }
}

#[derive(Debug, StructOpt)]
enum Command {
    /// Generate a report
    Report(report::Args),

    /// Initialize a brevdash repository
    Init(init::Args),

    /// Add data to the repository
    Add(add::Command),

    /// List data from the repository
    List(list::Command),

    /// Show data from the repository
    Show(show::Args),
}

#[derive(Debug, StructOpt)]
struct Opt {
    /// Run as if the command was started in this directory instead of the
    /// current working directory.
    #[structopt(long, short = "C", name = "path")]
    change_directory: Option<PathBuf>,

    /// The subcommand
    #[structopt(subcommand)]
    command: Command,
}

impl Opt {
    fn process(&self) -> Result<()> {
        let working_directory = self
            .change_directory
            .clone()
            .unwrap_or_else(|| Path::new(".").to_path_buf());

        match &self.command {
            Command::Report(opt) => opt.process(&working_directory),
            Command::Init(opt) => opt.process(&working_directory),
            Command::Add(opt) => opt.process(&working_directory),
            Command::List(opt) => opt.process(&working_directory),
            Command::Show(opt) => opt.process(&working_directory),
        }
    }
}

#[derive(Debug)]
struct Project {
    description: data::ProjectDescription,
    datapoints: BTreeMap<NaiveDate, data::DataPoint>,
}

fn main() -> Result<()> {
    env_logger::init();

    let opt = Opt::from_args();

    debug!("Parameters: {:#?}", opt);

    opt.process()
}
