use std::error::Error as StdErr;
use std::{fmt, result};

/// The type of Branca errors that can occur when encoding or decoding Branca tokens.
#[derive(Debug, Eq, PartialEq, Copy, Clone)]
pub enum Error {
    /// When the given input is not a valid Base62 encoding.
    InvalidBase62Token,
    /// When the version of the Branca token is mismatched.
    InvalidTokenVersion,
    /// When the nonce is of an incorrect size.
    BadNonceLength,
    /// When the key is of an incorrect size.
    BadKeyLength,
    /// When the token has exceeded its TTL.
    ExpiredToken,
    /// When adding TTL to token timestamp overflows.
    OverflowingOperation,
    /// When the decryption of the ciphertext has failed.
    DecryptFailed,
    /// When encryption of the plaintext has failed.
    EncryptFailed,
}

impl fmt::Display for Error {
    fn fmt(&self, fmt: &mut fmt::Formatter) -> fmt::Result {
        write!(fmt, "{:?}", self)
    }
}

impl StdErr for Error {
    fn description(&self) -> &str {
        match *self {
            Error::InvalidBase62Token => "Base62 token is invalid.",
            Error::InvalidTokenVersion => "Token version is invalid.",
            Error::BadNonceLength => "Bad nonce length.",
            Error::BadKeyLength => "Bad key length.",
            Error::ExpiredToken => "This token has expired.",
            Error::OverflowingOperation => "Attempting to add TTL to the timestamp overflowed.",
            Error::DecryptFailed => "Decryption failed.",
            Error::EncryptFailed => "Encryption failed.",
        }
    }
}

/// Alias for `Result<T, Error>` used in the Branca module.
pub type Result<T> = result::Result<T, Error>;
