use borsh::{ BorshDeserialize };
use solana_program::{
    account_info::{ AccountInfo },
    program_error::ProgramError,
    pubkey::Pubkey,
    borsh::try_from_slice_unchecked,
};
use std::str::FromStr;

#[derive(BorshDeserialize)]
pub struct Report {
    pub magic: u64,
    pub bump: u8,
    pub active: bool,
    pub process: Pubkey,
    pub info: ReportInfo,
}

#[derive(BorshDeserialize)]
pub struct ReportInfo {
    pub source_chain: u32,
    pub price: u64,
    pub risk: u64,
    pub time: u64,
    pub decimal: u64,
    pub program_addr: Pubkey,
    pub token_addr: Pubkey,
    pub local_addr: Pubkey, // It is solana local mint address of the nft which is transfer from other chain
    pub name: String,
    pub price_type: String,
}

fn check_owner(owner: &Pubkey) -> bool {
    let oracle_program_id = Pubkey::from_str("BorcZEiGQJAL62M9QotWAvZYGkymuVnf42mj5HYnLZQj").unwrap();
    owner.eq(&oracle_program_id)
}

pub fn get_report_info(account_info: &AccountInfo) -> Result<ReportInfo, ProgramError> {
    if !check_owner(account_info.owner) {
        return Err(ProgramError::IllegalOwner);
    }

    let report: Report = try_from_slice_unchecked(&account_info.data.borrow())?;

    if !report.active {
        // the account is deprecated
        return Err(ProgramError::InvalidAccountData);
    }


    Ok(report.info)
}