// Copyright (c) 2021, BlockProject 3D
//
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification,
// are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice,
//       this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright notice,
//       this list of conditions and the following disclaimer in the documentation
//       and/or other materials provided with the distribution.
//     * Neither the name of BlockProject 3D nor the names of its contributors
//       may be used to endorse or promote products derived from this software
//       without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

use std::{
    ops::{Index, IndexMut},
    slice::Iter,
    vec::Vec
};

use crate::sd::Value;

/// Represents a BPX Structured Data Array.
#[derive(PartialEq, Clone)]
pub struct Array
{
    data: Vec<Value>
}

impl Default for Array
{
    fn default() -> Self
    {
        Self::new()
    }
}

impl Array
{
    /// Creates a new array.
    pub fn new() -> Array
    {
        Array { data: Vec::new() }
    }

    /// Allocates a new array with a specified initial capacity
    pub fn with_capacity(capacity: usize) -> Array
    {
        Array {
            data: Vec::with_capacity(capacity)
        }
    }

    /// Adds a value at the end of the array.
    ///
    /// # Arguments
    ///
    /// * `v`: the [Value](crate::sd::Value) to add.
    ///
    /// # Examples
    ///
    /// ```
    /// use bpx::sd::Array;
    ///
    /// let mut arr = Array::new();
    /// assert_eq!(arr.len(), 0);
    /// arr.add("Test".into());
    /// assert_eq!(arr.len(), 1);
    /// ```
    pub fn add(&mut self, v: Value)
    {
        self.data.push(v);
    }

    /// Removes a value from the array.
    ///
    /// # Arguments
    ///
    /// * `pos`: the position of the item in the array to remove.
    ///
    /// # Panics
    ///
    /// Panics if pos is out of bounds.
    ///
    /// # Examples
    ///
    /// ```
    /// use bpx::sd::Array;
    ///
    /// let mut arr = Array::new();
    /// arr.add("Test".into());
    /// assert_eq!(arr.len(), 1);
    /// arr.remove_at(0);
    /// assert_eq!(arr.len(), 0);
    /// ```
    pub fn remove_at(&mut self, pos: usize) -> Option<Value>
    {
        if pos > self.data.len() {
            None
        } else {
            Some(self.data.remove(pos))
        }
    }

    /// Removes a range of values from the array.
    ///
    /// # Arguments
    ///
    /// * `item`: the [Value](crate::sd::Value) to remove.
    ///
    /// # Examples
    ///
    /// ```
    /// use bpx::sd::Array;
    ///
    /// let mut arr = Array::new();
    /// arr.add("Test".into());
    /// assert_eq!(arr.len(), 1);
    /// arr.remove("Test".into());
    /// assert_eq!(arr.len(), 0);
    /// ```
    pub fn remove(&mut self, item: Value)
    {
        for i in 0..self.data.len() {
            if self.data[i] == item {
                self.data.remove(i);
            }
        }
    }

    /// Attempts to get an item at a given position.
    /// Returns None if no value could be found at the given position.
    ///
    /// # Arguments
    ///
    /// * `pos`: the position of the item.
    ///
    /// returns: Option<&Value>
    ///
    /// # Examples
    ///
    /// ```
    /// use std::convert::TryInto;
    /// use bpx::sd::Array;
    /// use bpx::sd::Value;
    ///
    /// let mut arr = Array::new();
    /// arr.add("Test".into());
    /// assert_eq!(arr.len(), 1);
    /// assert!(arr.get(0).is_some());
    /// assert!(arr.get(0).unwrap() == &Value::from("Test"));
    /// ```
    pub fn get(&self, pos: usize) -> Option<&Value>
    {
        self.data.get(pos)
    }

    /// Returns the length of the array.
    pub fn len(&self) -> usize
    {
        self.data.len()
    }

    /// Returns true if this array is empty.
    pub fn is_empty(&self) -> bool
    {
        self.len() == 0
    }

    /// Returns an iterator to this array.
    pub fn iter(&self) -> Iter<Value>
    {
        self.data.iter()
    }
}

impl<'a> IntoIterator for &'a Array
{
    type Item = &'a Value;
    type IntoIter = Iter<'a, Value>;

    fn into_iter(self) -> Self::IntoIter
    {
        self.iter()
    }
}

impl Index<usize> for Array
{
    type Output = Value;

    fn index(&self, i: usize) -> &Value
    {
        &self.data[i]
    }
}

impl IndexMut<usize> for Array
{
    fn index_mut(&mut self, i: usize) -> &mut Value
    {
        &mut self.data[i]
    }
}
