//! A crate for automatically generating boxes.
//!
//! They are very cool, and very nice. I'm quite proud of it! This is licensed under the MPL and anyone can use it in their cool projects!
//!
//! An example of a box generated by boxer:
//! ```text
//! ▛▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▜
//! ▌  Lorem ipsum dolor sit amet, consectetu  ▐
//! ▌  r adipiscing elit, sed do eiusmod temp  ▐
//! ▌  or incididunt ut labore et dolore magn  ▐
//! ▌  a aliqua. Volutpat consequat mauris nu  ▐
//! ▌  nc congue nisi vitae. Consectetur adip  ▐
//! ▌  iscing elit pellentesque habitant. In   ▐
//! ▌  hac habitasse platea dictumst quisque.  ▐
//! ▌   Sed blandit libero volutpat sed cras   ▐
//! ▌  ornare arcu. Metus vulputate eu sceler  ▐
//! ▌  isque felis imperdiet proin fermentum   ▐
//! ▌  leo vel. Ullamcorper eget nulla facili  ▐
//! ▌  si etiam dignissim diam quis. Facilisi  ▐
//! ▌   cras fermentum odio eu feugiat. Ac au  ▐
//! ▌  ctor augue mauris augue neque gravida.  ▐
//! ▌   Ullamcorper velit sed ullamcorper mor  ▐
//! ▌  bi tincidunt ornare massa eget. Nibh n  ▐
//! ▌  isl condimentum id venenatis a condime  ▐
//! ▌  ntum vitae sapien pellentesque. Sollic  ▐
//! ▌  itudin tempor id eu nisl nunc. Diam si  ▐
//! ▌  t amet nisl suscipit. Vitae nunc sed v  ▐
//! ▌  elit dignissim sodales ut eu.           ▐
//! ▙▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▟
//! ```
use clippet::Clippet;

/// A structure for containing border information for boxes.
#[derive(Debug)]
pub struct Border {
	pub tl: String, // Top left
	pub tr: String, // Top right
	pub bl: String, // Bottom left
	pub br: String, // Bottom right
	pub sl: String, // Side left
	pub sr: String, // Side right
	pub st: String, // Side top
	pub sb: String, // Side bottom
	pub xtra: u32, // Extra padding for top and bottom
	pub over: u32, // Extra padding for sides
}

impl Border {
	/// Takes in a category and spits out a templated Border structure.
	/// Available templates include receipt, hashes, blocky and solid.
	pub fn template(cat: &str) -> Border {
		match cat {
			"empty" => Border { tl: "".to_string(), tr: "".to_string(), bl: "".to_string(), br: "".to_string(), sl: "".to_string(), sr: "".to_string(), st: "".to_string(), sb: "".to_string(), xtra: 0, over: 3 },
			"hashes" => Border { tl: "#".to_string(), tr: "#".to_string(), bl: "#".to_string(), br: "#".to_string(), sl: "#".to_string(), sr: "#".to_string(), st: "#".to_string(), sb: "#".to_string(), xtra: 0, over: 1 },
			"sshalert" | "ats" => Border { tl: "@".to_string(), tr: "@".to_string(), bl: "@".to_string(), br: "@".to_string(), sl: "@".to_string(), sr: "@".to_string(), st: "@".to_string(), sb: "@".to_string(), xtra: 0, over: 1 },
			"stars" | "receipt" => Border { tl: "*".to_string(), tr: "*".to_string(), bl: "*".to_string(), br: "*".to_string(), sl: "*".to_string(), sr: "*".to_string(), st: "*".to_string(), sb: "*".to_string(), xtra: 0, over: 1 },
			"solid" => Border { tl: "█".to_string(), tr: "█".to_string(), bl: "█".to_string(), br: "█".to_string(), sl: "█".to_string(), sr: "█".to_string(), st: "█".to_string(), sb: "█".to_string(), xtra: 0, over: 3 },
			"lines" => Border { tl: "/".to_string(), tr: "\\".to_string(), bl: "\\".to_string(), br: "/".to_string(), sl: "|".to_string(), sr: "|".to_string(), st: "-".to_string(), sb: "-".to_string(), xtra: 2, over: 3 },
			"blocky" | _ => Border { tl: "▛".to_string(), tr: "▜".to_string(), bl: "▙".to_string(), br: "▟".to_string(), sl: "▌".to_string(), sr: "▐".to_string(), st: "▀".to_string(), sb: "▄".to_string(), xtra: 0, over: 3 }, // Blocky
		}
	}
	/// Equivalent to running `template` with the empty category, a blank Border structure.
	pub fn new() -> Border {
		return Border::template("empty")
	}
}

/// Takes in cosmetic information, a Border structure and some text and turns it into a box in a String.
/// ```no_run
/// let x = Border::template("receipt");
/// let y = makebox(1, 1, 40, x, "Hello, world!");
/// println!("{}", y);
/// ```
#[allow(unused_assignments)]
pub fn makebox(spacing: u32, padding: u32, columns: u32, borders: Border, text: &str) -> String {
	let txt = text.to_string();
	let mut top = borders.tl;
	let mut bot = borders.bl;
	let mut i: i32 = spacing as i32 + columns as i32;
	let mut pad = borders.sl.to_string();
	let mut fins = "".to_string();
	let over = borders.over;

	if padding > 0 {
		while i > -1 {
			top = format!("{}{}", top, borders.st);
			bot = format!("{}{}", bot, borders.sb);
			pad = format!("{}{}", pad, " ");
			i = i - 1
		}
	} else {
		while i > -1 {
			top = format!("{}{}", top, borders.st);
			bot = format!("{}{}", bot, borders.sb);
			i = i - 1
		}
	}
	i = borders.xtra as i32;
	while i > 0 {
		top = format!("{}{}", top, borders.st);
		bot = format!("{}{}", bot, borders.sb);
		i = i - 1
	}
	top = format!("{}{}", top, borders.tr);
	bot = format!("{}{}", bot, borders.br);

	let mut pads: i32 = -1;
	fins = top;

	if padding > 0 {
		while pad.len() < (spacing + columns + 1 + over) as usize {
			pad = format!("{}{}", pad, " ");
		}
		while pad.len() > (spacing + columns + 1 + over) as usize {
			pad.pop();
		}
		pad = format!("{}{}", pad, borders.sr);
		pads = padding as i32;
	}
	while pads > 0 {
		fins = format!("{}\n{}", fins, pad);
		pads = pads - 1;
	}

	let spliz = Clippet::from_string(&txt, columns as usize - 1 as usize - spacing as usize);
	let mut columnz = 0;
	while spliz.len() > columnz {
		let x = spliz.get_as_string(columnz as usize);
		let mut y = borders.sl.to_string();
		let mut s: i32 = spacing as i32;
		while s > -1 {
			y = format!("{}{}", y, " ");
			s = s - 1
		}
		y = format!("{}{}", y, x);
		let mut s: i32 = spacing as i32;
		while s > -1 {
			y = format!("{}{}", y, " ");
			s = s - 1
		}
		while y.len() < (spacing + columns + 1 + over) as usize {
			y = format!("{}{}", y, " ");
		}
		while y.len() > (spacing + columns + 1 + over) as usize {
			y.pop();
		}
		fins = format!("{}\n{}{}", fins, y.to_string(), borders.sr.to_string());
		columnz = columnz + 1;
	}

	if padding > 0 {
		pads = padding as i32;
	}
	while pads > 0 {
		fins = format!("{}\n{}", fins, pad);
		pads = pads - 1;
	}
	fins = format!("{}\n{}", fins, bot);
	return fins
}

/// Makes a new box with the blocky borders without having to choose the spacing, padding, columns or borders.
/// ```no_run
/// let y = rush("Hello, world!");
/// println!("{}", y);
/// ```
pub fn rush(text: &str) -> String {
	let borders = Border::template("blocky");
	let x = makebox(1, 0, 40, borders, text);
	return x
}
