use std::collections::BTreeMap;
use std::fmt::{Debug, Formatter};
use std::sync::{Arc};
use anyhow::Context;

use tracing::instrument;
use bytecheck::CheckBytes;
use futures_util::StreamExt;
use rkyv::{Archive, Deserialize, Serialize};
use tokio::net::{TcpListener, TcpStream};
use tokio::io::{AsyncRead, AsyncWrite};
use tokio::sync::{broadcast, mpsc};
use tokio::sync::{Notify, RwLock};
use tokio_tungstenite::tungstenite::client::IntoClientRequest;
use tokio_tungstenite::WebSocketStream;

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
pub enum BoxcarMessage {
    // Ping(Ping),
    // Pong(Pong),
    //
    /// Request a RPC
    RpcReq(MethodInvocation),

    /// RPC Request Status
    RpcReqStatus(RpcReqStatus),

    /// Ask the server to notify the client when Slot n changes
    WatchSlot(u16),

    /// RPC Execution Status
    RpcStatus(RpcStatus),

    ///
    RpcWait(u16),

    OpResult(bool),

    /// RPC Response
    RpcResponse(RpcResponse),
    ServerError(ServerError),
}

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
pub enum RpcReqStatus {
    /// Request has been Accepted
    Accepted(u16),
    /// Request has been Rejected
    Rejected(RpcReqRejectReason),
    /// RPC is Running
    Running,
    /// RPC has finished
    Finished(Vec<u8>),
    /// RPC has resulted in an error
    Failed(String),
}

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
pub enum RpcReqRejectReason {
    NoCap,
    NoMethod,
}

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
pub enum ServerError {
    PacketDecodeError,
    UnexpectedPacketType,
    MethodReturnError,
}

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
pub struct RpcResponse {
    pub body: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
pub struct RpcStatus {
    success: bool,
}

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
pub struct MethodInvocation {
    pub method: String,
    pub body: Vec<u8>,
    pub delay: bool,
}

#[derive(Clone, Debug)]
pub struct BoxcarRequest {
    slot: u16,
    /// This is a reference to the Slot Map where returned Responses are stored.
    mapping: Arc<RwLock<BTreeMap<u16, RpcResponse>>>,
}

type ContextIdentifier = Option<u16>;

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
#[allow(dead_code)]
struct WireMessage {
    context: ContextIdentifier,
    message: BoxcarMessage,
}
impl WireMessage {
    /// Encode the Self
    #[instrument]
    pub fn encode(self) -> Vec<u8> {
        rkyv::to_bytes::<_, 256>(&self).unwrap().to_vec()
    }

    /// Attemtp to decode a buffer into Self
    #[instrument]
    pub fn decode(buf: Vec<u8>) -> anyhow::Result<Self> {
        let archive = rkyv::check_archived_root::<WireMessage>(&buf).unwrap();
        archive.deserialize(&mut rkyv::Infallible).context("")
    }
}

#[derive(Clone, Debug)]
#[allow(dead_code)]
pub struct Session {
    context: ContextIdentifier,
    // TODO is there a better way of storing this
    inbox: Arc<RwLock<Vec<BoxcarMessage>>>,

    recv_notify: Arc<Notify>,
}
impl Session {
    // #[allow(dead_code)]
    // pub async fn new(context: ContextIdentifier, endpoint: Endpoint) -> Self {
    //     Session {
    //         context,
    //         inbox: Arc::new(Default::default()),
    //         endpoint,
    //         recv_notify: Arc::new(Default::default()),
    //     }
    // }

    #[allow(dead_code)]
    async fn deliver(&self, message: BoxcarMessage) {
        self.inbox.write().await.push(message);
        self.recv_notify.notify_one();
    }

    #[allow(dead_code)]
    pub async fn send(&self, message: BoxcarMessage) -> anyhow::Result<()> {
        // self.endpoint.send(self.context, message).await
        todo!()
    }

    /// Try and consume a message, returning None if there is no message to consume
    #[allow(dead_code)]
    pub async fn try_recv(&self) -> Option<BoxcarMessage> {
        self.inbox.write().await.pop()
    }

    /// Wait for a message
    #[allow(dead_code)]
    pub async fn recv(&self) -> Option<BoxcarMessage> {
        // first, try try_recv to just pop a message off
        if let Some(msg) = self.try_recv().await {
            return Some(msg);
        }
        // otherwise, wait until the notify is triggered and then try again
        self.recv_notify.notified().await;
        self.try_recv().await
    }
}

pub(crate) type SessionStore = Arc<RwLock<BTreeMap<ContextIdentifier, Session>>>;

struct Client {
    sessions: SessionStore,
}
impl Client {
    pub async fn new<R>(address: R) -> anyhow::Result<Client> where R: IntoClientRequest + Unpin {
        let stream = tokio_tungstenite::connect_async(address).await?.0;
        let (tx, rx) = stream.split();

        let sessions: SessionStore = Arc::new(Default::default());

        let c_session = sessions.clone();
        tokio::spawn(async move {
            let mut rx = rx;
            let sessions = c_session;
            while let Some(msg) = rx.next().await {
                match msg {
                    Ok(packet) => match WireMessage::decode(packet.into_data()) {
                        Ok(message) => {
                            // establish a write handle
                            let mut handle = sessions.write().await;
                            let session;
                            // if let std::collections::btree_map::Entry::Vacant(e) = handle.entry(message.context) {
                            //     tracing::trace!("SessionStore does not have a context of {:?}", &message.context);
                            //     session = Session::new(message.context,).await;
                            //     e.insert(session.clone());
                            // } else {
                            //     session = handle.get(&message.context).unwrap().clone();
                            // }
                            //
                            // session.deliver(message.message).await;
                        }
                        Err(err) => tracing::warn!("unable to parse packet. {:?}", err),
                    },
                    Err(_err) => {}
                }
            }
        });

        Ok(Client {
            sessions
        })
    }
}

extern crate core;

use std::collections::BTreeMap;
use std::fmt::{Debug, Formatter};
use std::net::SocketAddr;
use std::sync::{Arc, Mutex, RwLock};

use crate::boxcar_socket::{Endpoint, Session, SessionStore};
use bytecheck::CheckBytes;
use rand::Rng;
use rkyv::{Archive, Deserialize, Serialize};
use tokio::net::{TcpListener, TcpStream};

use crate::service::Handler;

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
pub enum BoxcarMessage {
    // Ping(Ping),
    // Pong(Pong),
    //
    /// Request a RPC
    RpcReq(MethodInvocation),

    /// RPC Request Status
    RpcReqStatus(RpcReqStatus),

    /// Ask the server to notify the client when Slot n changes
    WatchSlot(u16),

    /// RPC Execution Status
    RpcStatus(RpcStatus),

    ///
    RpcWait(u16),

    OpResult(bool),

    /// RPC Response
    RpcResponse(RpcResponse),
    ServerError(ServerError),
}

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
pub enum RpcReqStatus {
    /// Request has been Accepted
    Accepted(u16),
    /// Request has been Rejected
    Rejected(RpcReqRejectReason),
    /// RPC is Running
    Running,
    /// RPC has finished
    Finished(Vec<u8>),
    /// RPC has resulted in an error
    Failed(String),
}

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
pub enum RpcReqRejectReason {
    NoCap,
    NoMethod,
}

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
pub enum ServerError {
    PacketDecodeError,
    UnexpectedPacketType,
    MethodReturnError,
}

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
pub struct RpcResponse {
    pub body: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
pub struct RpcStatus {
    success: bool,
}

#[derive(Archive, Deserialize, Serialize, Debug, PartialEq, Clone)]
#[archive_attr(derive(CheckBytes, Debug))]
pub struct MethodInvocation {
    pub method: String,
    pub body: Vec<u8>,
    pub delay: bool,
}

#[derive(Clone, Debug)]
pub struct BoxcarRequest {
    slot: u16,
    /// This is a reference to the Slot Map where returned Responses are stored.
    mapping: Arc<RwLock<BTreeMap<u16, RpcResponse>>>,
}

#[derive(Clone)]
struct Boxcar {
    handlers: Vec<Arc<Handler>>,

    // Running jobs
    slots: BTreeMap<u16, Arc<Mutex<Vec<u8>>>>,

    sessions: Arc<RwLock<Vec<Arc<RwLock<Session>>>>>,
}
impl Boxcar {
    fn allocate_slot(&self) -> u16 {
        let mut depth = 0;
        let mut rng = rand::thread_rng();

        loop {
            let slot: u16 = rng.gen();
            if !self.slots.contains_key(&slot) {
                return slot;
            }
            if depth > 1024 {
                panic!("oh no")
            }
            depth += 1;
        }
    }

    ///
    async fn execute_task(&mut self, request: MethodInvocation) -> RpcReqStatus {
        let handle = self
            .handlers
            .iter()
            .find(|v| v.contains(request.method.clone()));
        if handle.is_none() {
            return RpcReqStatus::Rejected(RpcReqRejectReason::NoMethod);
        }

        let handler = handle.unwrap().clone();

        // Allocate a slot number, create a Arc Mutex to store the job result, and then store it in
        // our memory for use down the road
        let slot = self.allocate_slot();
        let job_ref = Arc::new(Mutex::new(vec![]));
        self.slots.insert(slot, job_ref.clone());

        tracing::debug!("allocating slot {} for task {:?}", slot, &request);

        let closure = async move {
            // invoke the handler method that is expected to execute the given method in the handler
            // class/library.
            // TODO add metadata?

            let body = handler.handle(&request.method, request.body).await.unwrap();

            // TODO Loop over subscribers and see if there are any open channels expecting an
            //      update on this slot. if so- send the message to the receiver

            // TODO break this out into its own function

            *job_ref.lock().unwrap() = body;
        };

        tokio::task::spawn(closure);

        RpcReqStatus::Accepted(slot)
    }
}
impl Debug for Boxcar {
    fn fmt(&self, _f: &mut Formatter<'_>) -> std::fmt::Result {
        todo!()
    }
}

#[derive(Clone)]
pub struct Server {
    bind_addr: String,
    boxcar: Arc<RwLock<Boxcar>>,
    sessions: SessionStore
}
impl Server {
    pub fn new(addr: &str) -> Self {
        Self {
            bind_addr: addr.to_string(),
            boxcar: Arc::new(RwLock::new(Boxcar {
                handlers: vec![],
                slots: BTreeMap::new(),
                sessions: Arc::new(Default::default()),
            })),
            sessions: Arc::new(Default::default())
        }
    }

    /// Attach a Handler to Boxcar
    pub fn attach_handler<T: service::HandlerTrait + Sync + Send + 'static>(&self, handler: T) {
        self.boxcar
            .write()
            .unwrap()
            .handlers
            .push(Arc::new(Box::new(handler)));
    }

    /// Run the Server
    pub async fn serve(&self) -> anyhow::Result<()> {
        let try_socket = TcpListener::bind(&self.bind_addr).await;
        let listener = try_socket.expect("Failed to bind");
        tracing::info!("listening on: {}", &self.bind_addr);

        while let Ok((stream, addr)) = listener.accept().await {
            tokio::spawn(connection_handler(stream, addr, self.sessions.clone()));
        }

        Ok(())
    }
}
impl Debug for Server {
    fn fmt(&self, _f: &mut Formatter<'_>) -> std::fmt::Result {
        todo!()
    }
}

/// Handle a incoming connection and attempt to establish a websocket connection
async fn connection_handler(stream: TcpStream, addr: SocketAddr, session: SessionStore) {
    tracing::trace!("accepted tcp connection from: {}", &addr);

    let endpoint = Endpoint::server(stream, Some(session)).await;
    if endpoint.is_err() {
        tracing::error!(
            "unable to create Endpoint handler. {:?}",
            endpoint.err().unwrap()
        );
        return;
    }

    let endpoint = endpoint.unwrap();

    let handler = async move {

    };

    // by default, the Session context is None.
    // how should this take place. the server will recieve a message with a context.
    //
    // should the server create a session in the background and spawn a task to handle it?
    //
    // or should the per-connection handler do it? that would require ...
    //
    // endpoint has a map of all active sessions. makes sense because they can be shared between connections...
    // so, there needs to be a tak that, maps a session to one or more connections.
    //
    // each connection starts an async function to handle communication.
    // in that task, it starts a new endpoint instance with a copy of the shared map
    // the endpoint will accept messages and route each packet into the right session.
    // ... the task will now need to...
    //
    //
    //
    // We need a way to recieve messages, and do a .and_then which will handle the packet with a reference to the session
    //

    // let subscriber = Arc::new(RwLock::new());
    //
    // svr.write().unwrap().subscribers.write().unwrap().push(subscriber.clone());
    //
    //     let slot = message.c_slot;
    //     let root = message.root;
    //
    //     let response = match root {
    //         BoxcarMessage::RpcReq(request) => {
    //             let mut h = svr.write().unwrap().clone();
    //             Some(BoxcarMessage::RpcReqStatus(h.execute_task(request).await))
    //         },
    //         // When this message is received, send no response. The c_slot ID will be used to return
    //         // the result, when it comes in.
    //         BoxcarMessage::WatchSlot(target_slot) => {
    //             let mut h = subscriber.write().unwrap();
    //             if !h.subscribed(target_slot) {
    //                 tracing::debug!("client subscribed to slot {}. return c_slot is {}", &target_slot, slot);
    //                 h.slots.push((target_slot, slot));
    //             } else {
    //                 tracing::trace!("client is already watching slot {}, ignoring", &target_slot);
    //             }
    //
    //             None
    //         },
    //
    //         // everything is not something the client should be sending to the server, so just
    //         // spit back an error
    //         _ => Some(BoxcarMessage::ServerError(ServerError::UnexpectedPacketType))
    //     };
    //
    //     tracing::trace!("operation response message: {:?}", &response);
    //
    //     if let Some(packet) = response {
    //         let _ = endpoint.send_resp(slot, packet).await;
    //     }

    // }
}

#[cfg(test)]
mod tests {
    // #[test]
    // fn try_encode() {
    //
    //     let inner = RpcRequest {
    //         method: "undefined".to_string(),
    //         body: vec![],
    //         _ttl: None,
    //         respond: false
    //     };
    //
    //     let packet = BoxcarMessage::RpcReq(inner);
    //
    //     // encode it
    //     let bytes = rkyv::to_bytes::<_, 256>(&packet).unwrap();
    //     println!("raw: {:?}", &bytes);
    //
    //
    //     // decode it
    //     let archived = rkyv::check_archived_root::<BoxcarMessage>(&bytes[..]).unwrap();
    //     println!("{:?}", archived);
    //
    //     let deserialized: BoxcarMessage = archived.deserialize(&mut rkyv::Infallible).unwrap();
    //
    //     assert_eq!(packet, deserialized);
    // }

    #[test]
    fn it_works() {
        let result = 2 + 2;
        assert_eq!(result, 4);
    }
}
