use crate::BoxcarMessage;
use anyhow::Context;
use bytecheck::CheckBytes;
use futures_util::{SinkExt, StreamExt};
use rand::Rng;
use rkyv::{Archive, Deserialize, Serialize};
use std::collections::BTreeMap;
use std::sync::Arc;
use tokio::io::{AsyncRead, AsyncWrite};
use tokio::sync::{broadcast, mpsc};
use tokio::sync::{Notify, RwLock};
use tokio_tungstenite::tungstenite::client::IntoClientRequest;
use tokio_tungstenite::WebSocketStream;
use tracing::instrument;



pub(crate) type SessionStore = Arc<RwLock<BTreeMap<ContextIdentifier, Session>>>;
#[derive(Clone, Debug)]
#[allow(dead_code)]
/// Abstracts tokio_tungstenite to a server/client agnostic interface. Provides a Session interface
pub struct Endpoint {
    /// Outgoing messages
    outbox: mpsc::Sender<WireMessage>,

    /// Store for incoming messages
    sessions: SessionStore,

    rx: tokio::sync::broadcast::Sender<WireMessage>,
}
impl Endpoint {
    /// Build a Client endpoint
    pub async fn client<R: IntoClientRequest + Send + Unpin>(
        req: R,
        session: Option<SessionStore>,
    ) -> Result<Endpoint, tokio_tungstenite::tungstenite::Error> {
        Endpoint::build(
            tokio_tungstenite::connect_async(req).await?.0,
            session.unwrap_or_else(|| Arc::new(Default::default())),
        )
            .await
    }

    /// Build a Server endpoint
    pub async fn server<S: AsyncRead + AsyncWrite + Send + Unpin + 'static>(
        stream: S,
        session: Option<SessionStore>,
    ) -> Result<Endpoint, tokio_tungstenite::tungstenite::Error> {
        Endpoint::build(
            tokio_tungstenite::accept_async(stream).await?,
            session.unwrap_or_else(|| Arc::new(Default::default())),
        )
            .await
    }

    /// Build the Endpoint. Establish channels and tasks to handle messages
    async fn build<T: AsyncRead + AsyncWrite + Unpin + Send + 'static>(
        ws: WebSocketStream<T>,
        sessions: SessionStore,
    ) -> Result<Self, tokio_tungstenite::tungstenite::Error> {
        let (mut ws_tx, mut ws_rx) = ws.split();

        // create incoming
        let (outbox, mut tx): (
            tokio::sync::mpsc::Sender<WireMessage>,
            tokio::sync::mpsc::Receiver<WireMessage>,
        ) = mpsc::channel(16);
        let rx: (
            tokio::sync::broadcast::Sender<WireMessage>,
            tokio::sync::broadcast::Receiver<WireMessage>,
        ) = broadcast::channel(16);

        // Worker to handle outgoing messages
        tokio::spawn(async move {
            // handle each message in the mpsc channel
            while let Some(wire_message) = tx.recv().await {
                // send the message out over the socket
                match ws_tx
                    .send(tokio_tungstenite::tungstenite::Message::Binary(
                        wire_message.encode(),
                    ))
                    .await
                {
                    Ok(_) => tracing::trace!("successfully flushed message to socket"),
                    Err(err) => tracing::error!("unable to flush message to socket. {:?}", err),
                }
            }
        });

        // Worker to handle incoming messages
        let rx0 = rx.0.clone();
        tokio::spawn(async move {
            while let Some(msg) = ws_rx.next().await {
                match msg {
                    Ok(packet) => match WireMessage::decode(packet.into_data()) {
                        Ok(message) => {
                            let _ = rx0.send(message);
                        }
                        Err(err) => tracing::warn!("unable to parse packet. {:?}", err),
                    },
                    Err(_err) => {}
                }
            }
        });

        Ok(Endpoint {
            outbox,
            sessions,
            rx: rx.0,
        })
    }

    #[allow(dead_code)]
    async fn allocate_slot(&self) -> ContextIdentifier {
        let mut depth = 0;
        let mut rng = rand::thread_rng();
        let rh = self.sessions.read().await;

        loop {
            let slot: u16 = rng.gen();
            if !rh.contains_key(&Some(slot)) {
                drop(rh);

                self.sessions
                    .write()
                    .await
                    .insert(Some(slot), Session::new(Some(slot), self.clone()).await);

                tracing::trace!("allocated slot {} after {} iterations", slot, depth);

                return Some(slot);
            }

            if depth > 1024 {
                panic!("oh no")
            }
            depth += 1;
        }
    }

    #[allow(dead_code)]
    pub async fn session(&self) -> Session {
        Session::new(self.allocate_slot().await, self.clone()).await
    }

    #[instrument]
    async fn send_raw(
        &self,
        slot: ContextIdentifier,
        message: BoxcarMessage,
    ) -> anyhow::Result<()> {
        Ok(self
            .outbox
            .send(WireMessage {
                context: slot,
                message,
            })
            .await?)
    }

    /// Send a message to the other endpoint, waiting for the flush notifier to be notified.
    ///
    /// TODO There might be a race condition where two messages arrive to be flushed at once, and
    ///      this does not track slot IDs...
    #[instrument]
    pub(crate) async fn send(
        &self,
        slot: ContextIdentifier,
        message: BoxcarMessage,
    ) -> anyhow::Result<()> {
        self.send_raw(slot, message).await
    }

    /// Send a message, without waiting for acknowledgement from the other side.
    #[instrument]
    async fn yeet(&self, slot: ContextIdentifier, message: BoxcarMessage) -> anyhow::Result<()> {
        self.send_raw(slot, message).await
    }

    // #[allow(dead_code)]
    // pub async fn recv<F, Fut>(&self, f: F)
    // where
    //     F: Fn(&Session) -> Fut + Send,
    //     Fut: Future<Output = ()>,
    // {
    //     let mut rx = self.rx.subscribe();
    //     while let Ok(message) = rx.recv().await {
    //         let mut handle = self.sessions.write().await;
    //         let session;
    //
    //         if let std::collections::btree_map::Entry::Vacant(e) = handle.entry(message.context) {
    //             tracing::trace!("SessionStore does not have a context of {:?}", &message.context);
    //             session = Session::new(message.context, self.clone()).await;
    //             e.insert(session.clone());
    //         } else {
    //             session = handle.get(&message.context).unwrap().clone();
    //         }
    //
    //         session.deliver(message.message).await;
    //
    //         f(&session).await;
    //     }
    //
    //
    //     // // worker to route incoming messages
    //     // let sessions_arc = sessions.clone();
    //     // tokio::spawn(async move {
    //     //     while let Ok(msg) = rx.1.recv().await {
    //     //         // TODO add tracing
    //     //         let handle = sessions_arc.write().await;
    //     //         let session: Option<&Session> = handle.get(&msg.context);
    //     //         if session.is_some() {
    //     //             session.unwrap().deliver(msg.message).await;
    //     //         }
    //     //     }
    //     // });
    // }


}
