use crate::{consts::rancher::*, docker, rancher, types::cli::Run, util};
use anyhow::Result;
use bollard::{
    container,
    models::{HostConfig, PortBinding, RestartPolicy, RestartPolicyNameEnum},
};
use log::info;
use std::collections::HashMap;

pub async fn run(opt: &Run, docker_socket_path: Option<String>) -> Result<()> {
    let mut exposed_ports = HashMap::with_capacity(2);
    exposed_ports.insert(CONTAINER_PORT_80.to_string(), HashMap::with_capacity(0));
    exposed_ports.insert(CONTAINER_PORT_443.to_string(), HashMap::with_capacity(0));

    let mut port_bindings = HashMap::with_capacity(2);
    port_bindings.insert(
        String::from(CONTAINER_PORT_80),
        Some(vec![PortBinding {
            host_ip: None,
            host_port: Some(opt.common.host80.clone()),
        }]),
    );
    port_bindings.insert(
        String::from(CONTAINER_PORT_443),
        Some(vec![PortBinding {
            host_ip: None,
            host_port: Some(opt.common.host443.clone()),
        }]),
    );

    let image = util::get_full_image(&opt.image, &opt.image_tag);
    let mut env = Vec::new();
    if opt.common.enable_restricted_default_admin {
        env.push(CATTLE_RESTRICTED_DEFAULT_ADMIN.to_string());
    }
    if let Some(s) = &opt.common.proxy {
        env.push(format!("HTTP_PROXY={}", s));
        env.push(format!("HTTPS_PROXY={}", s));
        env.push(NO_PROXY.to_string());
    }
    if opt.common.host_certs.is_some() {
        env.push(SSL_CERT_DIR.to_string())
    }
    if opt.common.audit_log.is_some() {
        env.push(format!("AUDIT_LEVEL={}", opt.common.audit_level));
    }
    if let Some(s) = &opt.common.tls_min_version {
        env.push(format!("CATTLE_TLS_MIN_VERSION=\"{}\"", s));
    }
    if let Some(s) = &opt.common.tls_ciphers {
        env.push(format!("CATTLE_TLS_CIPHERS=\"{}\"", s));
    }
    if let Some(s) = &opt.common.bootstrap_password {
        env.push(format!("CATTLE_BOOTSTRAP_PASSWORD={}", s))
    }

    let mut cmd = Vec::with_capacity(0);
    if let Some(s) = &opt.common.domain_address {
        cmd.push("--acme-domain".to_string());
        cmd.push(s.clone());
    }
    if opt.common.no_cacerts
        || (opt.common.full_chain.is_some()
            && opt.common.private_key.is_some()
            && opt.common.ca_certs.is_none())
    {
        // If not explicitly selected, disable the default CA cert generated by Rancher if CA certs
        // are not provided.
        cmd.push("--no-cacerts".to_string());
    }

    let mut binds = Vec::with_capacity(0);
    if let Some(s) = &opt.common.full_chain {
        binds.push(rancher::build_bind(s, FULL_CHAIN)?);
    }
    if let Some(s) = &opt.common.private_key {
        binds.push(rancher::build_bind(s, PRIVATE_KEY)?);
    }
    if let Some(s) = &opt.common.ca_certs {
        binds.push(rancher::build_bind(s, CA_CERTS)?);
    }
    if let Some(s) = &opt.common.host_certs {
        binds.push(rancher::build_bind(s, CONTAINER_CERTS)?);
    }
    if let Some(s) = &opt.common.audit_log {
        binds.push(rancher::build_bind(s, AUDIT_LOG)?);
    }
    if let Some(s) = &opt.common.persistent_data {
        binds.push(rancher::build_bind(s, VAR_LIB_RANCHER)?);
    }

    let container_config = container::Config {
        image: Some(image.clone()),
        env: match env.is_empty() {
            true => None,
            false => Some(env),
        },
        cmd: match cmd.is_empty() {
            true => None,
            false => Some(cmd),
        },
        labels: Some(rancher::get_labels()),
        exposed_ports: Some(exposed_ports),
        host_config: Some(HostConfig {
            binds: match binds.is_empty() {
                true => None,
                false => Some(binds),
            },
            privileged: Some(true),
            port_bindings: Some(port_bindings),
            restart_policy: Some(RestartPolicy {
                name: Some(RestartPolicyNameEnum::UNLESS_STOPPED),
                maximum_retry_count: None,
            }),
            volumes_from: None,
            ..Default::default()
        }),
        ..Default::default()
    };

    match opt.dry_run {
        true => info!("{}", serde_json::to_string_pretty(&container_config)?),
        false => {
            let docker = docker::docker_client(docker_socket_path).await?;
            docker::pull_image(&docker, &image, opt.common.force_pull).await?;
            rancher::launch_rancher(docker, container_config).await?;
        }
    }
    Ok(())
}
