use chrono::prelude::*;
use colored::Colorize;
use mut_static::MutStatic;

///
/// internal structure used to store pointers for functions to call on logs
///
pub struct Logger {
    pub on_info: fn(&str),
    pub on_warn: fn(&str),
    pub on_error: fn(&str),
}

impl Default for Logger {
    fn default() -> Self {
        Logger {
            on_info: default_on_info,
            on_warn: default_on_warn,
            on_error: default_on_error,
        }
    }
}

fn time_segment() -> String {
    let now = Local::now();
    let result = now.format("%Y/%m/%d-%H:%M:%S").to_string();
    result
}

fn default_on_info(line: &str) {
    let identifier = "info".blue();
    let prefix = format!("[{}] [{}]", time_segment(), identifier).bright_black();
    println!("{} {}", prefix, line);
}

fn default_on_warn(line: &str) {
    let identifier = "warn".yellow();
    let prefix = format!("[{}] [{}]", time_segment(), identifier).bright_black();
    println!("{} {}", prefix, line);
}

fn default_on_error(line: &str) {
    let identifier = "error".red();
    let prefix = format!("[{}] [{}]", time_segment(), identifier).bright_black();
    println!("{} {}", prefix, line);
}

lazy_static! {
    static ref LOGGER: MutStatic<Logger> = MutStatic::new();
}

pub fn read_on_static<T: FnOnce(&Logger)>(read_opperation: T) {
    if !LOGGER.is_set().unwrap() {
        LOGGER.set(Logger::default()).unwrap();
    }
    let reference = LOGGER.read().unwrap();
    read_opperation(reference.as_ref())
}

#[test]
fn time() {
    println!("{}", time_segment());
}

#[test]
fn info() {
    default_on_info("lorem")
}
#[test]
fn warn() {
    default_on_warn("ipsum")
}
#[test]
fn error() {
    default_on_error("dolor")
}
