/*!
This crate provides flexible
**B**orrowed, **O**wned or **S**hared (B.O.S.) smart pointers.
They are like std's [`Cow`][std::borrow::Cow]
but with additonal [`Arc`][std::sync::Arc]
and [`Rc`][std::rc::Rc] variants.
Unlike std's [`Cow`][std::borrow::Cow],
they can also be used with types
that don't implement [`Clone`] or [`ToOwned`].

The `bos` smart pointers allow you to:

- share data efficiently
- decide ownership at runtime
- avoid cloning the data until it becomes necessary

# "A" stands for "atomic"

For example [`Abos`] implements [`Send`] and [`Sync`]
but does *not* have a [`Rc`][std::rc::Rc] variant.

[`Bos`] does *not* implement [`Send`] or [`Sync`]
but does have a [`Rc`][std::rc::Rc] variant.

# [`AbosStr`] and [`BosStr`] types

[`Abos`] and [`Bos`] are often used with [`str`] or similar types.
Feel free to have a look at our handy [type aliases](#types) below.

# `serde` feature

For [`serde`](https://crates.io/crates/serde) support,
you need to enable our `serde` feature:
```toml
[dependencies]
bos = { version = "?", features = ["serde"] }
```
Replace `version = "?"` with the version you want.

# Examples

```
use std::sync::Arc;
use bos::AbosStr;

// No need to allocate memory for a &'static str
let title = AbosStr::Borrowed("Dune");

// The author's name shouldn't be too long.
// So we can just store it in a normal String.
let author = AbosStr::Owned(read_file("dune/author.txt"));

// A whole book is really long.
// And we want to share this string with multiple threads.
// So it would be efficient to store in an Arc,
// since cloning the Arc does not clone the String data inside of it.
let book_text = AbosStr::Arc(Arc::new(read_file("dune/book.txt")));

// We can store a &str, String and Arc<String> inside the same Vec
// because we are using the AStr enum.
let book_infos: Vec<AbosStr> = vec![title, author, book_text];

fn read_file(path: &str) -> String {
    // ...
# String::new()
}
```

This example shows how to use `Abos` in a struct:
```
use bos::{AbosStr, AbosBrw};
use std::sync::Arc;

#[derive(Debug, Eq, PartialEq)]
struct Header<'a> {
    // AbosStr<'a> is a type alias for AbosBrw<'a, str>
    // and results in a type of Abos<'a, str, String>
    name: AbosStr<'a>,
    value: AbosStr<'a>,
}

// The struct with 'static is still very useful
// because it is not limited to &'static str.
// String and Arc<String> can be used, too.
fn add_global_header(header: Header<'static>) {
    // ...
}

// Let's pretend this header came over the network
// and was parsed into our Header struct.
let header = Header {
    name: AbosStr::Owned(String::from("content-type")),
    value: AbosStr::Arc(Arc::new(String::from("text/html; charset=UTF-8"))),
};

add_global_header(header);
```
*/

#![forbid(unsafe_code)]

mod abos;
mod bos;

#[cfg(feature = "serde")]
mod serde_support;

pub use self::abos::Abos;
pub use self::bos::Bos;

/// **A**tomic **B**orrowed, **O**wned or **S**hared smart pointer for **B**o**r**ro**w**ed types implementing the [`ToOwned`] trait.
pub type AbosBrw<'b, B> = Abos<'b, B, <B as ToOwned>::Owned, <B as ToOwned>::Owned>;

/// **B**orrowed, **O**wned or **S**hared smart pointer for **B**o**r**ro**w**ed types implementing the [`ToOwned`] trait.
pub type BosBrw<'b, B> = Bos<'b, B, <B as ToOwned>::Owned, <B as ToOwned>::Owned>;

/// **A**tomic **B**orrowed, **O**wned or **S**hared heap allocating smart pointer.
pub type BoxAbos<'b, T> = Abos<'b, T, Box<T>, T>;

/// **B**orrowed, **O**wned or **S**hared heap allocating smart pointer.
pub type BoxBos<'b, T> = Bos<'b, T, Box<T>, T>;

/// **A**tomic **B**orrowed, **O**wned or **S**hared [`str`] smart pointer.
pub type AbosStr<'b> = AbosBrw<'b, str>;

/// **B**orrowed, **O**wned or **S**hared [`str`] smart pointer.
pub type BosStr<'b> = BosBrw<'b, str>;
