/*!
This crate provides flexible **B**orrowed, **O**wned or **S**hared (B.O.S.) smart pointers.
They are like std's [`Cow`][std::borrow::Cow]
but with additonal [`Arc`][std::sync::Arc] and [`Rc`][std::rc::Rc] variants.

The `Abos` and `Bos` smart pointers allow you to:

- share data efficiently
- decide ownership at runtime
- avoid cloning the data until it becomes necessary

# `Abos` or `Bos`

`Abos` implements [`Send`] and [`Sync`] but does *not* have a [`Rc`][std::rc::Rc] variant.
The A stands for "atomic".

`Bos` does *not* implement [`Send`] or [`Sync`] but does have a [`Rc`][std::rc::Rc] variant.

# Examples

```
# use std::sync::Arc;
use bos::borrowed::Abos;

// No need to allocate memory for a &'static str
let title = Abos::Borrowed("Dune");

// The author's name shouldn't be too long.
// So we can just store it in a normal String.
let author = Abos::Owned(read_file("dune/author.txt"));

// A whole book is really long.
// And we want to share this string with multiple threads.
// So it would be efficient to store in an Arc,
// since cloning the Arc does not clone the String data inside of it.
let book_text = Abos::ArcOwned(Arc::new(read_file("dune/book.txt")));

// We can store a &str, String and Arc<String> inside the same Vec
// because we are using the AStr enum.
let book_infos: Vec<bos::AStr> = vec![title, author, book_text];

fn read_file(path: &str) -> String {
    // ...
# String::new()
}
```

```
use bos::borrowed::Abos;

struct Header<'a> {
    name: bos::AStr<'a>, // AStr<'a> is a type alias for bos::borrowed::Abos<'a, str>
    bytes: Abos<'a, [u8]>,
}

// The struct with 'static is still very useful
// because it is not limited to &'static str or &'static [u8].
// String, Arc<str>, Arc<String>, Vec<u8>, Arc<[u8]> and Arc<Vec<u8>> can be used, too.
fn add_global_header(header: Header<'static>) {
    // ...
}
```

# [`borrowed`], [`sized`] or [`boxed`]

1. If your type has an owned type and a different borrowed type, prefer the [`borrowed`] module.
2. Otherwise if your type should be heap allocated, prefer the [`boxed`] module.
3. Otherwise, use the [`sized`] module for your type.

Feel free to directly compare the `enum` declarations via these links to
[`borrowed::Abos`], [`sized::Abos`] and [`boxed::Abos`].
Or use the table below for comparison:

|                                           | [`borrowed`]            | [`sized`]     | [`boxed`]
| ----------------------------------------- | ----------------------- | ------------- | -----------
| type requirements                         | `B: ToOwned + ?Sized`   | `T: Sized`    | `T: ?Sized`
| different Owned and Borrowed types?       | yes                     | no            | no
| ideal for slices like `[u8]` or `str`     | yes                     | no            | no
| Owned is heap allocated?                  | no                      | no            | yes

# [`AStr`] and [`Str`] types

[`borrowed::Abos`] and [`borrowed::Bos`] are often used with [`str`] or similar types.
Feel free to have a look at our handy [type aliases](#types) below.

# Re-exports deprecation warning

Please note that the re-exports of
`crate::borrowed::Abos` and `crate::borrowed::Bos` are deprecated!
Unfortunately rust doesn't seem to actually warn you about that.
So you should take extra care.
These will be removed in version "0.3.0" or "1.0.0".
*/

#![forbid(unsafe_code)]

pub mod borrowed;
pub mod boxed;
pub mod sized;

#[doc(no_inline)]
#[deprecated(since = "0.2.2", note = "Please use bos:borrowed::Abos instead.")]
pub use crate::borrowed::Abos;

#[doc(no_inline)]
#[deprecated(since = "0.2.2", note = "Please use bos:borrowed::Bos instead.")]
pub use crate::borrowed::Bos;

/// **A**tomic [`str`] smart pointer, that is **B**orrowed, **O**wned or **S**hared.
pub type AStr<'a> = crate::borrowed::Abos<'a, str>;

/// [`str`] smart pointer, that is **B**orrowed, **O**wned or **S**hared.
pub type Str<'a> = crate::borrowed::Bos<'a, str>;
