use super::Abos;
use std::borrow::{Borrow, Cow};
use std::cmp::Ordering;
use std::fmt;
use std::hash::Hash;
use std::ops::Deref;
use std::rc::Rc;
use std::sync::Arc;

/// **B**orrowed, **O**wned or **S**hared smart pointer
/// to types that should be heap allocated,
/// like [trait objects](https://doc.rust-lang.org/reference/types/trait-object.html).
///
/// See the [crate-level documentation][crate#borrowed-sized-or-boxed]
/// for a comparison of borrowed, sized and boxed smart pointers.
pub enum Bos<'a, T: ?Sized> {
    Borrowed(&'a T),
    Owned(Box<T>),
    Arc(Arc<T>),
    Rc(Rc<T>),
}

impl<'a, T: ?Sized> Bos<'a, T> {
    /// Extracts the owned data.
    ///
    /// Clones the data if it is not already owned.
    #[inline]
    pub fn into_owned(self) -> Box<T>
    where
        T: Clone,
    {
        match self {
            Bos::Borrowed(x) => x.clone().into(),
            Bos::Owned(x) => x,
            Bos::Arc(x) => Arc::try_unwrap(x)
                .unwrap_or_else(|x| Borrow::borrow(&*x).clone())
                .into(),
            Bos::Rc(x) => Rc::try_unwrap(x)
                .unwrap_or_else(|x| Borrow::borrow(&*x).clone())
                .into(),
        }
    }

    /// Convert [`Bos::Borrowed`] to [`Bos::Owned`] so that the new `Bos` has a `'static` lifetime.
    #[inline]
    pub fn into_static_unborrowed(self) -> Bos<'static, T>
    where
        T: Clone,
    {
        match self {
            Bos::Borrowed(x) => Bos::Owned(x.clone().into()),
            Bos::Owned(x) => Bos::Owned(x),
            Bos::Arc(x) => Bos::Arc(x),
            Bos::Rc(x) => Bos::Rc(x),
        }
    }

    /// Convert to a shared variant so that the new `Bos` has a `'static` lifetime and [`Clone`] will not copy the inner data.
    #[inline]
    pub fn into_static_shared(self) -> Bos<'static, T>
    where
        T: Clone,
    {
        match self {
            Bos::Borrowed(x) => Bos::Rc(x.clone().into()),
            Bos::Owned(x) => Bos::Rc(x.into()),
            Bos::Arc(x) => Bos::Arc(x),
            Bos::Rc(x) => Bos::Rc(x),
        }
    }

    /// This is unstable and only available with the `unstable` feature.
    #[cfg(any(doc, feature = "unstable"))] // reason: recently added
    #[inline]
    pub fn share_owned(&mut self)
    where
        T: Default,
    {
        if let Self::Owned(x) = self {
            let x = std::mem::take(x);
            *self = Self::Rc(x.into());
        }
    }

    /// Acquires a mutable reference to the owned form of the data.
    ///
    /// Clones the data if it is not already owned.
    #[inline]
    pub fn to_mut(&mut self) -> &mut T
    where
        T: Clone,
    {
        match self {
            Bos::Borrowed(x) => {
                *self = Bos::Owned(x.clone().into());
                self.to_mut()
            }
            Bos::Owned(x) => x,
            Bos::Arc(x) => Arc::make_mut(x),
            Bos::Rc(x) => Rc::make_mut(x),
        }
    }

    #[inline]
    fn internal_borrow(&self) -> &T {
        match self {
            Bos::Borrowed(x) => x,
            Bos::Owned(x) => x,
            Bos::Arc(x) => x.deref(),
            Bos::Rc(x) => x.deref(),
        }
    }
}

impl<'a, T: ?Sized> AsRef<T> for Bos<'a, T> {
    #[inline]
    fn as_ref(&self) -> &T {
        self.internal_borrow()
    }
}

impl<'a, T: ?Sized> Borrow<T> for Bos<'a, T> {
    #[inline]
    fn borrow(&self) -> &T {
        self.internal_borrow()
    }
}

impl<'a, T: ?Sized> Clone for Bos<'a, T>
where
    T: Clone,
{
    #[inline]
    fn clone(&self) -> Self {
        match self {
            Self::Borrowed(x) => Self::Borrowed(x),
            Self::Owned(x) => Self::Owned(x.clone()),
            Self::Arc(x) => Self::Arc(x.clone()),
            Self::Rc(x) => Self::Rc(x.clone()),
        }
    }
}

impl<'a, T: ?Sized> Deref for Bos<'a, T> {
    type Target = T;

    #[inline]
    fn deref(&self) -> &Self::Target {
        self.borrow()
    }
}

impl<'a, T: ?Sized> Default for Bos<'a, T>
where
    Box<T>: Default,
{
    /// Creates an owned Bos with the default value for the contained owned type.
    #[inline]
    fn default() -> Self {
        Self::Owned(Box::default())
    }
}

impl<'a, T: ?Sized> fmt::Debug for Bos<'a, T>
where
    T: fmt::Debug,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Bos::Borrowed(x) => f.debug_tuple("Bos::Borrowed").field(x).finish(),
            Bos::Owned(x) => f.debug_tuple("Bos::Owned").field(x).finish(),
            Bos::Arc(x) => f.debug_tuple("Bos::Arc").field(x).finish(),
            Bos::Rc(x) => f.debug_tuple("Bos::Rc").field(x).finish(),
        }
    }
}

impl<'a, T: ?Sized> fmt::Display for Bos<'a, T>
where
    T: fmt::Display,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::Borrowed(x) => fmt::Display::fmt(x, f),
            Self::Owned(x) => fmt::Display::fmt(x, f),
            Self::Arc(x) => fmt::Display::fmt(x, f),
            Self::Rc(x) => fmt::Display::fmt(x, f),
        }
    }
}

impl<'a, T: ?Sized> Hash for Bos<'a, T>
where
    T: Hash,
{
    #[inline]
    fn hash<H: std::hash::Hasher>(&self, state: &mut H) {
        self.internal_borrow().hash(state)
    }
}

impl<'a, T: ?Sized> PartialEq<T> for Bos<'a, T>
where
    T: PartialEq,
{
    #[inline]
    fn eq(&self, other: &T) -> bool {
        self.internal_borrow() == other
    }
}

impl<'a, T: ?Sized> PartialEq for Bos<'a, T>
where
    T: PartialEq,
{
    #[inline]
    fn eq(&self, other: &Self) -> bool {
        self.internal_borrow() == other.internal_borrow()
    }
}

impl<'a, T: ?Sized> Eq for Bos<'a, T> where T: Eq {}

impl<'a, T: ?Sized> PartialOrd<T> for Bos<'a, T>
where
    T: PartialOrd,
{
    #[inline]
    fn partial_cmp(&self, other: &T) -> Option<Ordering> {
        self.internal_borrow().partial_cmp(other)
    }
}

impl<'a, T: ?Sized> PartialOrd for Bos<'a, T>
where
    T: PartialOrd,
{
    #[inline]
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        self.internal_borrow().partial_cmp(other.internal_borrow())
    }
}

impl<'a, T: ?Sized> Ord for Bos<'a, T>
where
    T: Ord,
{
    #[inline]
    fn cmp(&self, other: &Self) -> Ordering {
        self.internal_borrow().cmp(other.internal_borrow())
    }
}

impl<'a, T> From<Cow<'a, T>> for Bos<'a, T>
where
    T: ToOwned<Owned = T>,
{
    #[inline]
    fn from(c: Cow<'a, T>) -> Self {
        match c {
            Cow::Borrowed(x) => Self::Borrowed(x),
            Cow::Owned(x) => Self::Owned(x.into()),
        }
    }
}

impl<'a, T: ?Sized> From<Cow<'a, Box<T>>> for Bos<'a, T>
where
    Box<T>: ToOwned<Owned = Box<T>>,
{
    #[inline]
    fn from(p: Cow<'a, Box<T>>) -> Self {
        match p {
            Cow::Borrowed(x) => Self::Borrowed(x),
            Cow::Owned(x) => Self::Owned(x),
        }
    }
}

impl<'a, T> From<Abos<'a, T>> for Bos<'a, T> {
    #[inline]
    fn from(p: Abos<'a, T>) -> Self {
        match p {
            Abos::Borrowed(x) => Self::Borrowed(x),
            Abos::Owned(x) => Self::Owned(x),
            Abos::Arc(x) => Self::Arc(x),
        }
    }
}

impl<'a, T: ?Sized> From<&'a T> for Bos<'a, T> {
    #[inline]
    fn from(x: &'a T) -> Self {
        Self::Borrowed(x)
    }
}

impl<'a, T: ?Sized> From<Box<T>> for Bos<'a, T> {
    #[inline]
    fn from(x: Box<T>) -> Self {
        Self::Owned(x)
    }
}

impl<'a, T: ?Sized> From<Arc<T>> for Bos<'a, T> {
    #[inline]
    fn from(x: Arc<T>) -> Self {
        Self::Arc(x)
    }
}

impl<'a, T: ?Sized> From<Rc<T>> for Bos<'a, T> {
    #[inline]
    fn from(x: Rc<T>) -> Self {
        Self::Rc(x)
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use assert_matches::assert_matches;

    #[cfg(feature = "unstable")]
    #[test]
    fn share_owned() {
        use tap::Tap;

        let text = "example".to_owned();
        let owned = Bos::<String>::Owned(text.clone().into());
        assert_matches!(&owned, &Bos::Owned(_));
        let shared = owned.tap_mut(|s| s.share_owned());
        assert_matches!(&shared, &Bos::Rc(_));
        assert_eq!(shared, Bos::Borrowed(&text));
    }

    #[test]
    fn owned_into_static_shared() {
        let text = "example".to_owned();
        let owned = Bos::<String>::Owned(text.clone().into());
        let shared = owned.into_static_shared();
        assert_matches!(&shared, &Bos::Rc(_));
        assert_eq!(shared, Bos::Borrowed(&text));
    }

    #[test]
    fn borrowed_into_static_shared() {
        let text = "example".to_owned();
        let borrowed = Bos::Borrowed(&text);
        let shared = borrowed.into_static_shared();
        assert_matches!(&shared, &Bos::Rc(_));
        assert_eq!(shared, Bos::Borrowed(&text));
    }
}
