use super::Abos;
use std::borrow::{Borrow, Cow};
use std::cmp::Ordering;
use std::ffi::{CStr, CString, OsStr, OsString};
use std::fmt;
use std::hash::Hash;
use std::ops::{Add, AddAssign, Deref};
use std::path::{Path, PathBuf};
use std::rc::Rc;
use std::sync::Arc;

/// **B**orrowed, **O**wned or **S**hared smart pointer
/// to dynamically sized borrowed types like [`str`], the borrowed form of [`String`].
///
/// See the [crate-level documentation][crate#borrowed-sized-or-boxed]
/// for a comparison of borrowed, sized and boxed smart pointers.
pub enum Bos<'b, B>
where
    B: 'b + ToOwned + ?Sized,
{
    Borrowed(&'b B),
    Owned(B::Owned),
    ArcBorrowed(Arc<B>),
    ArcOwned(Arc<B::Owned>),
    RcBorrowed(Rc<B>),
    RcOwned(Rc<B::Owned>),
}

impl<'b, B: 'b + ToOwned + ?Sized> Bos<'b, B> {
    /// Extracts the owned data.
    ///
    /// Clones the data if it is not already owned.
    #[inline]
    pub fn into_owned(self) -> B::Owned {
        match self {
            Bos::Borrowed(x) => x.to_owned(),
            Bos::Owned(x) => x,
            Bos::ArcBorrowed(x) => x.as_ref().to_owned(),
            Bos::ArcOwned(x) => {
                Arc::try_unwrap(x).unwrap_or_else(|x| Borrow::borrow(&*x).to_owned())
            }
            Bos::RcBorrowed(x) => x.as_ref().to_owned(),
            Bos::RcOwned(x) => Rc::try_unwrap(x).unwrap_or_else(|x| Borrow::borrow(&*x).to_owned()),
        }
    }

    /// Convert [`Bos::Borrowed`] to [`Bos::Owned`] so that the new `Bos` has a `'static` lifetime.
    #[inline]
    pub fn into_static_unborrowed(self) -> Bos<'static, B> {
        match self {
            Bos::Borrowed(x) => Bos::Owned(x.to_owned()),
            Bos::Owned(x) => Bos::Owned(x),
            Bos::ArcBorrowed(x) => Bos::ArcBorrowed(x),
            Bos::ArcOwned(x) => Bos::ArcOwned(x),
            Bos::RcBorrowed(x) => Bos::RcBorrowed(x),
            Bos::RcOwned(x) => Bos::RcOwned(x),
        }
    }

    /// Convert to a shared variant so that the new `Bos` has a `'static` lifetime and [`Clone`] will not copy the inner data.
    #[inline]
    pub fn into_static_shared(self) -> Bos<'static, B>
    where
        for<'a> &'a B: Into<Rc<B>>,
    {
        match self {
            Bos::Borrowed(x) => Bos::RcBorrowed(x.into()),
            Bos::Owned(x) => Bos::RcOwned(x.into()),
            Bos::ArcBorrowed(x) => Bos::ArcBorrowed(x),
            Bos::ArcOwned(x) => Bos::ArcOwned(x),
            Bos::RcBorrowed(x) => Bos::RcBorrowed(x),
            Bos::RcOwned(x) => Bos::RcOwned(x),
        }
    }

    /// This is unstable and only available with the `unstable` feature.
    #[cfg(any(doc, feature = "unstable"))] // reason: recently added
    #[inline]
    pub fn share_owned(&mut self)
    where
        B::Owned: Default,
    {
        if let Self::Owned(x) = self {
            let x = std::mem::take(x);
            *self = Self::RcOwned(x.into());
        }
    }

    /// Acquires a mutable reference to the owned form of the data.
    ///
    /// Clones the data if it is not already owned.
    #[inline]
    pub fn to_mut(&mut self) -> &mut B::Owned
    where
        B::Owned: Clone,
    {
        match self {
            Bos::Borrowed(x) => {
                *self = Bos::Owned(x.to_owned());
                self.to_mut()
            }
            Bos::Owned(x) => x,
            Bos::ArcBorrowed(x) => {
                *self = Bos::Owned(x.as_ref().to_owned());
                self.to_mut()
            }
            Bos::ArcOwned(x) => Arc::make_mut(x),
            Bos::RcBorrowed(x) => {
                *self = Bos::Owned(x.as_ref().to_owned());
                self.to_mut()
            }
            Bos::RcOwned(x) => Rc::make_mut(x),
        }
    }

    #[inline]
    fn internal_borrow(&self) -> &B {
        match self {
            Bos::Borrowed(x) => x,
            Bos::Owned(x) => x.borrow(),
            Bos::ArcBorrowed(x) => x.deref(),
            Bos::ArcOwned(x) => x.deref().borrow(),
            Bos::RcBorrowed(x) => x.deref(),
            Bos::RcOwned(x) => x.deref().borrow(),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized, Rhs> Add<Rhs> for Bos<'b, B>
where
    B::Owned: Add<Rhs>,
{
    type Output = <B::Owned as Add<Rhs>>::Output;

    #[inline]
    fn add(self, rhs: Rhs) -> <B::Owned as Add<Rhs>>::Output {
        self.into_owned() + rhs
    }
}

impl<'b, B: 'b + ToOwned + ?Sized, Rhs> AddAssign<Rhs> for Bos<'b, B>
where
    B::Owned: AddAssign<Rhs> + Clone,
{
    #[inline]
    fn add_assign(&mut self, rhs: Rhs) {
        *self.to_mut() += rhs;
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> AsRef<B> for Bos<'b, B> {
    #[inline]
    fn as_ref(&self) -> &B {
        self.internal_borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Borrow<B> for Bos<'b, B> {
    #[inline]
    fn borrow(&self) -> &B {
        self.internal_borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Clone for Bos<'b, B>
where
    B::Owned: Clone,
{
    #[inline]
    fn clone(&self) -> Self {
        match self {
            Self::Borrowed(x) => Self::Borrowed(x),
            Self::Owned(x) => Self::Owned(x.clone()),
            Self::ArcBorrowed(x) => Self::ArcBorrowed(x.clone()),
            Self::ArcOwned(x) => Self::ArcOwned(x.clone()),
            Self::RcBorrowed(x) => Self::RcBorrowed(x.clone()),
            Self::RcOwned(x) => Self::RcOwned(x.clone()),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Deref for Bos<'b, B> {
    type Target = B;

    #[inline]
    fn deref(&self) -> &Self::Target {
        self.borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Default for Bos<'b, B>
where
    B::Owned: Default,
{
    /// Creates an owned Bos with the default value for the contained owned type.
    #[inline]
    fn default() -> Self {
        Self::Owned(B::Owned::default())
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> fmt::Debug for Bos<'b, B>
where
    B: fmt::Debug,
    B::Owned: fmt::Debug,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Bos::Borrowed(x) => f.debug_tuple("Bos::Borrowed").field(x).finish(),
            Bos::Owned(x) => f.debug_tuple("Bos::Owned").field(x).finish(),
            Bos::ArcBorrowed(x) => f.debug_tuple("Bos::ArcBorrowed").field(x).finish(),
            Bos::ArcOwned(x) => f.debug_tuple("Bos::ArcOwned").field(x).finish(),
            Bos::RcBorrowed(x) => f.debug_tuple("Bos::RcBorrowed").field(x).finish(),
            Bos::RcOwned(x) => f.debug_tuple("Bos::RcOwned").field(x).finish(),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> fmt::Display for Bos<'b, B>
where
    B: fmt::Display,
    B::Owned: fmt::Display,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::Borrowed(x) => fmt::Display::fmt(x, f),
            Self::Owned(x) => fmt::Display::fmt(x, f),
            Self::ArcBorrowed(x) => fmt::Display::fmt(x, f),
            Self::ArcOwned(x) => fmt::Display::fmt(x, f),
            Self::RcBorrowed(x) => fmt::Display::fmt(x, f),
            Self::RcOwned(x) => fmt::Display::fmt(x, f),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Hash for Bos<'b, B>
where
    B: Hash,
{
    #[inline]
    fn hash<H: std::hash::Hasher>(&self, state: &mut H) {
        self.internal_borrow().hash(state)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialEq<B> for Bos<'b, B>
where
    B: PartialEq,
{
    #[inline]
    fn eq(&self, other: &B) -> bool {
        self.internal_borrow() == other
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialEq for Bos<'b, B>
where
    B: PartialEq,
{
    #[inline]
    fn eq(&self, other: &Self) -> bool {
        self.internal_borrow() == other.internal_borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Eq for Bos<'b, B> where B: Eq {}

impl<'b, B: 'b + ToOwned + ?Sized> PartialOrd<B> for Bos<'b, B>
where
    B: PartialOrd,
{
    #[inline]
    fn partial_cmp(&self, other: &B) -> Option<Ordering> {
        self.internal_borrow().partial_cmp(other)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialOrd for Bos<'b, B>
where
    B: PartialOrd,
{
    #[inline]
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        self.internal_borrow().partial_cmp(other.internal_borrow())
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Ord for Bos<'b, B>
where
    B: Ord,
{
    #[inline]
    fn cmp(&self, other: &Self) -> Ordering {
        self.internal_borrow().cmp(other.internal_borrow())
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> From<Cow<'b, B>> for Bos<'b, B> {
    #[inline]
    fn from(p: Cow<'b, B>) -> Self {
        match p {
            Cow::Borrowed(x) => Self::Borrowed(x),
            Cow::Owned(x) => Self::Owned(x),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> From<Abos<'b, B>> for Bos<'b, B> {
    #[inline]
    fn from(p: Abos<'b, B>) -> Self {
        match p {
            Abos::Borrowed(x) => Self::Borrowed(x),
            Abos::Owned(x) => Self::Owned(x),
            Abos::ArcBorrowed(x) => Self::ArcBorrowed(x),
            Abos::ArcOwned(x) => Self::ArcOwned(x),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> From<&'b B> for Bos<'b, B> {
    #[inline]
    fn from(x: &'b B) -> Self {
        Self::Borrowed(x)
    }
}

/// DEPRECATED! Use Bos::ArcOwned instead
// TODO: remove this impl before 1.0 for more flexibility going forwards.
// This impl currently blocks implementing From<Arc<str>> for Abos<str> etc.
impl<'b, B: 'b + ToOwned + ?Sized> From<Arc<B::Owned>> for Bos<'b, B> {
    #[inline]
    fn from(x: Arc<B::Owned>) -> Self {
        Self::ArcOwned(x)
    }
}

impl<'b, T> From<Vec<T>> for Bos<'b, [T]>
where
    T: Clone,
{
    #[inline]
    fn from(x: Vec<T>) -> Self {
        Self::Owned(x)
    }
}

impl<'b, T> From<Bos<'b, [T]>> for Vec<T>
where
    [T]: ToOwned<Owned = Vec<T>>,
{
    #[inline]
    fn from(x: Bos<'b, [T]>) -> Self {
        x.into_owned()
    }
}

impl<'b> From<String> for Bos<'b, str> {
    #[inline]
    fn from(x: String) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Bos<'b, str>> for String {
    #[inline]
    fn from(x: Bos<'b, str>) -> Self {
        x.into_owned()
    }
}

impl<'b> From<CString> for Bos<'b, CStr> {
    #[inline]
    fn from(x: CString) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Bos<'b, CStr>> for CString {
    #[inline]
    fn from(x: Bos<'b, CStr>) -> Self {
        x.into_owned()
    }
}

impl<'b> From<OsString> for Bos<'b, OsStr> {
    #[inline]
    fn from(x: OsString) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Bos<'b, OsStr>> for OsString {
    #[inline]
    fn from(x: Bos<'b, OsStr>) -> Self {
        x.into_owned()
    }
}

impl<'b> From<PathBuf> for Bos<'b, Path> {
    #[inline]
    fn from(x: PathBuf) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Bos<'b, Path>> for PathBuf {
    #[inline]
    fn from(x: Bos<'b, Path>) -> Self {
        x.into_owned()
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use assert_matches::assert_matches;

    #[cfg(feature = "unstable")]
    #[test]
    fn share_owned() {
        use tap::Tap;

        let text = "example";
        let owned = Bos::<str>::Owned(text.to_owned());
        assert_matches!(&owned, &Bos::Owned(_));
        let shared = owned.tap_mut(|s| s.share_owned());
        assert_matches!(&shared, &Bos::RcOwned(_));
        assert_eq!(shared, Bos::Borrowed(text));
    }

    #[test]
    fn owned_into_static_shared() {
        let text = "example";
        let owned = Bos::<str>::Owned(text.to_owned());
        let shared = owned.into_static_shared();
        assert_matches!(&shared, &Bos::RcOwned(_));
        assert_eq!(shared, Bos::Borrowed(text));
    }

    #[test]
    fn borrowed_into_static_shared() {
        let text = "example";
        let borrowed = Bos::Borrowed(text);
        let shared = borrowed.into_static_shared();
        assert_matches!(&shared, &Bos::RcBorrowed(_));
        assert_eq!(shared, Bos::Borrowed(text));
    }
}
