use std::borrow::{Borrow, Cow};
use std::cmp::Ordering;
use std::ffi::{CStr, CString, OsStr, OsString};
use std::fmt;
use std::hash::Hash;
use std::ops::{Add, AddAssign, Deref};
use std::path::{Path, PathBuf};
use std::sync::Arc;

/// **A**tomic **B**orrowed, **O**wned or **S**hared smart pointer.
/// to dynamically sized borrowed types like [`str`], the borrowed form of [`String`].
///
/// See the [crate-level documentation][crate#borrowed-sized-or-boxed]
/// for a comparison of borrowed, sized and boxed smart pointers.
pub enum Abos<'b, B>
where
    B: 'b + ToOwned + ?Sized,
{
    Borrowed(&'b B),
    Owned(B::Owned),
    ArcBorrowed(Arc<B>),
    ArcOwned(Arc<B::Owned>),
}

impl<'b, B: 'b + ToOwned + ?Sized> Abos<'b, B> {
    /// Extracts the owned data.
    ///
    /// Clones the data if it is not already owned.
    #[inline]
    pub fn into_owned(self) -> B::Owned {
        match self {
            Abos::Borrowed(x) => x.to_owned(),
            Abos::Owned(x) => x,
            Abos::ArcBorrowed(x) => x.as_ref().to_owned(),
            Abos::ArcOwned(x) => {
                Arc::try_unwrap(x).unwrap_or_else(|x| Borrow::borrow(&*x).to_owned())
            }
        }
    }

    /// Convert [`Abos::Borrowed`] to [`Abos::Owned`] so that the new `Abos` has a `'static` lifetime.
    #[inline]
    pub fn into_static_unborrowed(self) -> Abos<'static, B> {
        match self {
            Abos::Borrowed(x) => Abos::Owned(x.to_owned()),
            Abos::Owned(x) => Abos::Owned(x),
            Abos::ArcBorrowed(x) => Abos::ArcBorrowed(x),
            Abos::ArcOwned(x) => Abos::ArcOwned(x),
        }
    }

    /// Convert to a shared variant so that the new `Abos` has a `'static` lifetime and [`Clone`] will not copy the inner data.
    #[inline]
    pub fn into_static_shared(self) -> Abos<'static, B>
    where
        for<'a> &'a B: Into<Arc<B>>,
    {
        match self {
            Abos::Borrowed(x) => Abos::ArcBorrowed(x.into()),
            Abos::Owned(x) => Abos::ArcOwned(x.into()),
            Abos::ArcBorrowed(x) => Abos::ArcBorrowed(x),
            Abos::ArcOwned(x) => Abos::ArcOwned(x),
        }
    }

    /// This is unstable and only available with the `unstable` feature.
    #[cfg(any(doc, feature = "unstable"))] // reason: recently added
    #[inline]
    pub fn share_owned(&mut self)
    where
        B::Owned: Default,
    {
        if let Self::Owned(x) = self {
            let x = std::mem::take(x);
            *self = Self::ArcOwned(x.into());
        }
    }

    /// Acquires a mutable reference to the owned form of the data.
    ///
    /// Clones the data if it is not already owned.
    #[inline]
    pub fn to_mut(&mut self) -> &mut B::Owned
    where
        B::Owned: Clone,
    {
        match self {
            Abos::Borrowed(x) => {
                *self = Abos::Owned(x.to_owned());
                self.to_mut()
            }
            Abos::Owned(x) => x,
            Abos::ArcBorrowed(x) => {
                *self = Abos::Owned(x.as_ref().to_owned());
                self.to_mut()
            }
            Abos::ArcOwned(x) => Arc::make_mut(x),
        }
    }

    #[inline]
    fn internal_borrow(&self) -> &B {
        match self {
            Abos::Borrowed(x) => x,
            Abos::Owned(x) => x.borrow(),
            Abos::ArcBorrowed(x) => x.deref(),
            Abos::ArcOwned(x) => x.deref().borrow(),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized, Rhs> Add<Rhs> for Abos<'b, B>
where
    B::Owned: Add<Rhs>,
{
    type Output = <B::Owned as Add<Rhs>>::Output;

    #[inline]
    fn add(self, rhs: Rhs) -> <B::Owned as Add<Rhs>>::Output {
        self.into_owned() + rhs
    }
}

impl<'b, B: 'b + ToOwned + ?Sized, Rhs> AddAssign<Rhs> for Abos<'b, B>
where
    B::Owned: AddAssign<Rhs> + Clone,
{
    #[inline]
    fn add_assign(&mut self, rhs: Rhs) {
        *self.to_mut() += rhs;
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> AsRef<B> for Abos<'b, B> {
    #[inline]
    fn as_ref(&self) -> &B {
        self.internal_borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Borrow<B> for Abos<'b, B> {
    #[inline]
    fn borrow(&self) -> &B {
        self.internal_borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Clone for Abos<'b, B>
where
    B::Owned: Clone,
{
    #[inline]
    fn clone(&self) -> Self {
        match self {
            Self::Borrowed(x) => Self::Borrowed(x),
            Self::Owned(x) => Self::Owned(x.clone()),
            Self::ArcBorrowed(x) => Self::ArcBorrowed(x.clone()),
            Self::ArcOwned(x) => Self::ArcOwned(x.clone()),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Deref for Abos<'b, B> {
    type Target = B;

    #[inline]
    fn deref(&self) -> &Self::Target {
        self.borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Default for Abos<'b, B>
where
    B::Owned: Default,
{
    /// Creates an owned Abos with the default value for the contained owned type.
    #[inline]
    fn default() -> Self {
        Self::Owned(B::Owned::default())
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> fmt::Debug for Abos<'b, B>
where
    B: fmt::Debug,
    B::Owned: fmt::Debug,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Abos::Borrowed(x) => f.debug_tuple("Abos::Borrowed").field(x).finish(),
            Abos::Owned(x) => f.debug_tuple("Abos::Owned").field(x).finish(),
            Abos::ArcBorrowed(x) => f.debug_tuple("Abos::ArcBorrowed").field(x).finish(),
            Abos::ArcOwned(x) => f.debug_tuple("Abos::ArcOwned").field(x).finish(),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> fmt::Display for Abos<'b, B>
where
    B: fmt::Display,
    B::Owned: fmt::Display,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::Borrowed(x) => fmt::Display::fmt(x, f),
            Self::Owned(x) => fmt::Display::fmt(x, f),
            Self::ArcBorrowed(x) => fmt::Display::fmt(x, f),
            Self::ArcOwned(x) => fmt::Display::fmt(x, f),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Hash for Abos<'b, B>
where
    B: Hash,
{
    #[inline]
    fn hash<H: std::hash::Hasher>(&self, state: &mut H) {
        self.internal_borrow().hash(state)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialEq<B> for Abos<'b, B>
where
    B: PartialEq,
{
    #[inline]
    fn eq(&self, other: &B) -> bool {
        self.internal_borrow() == other
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialEq for Abos<'b, B>
where
    B: PartialEq,
{
    #[inline]
    fn eq(&self, other: &Self) -> bool {
        self.internal_borrow() == other.internal_borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Eq for Abos<'b, B> where B: Eq {}

impl<'b, B: 'b + ToOwned + ?Sized> PartialOrd<B> for Abos<'b, B>
where
    B: PartialOrd,
{
    #[inline]
    fn partial_cmp(&self, other: &B) -> Option<Ordering> {
        self.internal_borrow().partial_cmp(other)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialOrd for Abos<'b, B>
where
    B: PartialOrd,
{
    #[inline]
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        self.internal_borrow().partial_cmp(other.internal_borrow())
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Ord for Abos<'b, B>
where
    B: Ord,
{
    #[inline]
    fn cmp(&self, other: &Self) -> Ordering {
        self.internal_borrow().cmp(other.internal_borrow())
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> From<Cow<'b, B>> for Abos<'b, B> {
    #[inline]
    fn from(p: Cow<'b, B>) -> Self {
        match p {
            Cow::Borrowed(x) => Self::Borrowed(x),
            Cow::Owned(x) => Self::Owned(x),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> From<&'b B> for Abos<'b, B> {
    #[inline]
    fn from(x: &'b B) -> Self {
        Self::Borrowed(x)
    }
}

// TODO: remove this impl before 1.0 for more flexibility going forwards.
// This impl currently blocks implementing From<Arc<str>> for Abos<str> etc.
impl<'b, B: 'b + ToOwned + ?Sized> From<Arc<B::Owned>> for Abos<'b, B> {
    #[inline]
    fn from(x: Arc<B::Owned>) -> Self {
        Self::ArcOwned(x)
    }
}

impl<'b, T> From<Vec<T>> for Abos<'b, [T]>
where
    T: Clone,
{
    #[inline]
    fn from(x: Vec<T>) -> Self {
        Self::Owned(x)
    }
}

impl<'b, T> From<Abos<'b, [T]>> for Vec<T>
where
    [T]: ToOwned<Owned = Vec<T>>,
{
    #[inline]
    fn from(x: Abos<'b, [T]>) -> Self {
        x.into_owned()
    }
}

impl<'b> From<String> for Abos<'b, str> {
    #[inline]
    fn from(x: String) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Abos<'b, str>> for String {
    #[inline]
    fn from(x: Abos<'b, str>) -> Self {
        x.into_owned()
    }
}

impl<'b> From<CString> for Abos<'b, CStr> {
    #[inline]
    fn from(x: CString) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Abos<'b, CStr>> for CString {
    #[inline]
    fn from(x: Abos<'b, CStr>) -> Self {
        x.into_owned()
    }
}

impl<'b> From<OsString> for Abos<'b, OsStr> {
    #[inline]
    fn from(x: OsString) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Abos<'b, OsStr>> for OsString {
    #[inline]
    fn from(x: Abos<'b, OsStr>) -> Self {
        x.into_owned()
    }
}

impl<'b> From<PathBuf> for Abos<'b, Path> {
    #[inline]
    fn from(x: PathBuf) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Abos<'b, Path>> for PathBuf {
    #[inline]
    fn from(x: Abos<'b, Path>) -> Self {
        x.into_owned()
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use assert_matches::assert_matches;

    #[cfg(feature = "unstable")]
    #[test]
    fn share_owned() {
        use tap::Tap;

        let text = "example";
        let owned = Abos::<str>::Owned(text.to_owned());
        assert_matches!(&owned, &Abos::Owned(_));
        let shared = owned.tap_mut(|s| s.share_owned());
        assert_matches!(&shared, &Abos::ArcOwned(_));
        assert_eq!(shared, Abos::Borrowed(text));
    }

    #[test]
    fn owned_into_static_shared() {
        let text = "example";
        let owned = Abos::<str>::Owned(text.to_owned());
        let shared = owned.into_static_shared();
        assert_matches!(&shared, &Abos::ArcOwned(_));
        assert_eq!(shared, Abos::Borrowed(text));
    }

    #[test]
    fn borrowed_into_static_shared() {
        let text = "example";
        let borrowed = Abos::Borrowed(text);
        let shared = borrowed.into_static_shared();
        assert_matches!(&shared, &Abos::ArcBorrowed(_));
        assert_eq!(shared, Abos::Borrowed(text));
    }
}
