use std::borrow::{Borrow, Cow};
use std::cmp::Ordering;
use std::fmt;
use std::hash::Hash;
use std::ops::{Add, AddAssign, Deref};
use std::rc::Rc;
use std::sync::Arc;

/// **B**orrowed, **O**wned or **S**hared smart pointer.
pub enum Bos<'b, B>
where
    B: 'b + ToOwned + ?Sized,
{
    Borrowed(&'b B),
    Owned(B::Owned),
    ArcBorrowed(Arc<B>),
    ArcOwned(Arc<B::Owned>),
    RcBorrowed(Rc<B>),
    RcOwned(Rc<B::Owned>),
}

impl<'b, B: 'b + ToOwned + ?Sized> Bos<'b, B> {
    /// Extracts the owned data.
    ///
    /// Clones the data if it is not already owned.
    #[inline]
    pub fn into_owned(self) -> B::Owned {
        match self {
            Bos::Borrowed(x) => x.to_owned(),
            Bos::Owned(x) => x,
            Bos::ArcBorrowed(x) => x.as_ref().to_owned(),
            Bos::ArcOwned(x) => {
                Arc::try_unwrap(x).unwrap_or_else(|x| Borrow::borrow(&*x).to_owned())
            }
            Bos::RcBorrowed(x) => x.as_ref().to_owned(),
            Bos::RcOwned(x) => Rc::try_unwrap(x).unwrap_or_else(|x| Borrow::borrow(&*x).to_owned()),
        }
    }

    /// This is unstable and only available with the `unstable` feature.
    #[cfg(any(doc, feature = "unstable"))] // reason: recently added, maybe we should add a check, if Borrowed should become Shared instead?
    #[inline]
    pub fn into_static(self) -> Bos<'static, B> {
        match self {
            Bos::Borrowed(x) => Bos::Owned(x.to_owned()),
            Bos::Owned(x) => Bos::Owned(x),
            Bos::ArcBorrowed(x) => Bos::ArcBorrowed(x),
            Bos::ArcOwned(x) => Bos::ArcOwned(x),
            Bos::RcBorrowed(x) => Bos::RcBorrowed(x),
            Bos::RcOwned(x) => Bos::RcOwned(x),
        }
    }

    /// Acquires a mutable reference to the owned form of the data.
    ///
    /// Clones the data if it is not already owned.
    #[inline]
    pub fn to_mut(&mut self) -> &mut B::Owned
    where
        B::Owned: Clone,
    {
        match self {
            Bos::Borrowed(x) => {
                *self = Bos::Owned(x.to_owned());
                self.to_mut()
            }
            Bos::Owned(x) => x,
            Bos::ArcBorrowed(x) => {
                *self = Bos::Owned(x.as_ref().to_owned());
                self.to_mut()
            }
            Bos::ArcOwned(x) => Arc::make_mut(x),
            Bos::RcBorrowed(x) => {
                *self = Bos::Owned(x.as_ref().to_owned());
                self.to_mut()
            }
            Bos::RcOwned(x) => Rc::make_mut(x),
        }
    }

    #[inline]
    fn internal_borrow(&self) -> &B {
        match self {
            Bos::Borrowed(x) => x,
            Bos::Owned(x) => x.borrow(),
            Bos::ArcBorrowed(x) => x.deref(),
            Bos::ArcOwned(x) => x.deref().borrow(),
            Bos::RcBorrowed(x) => x.deref(),
            Bos::RcOwned(x) => x.deref().borrow(),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized, Rhs> Add<Rhs> for Bos<'b, B>
where
    B::Owned: Add<Rhs>,
{
    type Output = <B::Owned as Add<Rhs>>::Output;

    #[inline]
    fn add(self, rhs: Rhs) -> <B::Owned as Add<Rhs>>::Output {
        self.into_owned() + rhs
    }
}

impl<'b, B: 'b + ToOwned + ?Sized, Rhs> AddAssign<Rhs> for Bos<'b, B>
where
    B::Owned: AddAssign<Rhs> + Clone,
{
    #[inline]
    fn add_assign(&mut self, rhs: Rhs) {
        *self.to_mut() += rhs;
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> AsRef<B> for Bos<'b, B> {
    #[inline]
    fn as_ref(&self) -> &B {
        self.internal_borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Borrow<B> for Bos<'b, B> {
    #[inline]
    fn borrow(&self) -> &B {
        self.internal_borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Clone for Bos<'b, B>
where
    B::Owned: Clone,
{
    #[inline]
    fn clone(&self) -> Self {
        match self {
            Self::Borrowed(x) => Self::Borrowed(x),
            Self::Owned(x) => Self::Owned(x.clone()),
            Self::ArcBorrowed(x) => Self::ArcBorrowed(x.clone()),
            Self::ArcOwned(x) => Self::ArcOwned(x.clone()),
            Self::RcBorrowed(x) => Self::RcBorrowed(x.clone()),
            Self::RcOwned(x) => Self::RcOwned(x.clone()),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Deref for Bos<'b, B> {
    type Target = B;

    #[inline]
    fn deref(&self) -> &Self::Target {
        self.borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Default for Bos<'b, B>
where
    B::Owned: Default,
{
    /// Creates an owned Bos with the default value for the contained owned type.
    #[inline]
    fn default() -> Self {
        Self::Owned(B::Owned::default())
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> fmt::Debug for Bos<'b, B>
where
    B: fmt::Debug,
    B::Owned: fmt::Debug,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Bos::Borrowed(x) => f.debug_tuple("Bos::Borrowed").field(x).finish(),
            Bos::Owned(x) => f.debug_tuple("Bos::Owned").field(x).finish(),
            Bos::ArcBorrowed(x) => f.debug_tuple("Bos::ArcBorrowed").field(x).finish(),
            Bos::ArcOwned(x) => f.debug_tuple("Bos::ArcOwned").field(x).finish(),
            Bos::RcBorrowed(x) => f.debug_tuple("Bos::RcBorrowed").field(x).finish(),
            Bos::RcOwned(x) => f.debug_tuple("Bos::RcOwned").field(x).finish(),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> fmt::Display for Bos<'b, B>
where
    B: fmt::Display,
    B::Owned: fmt::Display,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::Borrowed(x) => fmt::Display::fmt(x, f),
            Self::Owned(x) => fmt::Display::fmt(x, f),
            Self::ArcBorrowed(x) => fmt::Display::fmt(x, f),
            Self::ArcOwned(x) => fmt::Display::fmt(x, f),
            Self::RcBorrowed(x) => fmt::Display::fmt(x, f),
            Self::RcOwned(x) => fmt::Display::fmt(x, f),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Hash for Bos<'b, B>
where
    B: Hash,
{
    #[inline]
    fn hash<H: std::hash::Hasher>(&self, state: &mut H) {
        self.internal_borrow().hash(state)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialEq<B> for Bos<'b, B>
where
    B: PartialEq,
{
    #[inline]
    fn eq(&self, other: &B) -> bool {
        self.internal_borrow() == other
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialEq for Bos<'b, B>
where
    B: PartialEq,
{
    #[inline]
    fn eq(&self, other: &Self) -> bool {
        self.internal_borrow() == other.internal_borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Eq for Bos<'b, B> where B: Eq {}

impl<'b, B: 'b + ToOwned + ?Sized> PartialOrd<B> for Bos<'b, B>
where
    B: PartialOrd,
{
    #[inline]
    fn partial_cmp(&self, other: &B) -> Option<Ordering> {
        self.internal_borrow().partial_cmp(other)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialOrd for Bos<'b, B>
where
    B: PartialOrd,
{
    #[inline]
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        self.internal_borrow().partial_cmp(other.internal_borrow())
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Ord for Bos<'b, B>
where
    B: Ord,
{
    #[inline]
    fn cmp(&self, other: &Self) -> Ordering {
        self.internal_borrow().cmp(other.internal_borrow())
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> From<Cow<'b, B>> for Bos<'b, B> {
    #[inline]
    fn from(c: Cow<'b, B>) -> Self {
        match c {
            Cow::Borrowed(x) => Self::Borrowed(x),
            Cow::Owned(x) => Self::Owned(x),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> From<&'b B> for Bos<'b, B> {
    #[inline]
    fn from(x: &'b B) -> Self {
        Self::Borrowed(x)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> From<Arc<B::Owned>> for Bos<'b, B> {
    #[inline]
    fn from(x: Arc<B::Owned>) -> Self {
        Self::ArcOwned(x)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> From<Rc<B::Owned>> for Bos<'b, B> {
    #[inline]
    fn from(x: Rc<B::Owned>) -> Self {
        Self::RcOwned(x)
    }
}
