/*!
This crate provides flexible **B**orrowed, **O**wned or **S**hared (B.O.S.) smart pointers.
They are like std's [`Cow`][std::borrow::Cow]
but with additonal [`Arc`][std::sync::Arc] and [`Rc`][std::rc::Rc] variants.

# [`Abos`] or [`Bos`]

[`Abos`] implements [`Send`] and [`Sync`] but does *not* have a [`Rc`][std::rc::Rc] variant.
The A stands for "atomic".

[`Bos`] does *not* implement [`Send`] or [`Sync`] but does have a [`Rc`][std::rc::Rc] variant.

# Examples

```
use bos::Abos;

fn maybe_replace_with_reply(string: &mut Abos<str>) {
    if string.contains("Hello") {
        *string.to_mut() = String::from("Hello to you, too.");
    }
}

let mut string = Abos::Borrowed("Hello bos!");
maybe_replace_with_reply(&mut string);
assert_eq!(&string, "Hello to you, too.")
```

```
use bos::Abos;

struct Header<'a> {
    name: bos::AStr<'a>, // AStr<'a> is a type alias for Abos<'a, str>
    bytes: Abos<'a, [u8]>,
}

fn add_global_header(header: Header<'static>) {
    // ...
}
```

```
struct Logger {
    // Use the 'static lifetime, if you don't want
    // to introduce generic lifetimes to your struct.
    name: bos::AStr<'static>,
}
```

# [`AStr`] and [`Str`] types

[`Abos`] and [`Bos`] are often used with [`str`] or similar types.
Feel free to have a look at our handy [type aliases](#types) below.
*/

#![forbid(unsafe_code)]

mod abos;
mod bos;

pub use crate::abos::Abos;
pub use crate::bos::Bos;

#[cfg(feature = "unstable")]
use std::{
    ffi::{CStr, OsStr},
    path::Path,
};

/// Atomic [`str`] smart pointer, that is **B**orrowed, **O**wned or **S**hared.
pub type AStr<'a> = Abos<'a, str>;

/// [`str`] smart pointer, that is **B**orrowed, **O**wned or **S**hared.
pub type Str<'a> = Bos<'a, str>;

#[cfg(feature = "unstable")]
pub type StaticAStr = Abos<'static, str>;

#[cfg(feature = "unstable")]
pub type StaticStr = Bos<'static, str>;

#[cfg(feature = "unstable")]
pub type AbosOsStr<'a> = Abos<'a, OsStr>;

#[cfg(feature = "unstable")]
pub type BosOsStr<'a> = Bos<'a, OsStr>;

#[cfg(feature = "unstable")]
pub type AbosCStr<'a> = Abos<'a, CStr>;

#[cfg(feature = "unstable")]
pub type BosCStr<'a> = Bos<'a, CStr>;

#[cfg(feature = "unstable")]
pub type AbosPath<'a> = Abos<'a, Path>;

#[cfg(feature = "unstable")]
pub type BosPath<'a> = Bos<'a, Path>;
