use std::borrow::Borrow;
use std::cmp::Ordering;
use std::fmt;
use std::hash::Hash;
use std::ops::{Add, AddAssign, Deref};
use std::rc::Rc;
use std::sync::Arc;

/// **B**orrowed, **O**wned or **S**hared smart pointer.
#[non_exhaustive]
pub enum Bos<'b, B>
where
    B: 'b + ToOwned + ?Sized,
{
    Borrowed(&'b B),
    Owned(<B as ToOwned>::Owned),
    Arc(Arc<B>),
    Rc(Rc<B>),
}

impl<'b, B: 'b + ToOwned + ?Sized> Bos<'b, B> {
    pub fn into_owned(self) -> <B as ToOwned>::Owned {
        match self {
            Bos::Borrowed(x) => x.to_owned(),
            Bos::Owned(x) => x,
            Bos::Arc(x) => x.as_ref().to_owned(),
            Bos::Rc(x) => x.as_ref().to_owned(),
        }
    }

    pub fn into_static(self) -> Bos<'static, B> {
        match self {
            Bos::Borrowed(x) => Bos::Owned(x.to_owned()),
            Bos::Owned(x) => Bos::Owned(x),
            Bos::Arc(x) => Bos::Arc(x),
            Bos::Rc(x) => Bos::Rc(x),
        }
    }

    pub fn to_mut(&mut self) -> &mut <B as ToOwned>::Owned {
        match self {
            Bos::Borrowed(x) => {
                *self = Bos::Owned(x.to_owned());
                self.to_mut()
            }
            Bos::Owned(o) => o,
            Bos::Arc(x) => {
                *self = Bos::Owned(x.as_ref().to_owned());
                self.to_mut()
            }
            Bos::Rc(x) => {
                *self = Bos::Owned(x.as_ref().to_owned());
                self.to_mut()
            }
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized, Rhs> Add<Rhs> for Bos<'b, B>
where
    <B as ToOwned>::Owned: Add<Rhs>,
{
    type Output = <<B as ToOwned>::Owned as Add<Rhs>>::Output;

    fn add(self, rhs: Rhs) -> <<B as ToOwned>::Owned as Add<Rhs>>::Output {
        self.into_owned() + rhs
    }
}

impl<'b, B: 'b + ToOwned + ?Sized, Rhs> AddAssign<Rhs> for Bos<'b, B>
where
    <B as ToOwned>::Owned: AddAssign<Rhs>,
{
    #[inline]
    fn add_assign(&mut self, rhs: Rhs) {
        *self.to_mut() += rhs;
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> AsRef<B> for Bos<'b, B> {
    #[inline]
    fn as_ref(&self) -> &B {
        self.borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Borrow<B> for Bos<'b, B> {
    #[inline]
    fn borrow(&self) -> &B {
        match self {
            Self::Borrowed(x) => x,
            Self::Owned(x) => x.borrow(),
            Self::Arc(x) => x.as_ref().borrow(),
            Self::Rc(x) => x.as_ref().borrow(),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Clone for Bos<'b, B>
where
    <B as ToOwned>::Owned: Clone,
{
    #[inline]
    fn clone(&self) -> Self {
        match self {
            Self::Borrowed(x) => Self::Borrowed(x),
            Self::Owned(x) => Self::Owned(x.clone()),
            Self::Arc(x) => Self::Arc(x.clone()),
            Self::Rc(x) => Self::Rc(x.clone()),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Deref for Bos<'b, B> {
    type Target = B;

    #[inline]
    fn deref(&self) -> &Self::Target {
        self.borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Default for Bos<'b, B>
where
    <B as ToOwned>::Owned: Default,
{
    /// Creates an owned Bos with the default value for the contained owned type.
    #[inline]
    fn default() -> Self {
        Self::Owned(<B as ToOwned>::Owned::default())
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> fmt::Debug for Bos<'b, B>
where
    B: fmt::Debug,
    <B as ToOwned>::Owned: fmt::Debug,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Bos::Borrowed(x) => f.debug_tuple("Bos::Borrowed").field(x).finish(),
            Bos::Owned(x) => f.debug_tuple("Bos::Owned").field(x).finish(),
            Bos::Arc(x) => f.debug_tuple("Bos::Arc").field(x).finish(),
            Bos::Rc(x) => f.debug_tuple("Bos::Rc").field(x).finish(),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> fmt::Display for Bos<'b, B>
where
    B: fmt::Display,
    <B as ToOwned>::Owned: fmt::Display,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::Borrowed(x) => fmt::Display::fmt(x, f),
            Self::Owned(x) => fmt::Display::fmt(x, f),
            Self::Arc(x) => fmt::Display::fmt(x, f),
            Self::Rc(x) => fmt::Display::fmt(x, f),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Hash for Bos<'b, B>
where
    B: Hash,
{
    fn hash<H: std::hash::Hasher>(&self, state: &mut H) {
        let b: &B = self.borrow();
        b.hash(state)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialEq<B> for Bos<'b, B>
where
    B: PartialEq,
{
    fn eq(&self, other: &B) -> bool {
        let b: &B = self.borrow();
        b == other
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialEq for Bos<'b, B>
where
    B: PartialEq,
{
    fn eq(&self, other: &Self) -> bool {
        let b: &B = self.borrow();
        let other_b: &B = other.borrow();
        b == other_b
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Eq for Bos<'b, B> where B: Eq {}

impl<'b, B: 'b + ToOwned + ?Sized> PartialOrd<B> for Bos<'b, B>
where
    B: PartialOrd,
{
    fn partial_cmp(&self, other: &B) -> Option<Ordering> {
        let b: &B = self.borrow();
        b.partial_cmp(other)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialOrd for Bos<'b, B>
where
    B: PartialOrd,
{
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        let b: &B = self.borrow();
        let other_b: &B = other.borrow();
        b.partial_cmp(other_b)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Ord for Bos<'b, B>
where
    B: Ord,
{
    fn cmp(&self, other: &Self) -> Ordering {
        let b: &B = self.borrow();
        let other_b: &B = other.borrow();
        b.cmp(other_b)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> From<&'b B> for Bos<'b, B> {
    fn from(b: &'b B) -> Self {
        Self::Borrowed(b)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> From<Arc<B>> for Bos<'b, B> {
    fn from(a: Arc<B>) -> Self {
        Self::Arc(a)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> From<Rc<B>> for Bos<'b, B> {
    fn from(r: Rc<B>) -> Self {
        Self::Rc(r)
    }
}
