use std::borrow::Borrow;
use std::cmp::Ordering;
use std::fmt;
use std::hash::Hash;
use std::ops::{Add, AddAssign, Deref};
use std::sync::Arc;

/// **A**tomic **B**orrowed, **O**wned or **S**hared smart pointer.
#[non_exhaustive]
pub enum Abos<'b, B>
where
    B: 'b + ToOwned + ?Sized,
{
    Borrowed(&'b B),
    Owned(<B as ToOwned>::Owned),
    Arc(Arc<B>),
}

impl<'b, B: 'b + ToOwned + ?Sized> Abos<'b, B> {
    pub fn into_owned(self) -> <B as ToOwned>::Owned {
        match self {
            Abos::Borrowed(x) => x.to_owned(),
            Abos::Owned(x) => x,
            Abos::Arc(x) => x.as_ref().to_owned(),
        }
    }

    pub fn into_arc(self) -> Arc<B>
    where
        <B as ToOwned>::Owned: Into<Arc<B>>,
    {
        match self {
            Abos::Borrowed(x) => x.to_owned().into(),
            Abos::Owned(x) => x.into(),
            Abos::Arc(x) => x,
        }
    }

    pub fn into_static(self) -> Abos<'static, B> {
        match self {
            Abos::Borrowed(x) => Abos::Owned(x.to_owned()),
            Abos::Owned(x) => Abos::Owned(x),
            Abos::Arc(x) => Abos::Arc(x),
        }
    }

    pub fn to_mut(&mut self) -> &mut <B as ToOwned>::Owned {
        match self {
            Abos::Borrowed(x) => {
                *self = Abos::Owned(x.to_owned());
                self.to_mut()
            }
            Abos::Owned(x) => x,
            Abos::Arc(x) => {
                *self = Abos::Owned(x.as_ref().to_owned());
                self.to_mut()
            }
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized, Rhs> Add<Rhs> for Abos<'b, B>
where
    <B as ToOwned>::Owned: Add<Rhs>,
{
    type Output = <<B as ToOwned>::Owned as Add<Rhs>>::Output;

    fn add(self, rhs: Rhs) -> <<B as ToOwned>::Owned as Add<Rhs>>::Output {
        self.into_owned() + rhs
    }
}

impl<'b, B: 'b + ToOwned + ?Sized, Rhs> AddAssign<Rhs> for Abos<'b, B>
where
    <B as ToOwned>::Owned: AddAssign<Rhs>,
{
    #[inline]
    fn add_assign(&mut self, rhs: Rhs) {
        *self.to_mut() += rhs;
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> AsRef<B> for Abos<'b, B> {
    #[inline]
    fn as_ref(&self) -> &B {
        self.borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Borrow<B> for Abos<'b, B> {
    #[inline]
    fn borrow(&self) -> &B {
        match self {
            Self::Borrowed(x) => x,
            Self::Owned(x) => x.borrow(),
            Self::Arc(x) => x.as_ref().borrow(),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Clone for Abos<'b, B>
where
    <B as ToOwned>::Owned: Clone,
{
    #[inline]
    fn clone(&self) -> Self {
        match self {
            Self::Borrowed(x) => Self::Borrowed(x),
            Self::Owned(x) => Self::Owned(x.clone()),
            Self::Arc(x) => Self::Arc(x.clone()),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Deref for Abos<'b, B> {
    type Target = B;

    #[inline]
    fn deref(&self) -> &Self::Target {
        self.borrow()
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Default for Abos<'b, B>
where
    <B as ToOwned>::Owned: Default,
{
    /// Creates an owned Abos with the default value for the contained owned type.
    #[inline]
    fn default() -> Self {
        Self::Owned(<B as ToOwned>::Owned::default())
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> fmt::Debug for Abos<'b, B>
where
    B: fmt::Debug,
    <B as ToOwned>::Owned: fmt::Debug,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Abos::Borrowed(x) => f.debug_tuple("Abos::Borrowed").field(x).finish(),
            Abos::Owned(x) => f.debug_tuple("Abos::Owned").field(x).finish(),
            Abos::Arc(x) => f.debug_tuple("Abos::Arc").field(x).finish(),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> fmt::Display for Abos<'b, B>
where
    B: fmt::Display,
    <B as ToOwned>::Owned: fmt::Display,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::Borrowed(x) => fmt::Display::fmt(x, f),
            Self::Owned(x) => fmt::Display::fmt(x, f),
            Self::Arc(x) => fmt::Display::fmt(x, f),
        }
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Hash for Abos<'b, B>
where
    B: Hash,
{
    fn hash<H: std::hash::Hasher>(&self, state: &mut H) {
        let b: &B = self.borrow();
        b.hash(state)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialEq<B> for Abos<'b, B>
where
    B: PartialEq,
{
    fn eq(&self, other: &B) -> bool {
        let b: &B = self.borrow();
        b == other
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialEq for Abos<'b, B>
where
    B: PartialEq,
{
    fn eq(&self, other: &Self) -> bool {
        let b: &B = self.borrow();
        let other_b: &B = other.borrow();
        b == other_b
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Eq for Abos<'b, B> where B: Eq {}

impl<'b, B: 'b + ToOwned + ?Sized> PartialOrd<B> for Abos<'b, B>
where
    B: PartialOrd,
{
    fn partial_cmp(&self, other: &B) -> Option<Ordering> {
        let b: &B = self.borrow();
        b.partial_cmp(other)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> PartialOrd for Abos<'b, B>
where
    B: PartialOrd,
{
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        let b: &B = self.borrow();
        let other_b: &B = other.borrow();
        b.partial_cmp(other_b)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> Ord for Abos<'b, B>
where
    B: Ord,
{
    fn cmp(&self, other: &Self) -> Ordering {
        let b: &B = self.borrow();
        let other_b: &B = other.borrow();
        b.cmp(other_b)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> From<&'b B> for Abos<'b, B> {
    fn from(b: &'b B) -> Self {
        Self::Borrowed(b)
    }
}

impl<'b, B: 'b + ToOwned + ?Sized> From<Arc<B>> for Abos<'b, B> {
    fn from(a: Arc<B>) -> Self {
        Self::Arc(a)
    }
}
