# Borsh in Rust &emsp; [![Build Status]][travis-ci] [![Latest Version]][crates.io] [![borsh: rustc 1.40+]][Rust 1.40] [![License Apache-2.0 badge]][License Apache-2.0] [![License MIT badge]][License MIT]

[Borsh]: https://borsh.io
[Build Status]: https://travis-ci.com/near/borsh-rs.svg?branch=master
[travis-ci]: https://travis-ci.com/near/borsh-rs
[Latest Version]: https://img.shields.io/crates/v/borsh.svg
[crates.io]: https://crates.io/crates/borsh
[borsh: rustc 1.40+]: https://img.shields.io/badge/rustc-1.40+-lightgray.svg
[Rust 1.40]: https://blog.rust-lang.org/2019/12/19/Rust-1.40.0.html
[License Apache-2.0 badge]: https://img.shields.io/badge/license-Apache2.0-blue.svg
[License Apache-2.0]: https://opensource.org/licenses/Apache-2.0
[License MIT badge]: https://img.shields.io/badge/license-MIT-blue.svg
[License MIT]: https://opensource.org/licenses/MIT

**borsh-rs** is Rust implementation of the [Borsh] binary serialization format.

Borsh stands for _Binary Object Representation Serializer for Hashing_. It is meant to be used in
security-critical projects as it prioritizes [consistency, safety, speed][Borsh], and comes with a
strict [specification](https://github.com/near/borsh#specification).

## Example

```rust
use borsh::{BorshSerialize, BorshDeserialize};

#[derive(BorshSerialize, BorshDeserialize, PartialEq, Debug)]
struct A {
    x: u64,
    y: String,
}

#[test]
fn test_simple_struct() {
    let a = A {
        x: 3301,
        y: "liber primus".to_string(),
    };
    let encoded_a = a.try_to_vec().unwrap();
    let decoded_a = A::try_from_slice(&encoded_a).unwrap();
    assert_eq!(a, decoded_a);
}
```

## Features

Opting out from Serde allows borsh to have some features that currently are not available for serde-compatible serializers.
Currently we support two features: `borsh_init` and `borsh_skip` (the former one not available in Serde).

`borsh_init` allows to automatically run an initialization function right after deserialization. This adds a lot of convenience for objects that are architectured to be used as strictly immutable. Usage example:

```rust
#[derive(BorshSerialize, BorshDeserialize)]
#[borsh_init(init)]
struct Message {
    message: String,
    timestamp: u64,
    public_key: CryptoKey,
    signature: CryptoSignature
    hash: CryptoHash
}

impl Message {
    pub fn init(&mut self) {
        self.hash = CryptoHash::new().write_string(self.message).write_u64(self.timestamp);
        self.signature.verify(self.hash, self.public_key);
    }
}
```

`borsh_skip` allows to skip serializing/deserializing fields, assuming they implement `Default` trait, similary to `#[serde(skip)]`.

```rust
#[derive(BorshSerialize, BorshDeserialize)]
struct A {
    x: u64,
    #[borsh_skip]
    y: f32,
}
```

## Releasing

Before you release, make sure CHANGELOG.md is up to date.

Use [`cargo-workspaces`](https://github.com/pksunkara/cargo-workspaces) to save time.

### Bump Versions

```sh
cargo workspaces version --force 'borsh*' --exact --no-individual-tags patch
```

This will bump all the versions to the next "patch" release (see `cargo workspaces version --help`
for more options), create a new commit, push `v0.x.x` tag, push to the master.

### Publish

To publish on crates.io the version that is currently in git:

```sh
cargo workspaces publish --from-git --skip-published
```

Alternatively, you may want to combine the version bumping with publishing:

```sh
cargo workspaces publish --force 'borsh*' --exact --no-individual-tags patch
```

### Release on GitHub

1. Navigate to the [New Release](https://github.com/near/borsh-rs/releases/new) page
2. Enter the tag name, e.g. `v0.8.0`
3. Write down the release log (basically, copy-paste from the CHANGELOG)
4. Publish the release

## License

This repository is distributed under the terms of both the MIT license and the Apache License (Version 2.0).
See [LICENSE-MIT](LICENSE-MIT) and [LICENSE-APACHE](LICENSE-APACHE) for details.
