// bls.rs
//
// Copyright 2022 Alberto Ruiz <aruiz@gnome.org>
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.
//
// SPDX-License-Identifier: MPL-2.0


#![cfg_attr(not(feature = "std"), no_std)]

#![cfg(not(feature = "std"))]
extern crate alloc;

#[cfg(not(feature = "std"))]
use alloc::string::String;
#[cfg(not(feature = "std"))]
use alloc::vec::Vec;
#[cfg(not(feature = "std"))]
use alloc::format;

pub type KeyValue = Vec<String>;

#[derive(Debug)]
pub enum BLSValue {
    Value(String),
    ValueWithComment(String, String),
}

#[derive(Debug)]
pub struct BLSEntry {
    pub title: Option<BLSValue>,
    pub version: Option<BLSValue>,
    pub machine_id: Option<BLSValue>,
    pub sort_key: Option<BLSValue>,
    pub linux: BLSValue,
    pub efi: Option<BLSValue>,
    pub initrd: Vec<BLSValue>,
    pub options: Vec<BLSValue>,
    pub devicetree: Option<BLSValue>,
    pub devicetree_overlay: Option<BLSValue>,
    pub architecture: Option<BLSValue>,
    pub grub_hotkey: Option<BLSValue>,
    pub grub_users: Option<BLSValue>,
    pub grub_class: Vec<BLSValue>,
    pub grub_arg: Option<BLSValue>,
    // NOTE: All comments are moved to the header of the file upon rendering back the content
    pub comments: Vec<String>,
}

impl BLSEntry {
    pub fn new() -> BLSEntry {
        BLSEntry {
            title: None,
            version: None,
            machine_id: None,
            sort_key: None,
            linux: BLSValue::Value(String::new()),
            efi: None,
            initrd: Vec::new(),
            options: Vec::new(),
            devicetree: None,
            devicetree_overlay: None,
            architecture: None,
            grub_hotkey: None,
            grub_users: None,
            grub_class: Vec::new(),
            grub_arg: None,
            comments: Vec::new(),
        }
    }

    pub fn parse(buffer: &str) -> Result<BLSEntry, String> {
        let mut entry = BLSEntry::new();
        let mut has_kernel = false;

        for line in buffer.lines() {
            let mut comment = None;
            // Extract the comment string from the line
            let line = if line.contains("#") {
                let split: Vec<_> = line.splitn(2, "#").collect();
                comment = Some(String::from(split[1]));
                split[0]
            } else {
                line
            };

            // NOTE: For now we put all comment lines in the header
            if line.trim().contains(" ") {
                let key_value: Vec<&str> = line.trim().splitn(2, " ").collect();
                if key_value[0] == "linux" {
                    has_kernel = true;
                }
                if let Err(error) = entry.set(key_value[0], String::from(key_value[1]), comment) {
                    return Err(error);
                }
            } else {
                match comment {
                    Some(comment) => {
                        entry.comments.push(comment);
                    }
                    None => {}
                }
            }
        }

        if has_kernel {
            Ok(entry)
        } else {
            Err(String::from("No 'linux' command found."))
        }
    }

    pub fn render(&self) -> String {
        let mut content = String::new();

        fn render_value(content: &mut String, key: &str, value: &BLSValue) {
            content.push_str(key);
            content.push(' ');
            match value {
                BLSValue::Value(value) => content.push_str(&value),
                BLSValue::ValueWithComment(value, comment) => {
                    content.push_str(&value);
                    content.push_str(" #");
                    content.push_str(&comment);
                }
            }
        }

        fn render_single_value(content: &mut String, key: &str, value: &Option<BLSValue>) {
            if let Some(value) = value {
                render_value(content, key, &value)
            }
        }

        fn render_multiple_values(content: &mut String, key: &str, values: &Vec<BLSValue>) {
            for val in values {
                render_value(content, key, &val)
            }
        }

        // We push all comments in the header
        for comment in &self.comments {
            content.push_str("#");
            content.push_str(&comment)
        }

        // Mandatory commands
        render_value(&mut content, "linux", &self.linux);

        // Optional commands
        render_single_value(&mut content, "title", &self.title);
        render_single_value(&mut content, "version", &self.version);
        render_single_value(&mut content, "machine-id", &self.machine_id);
        render_single_value(&mut content, "sort-key", &self.sort_key);
        render_single_value(&mut content, "efi", &self.efi);
        render_single_value(&mut content, "devicetree", &self.devicetree);
        render_single_value(&mut content, "devicetree-overlay", &self.devicetree_overlay);
        render_single_value(&mut content, "architecture", &self.architecture);
        render_single_value(&mut content, "grub_hotkey", &self.devicetree_overlay);
        render_single_value(&mut content, "grub_users", &self.devicetree_overlay);
        render_single_value(&mut content, "grub_arg", &self.devicetree_overlay);

        // Commands with multiple values
        render_multiple_values(&mut content, "initrd", &self.initrd);
        render_multiple_values(&mut content, "options", &self.options);
        render_multiple_values(&mut content, "grub_class", &self.grub_class);

        content
    }

    pub fn set(&mut self, key: &str, value: String, comment: Option<String>) -> Result<(), String> {
        fn value_generator(value: String, comment: Option<String>) -> BLSValue {
            match comment {
                Some(comment) => BLSValue::ValueWithComment(value, comment),
                None => BLSValue::Value(value),
            }
        }
        match key {
            "title" => self.title = Some(value_generator(value, comment)),
            "version" => self.version = Some(value_generator(value, comment)),
            "machine-id" => self.machine_id = Some(value_generator(value, comment)),
            "sort-key" => self.sort_key = Some(value_generator(value, comment)),
            "linux" => self.linux = value_generator(value, comment),
            "efi" => self.efi = Some(value_generator(value, comment)),
            "initrd" => self.initrd.push(value_generator(value, comment)),
            "options" => self.options.push(value_generator(value, comment)),
            "devicetree" => self.devicetree = Some(value_generator(value, comment)),
            "devicetree-overlay" => self.devicetree_overlay = Some(value_generator(value, comment)),
            "architecture" => self.architecture = Some(value_generator(value, comment)),
            "grub_hotkey" => self.grub_hotkey = Some(value_generator(value, comment)),
            "grub_users" => self.grub_users = Some(value_generator(value, comment)),
            "grub_class" => {
                self.grub_class.push(value_generator(value, comment));
            }
            "grub_arg" => {
                self.grub_arg = Some(value_generator(value, comment));
            }
            _ => return Err(format!("Invalid key {}", key)),
        }

        Ok(())
    }
}

#[cfg(test)]
mod bls_tests {
    use super::BLSEntry;
    use super::BLSValue;

    #[test]
    fn new_entry() {
        let entry = BLSEntry::new();
        match entry.linux {
            BLSValue::Value(linux) => {
                assert_eq!(linux, "");
            }
            _ => {
                panic!("Invalid 'linux' value {:?}", entry.linux);
            }
        }
        assert_eq!(entry.initrd.len(), 0);
    }

    #[test]
    fn parse_entry() {
        let entry_txt = "#Comment\n\
                     linux foobar-2.4\n\
                     options foo=bar #Another Comment";
        let entry = BLSEntry::parse(entry_txt);

        assert!(entry.is_ok());
        let entry = entry.unwrap();
        assert_eq!(entry.comments.len(), 1);
        assert_eq!(entry.comments[0], "Comment");

        if let BLSValue::Value(linux) = entry.linux {
            assert_eq!(linux, "foobar-2.4");
        }

        assert_eq!(entry.options.len(), 1);
        match &entry.options[0] {
            BLSValue::ValueWithComment(option, comment) => {
                assert_eq!(option, "foo=bar");
                assert_eq!(comment, "Another Comment");
            }
            _ => {
                panic!("Invalid 'options' value {:?}", entry.options[0])
            }
        }
    }

    #[test]
    fn parse_errors() {
        // Missing 'linux' command
        let entry_txt = "options foo=bar";
        let entry = BLSEntry::parse(entry_txt);
        assert!(entry.is_err());

        // Invalid command
        let entry_txt = "linux asdasdasdas\n\
                     invalid_command foo=bar";
        let entry = BLSEntry::parse(entry_txt);
        assert!(entry.is_err());
    }
}
