use boostvoronoi::builder as VB;
use boostvoronoi::diagram as VD;
use boostvoronoi::file_reader as FR;
use boostvoronoi::sync_diagram as VS;
use boostvoronoi::BvError;
use std::io::{BufReader, Cursor};

#[allow(dead_code)]
fn almost_equal(x1: F, x2: F, y1: F, y2: F) -> bool {
    let delta = 0.000001;
    assert!(F::abs(x1 - x2) < delta, "{} != {}", x1, x2);
    assert!(F::abs(y1 - y2) < delta, "{} != {}", y1, y2);

    (F::abs(x1 - x2) < delta) && (F::abs(y1 - y2) < delta)
}

type I = i32;
type F = f64;

//#[ignore]
#[test]
fn sample_primary_069() -> Result<(), BvError> {
    let output: VS::SyncDiagram<I, F> = {
        let input = r#"0
81
-50026 -49688 -49978 -49701
-49998 -49860 -49969 -49782
-49974 -49744 -49897 -49748
-49972 -49661 -49971 -49727
-49971 -49845 -49902 -49789
-49967 -49338 -49934 -49244
-49947 -49670 -49896 -49744
-49939 -49502 -49885 -49564
-49937 -49150 -49874 -49115
-49930 -49608 -49885 -49665
-49925 -48976 -49888 -49019
-49919 -49430 -49851 -49399
-49909 -49052 -49908 -49015
-49902 -49341 -49895 -49344
-49899 -49717 -49827 -49792
-49878 -49284 -49790 -49223
-49844 -49250 -49784 -49171
-49792 -49805 -49708 -49872
-49781 -49381 -49727 -49379
-49772 -49162 -49702 -49126
-49768 -49290 -49696 -49324
-49764 -49266 -49705 -49194
-49737 -49279 -49659 -49200
-49701 -49846 -49640 -49812
-49699 -49415 -49670 -49409
-49694 -49710 -49670 -49623
-49688 -49732 -49653 -49759
-49682 -49014 -49675 -49030
-49679 -49421 -49641 -49428
-49671 -49124 -49659 -49117
-49642 -49113 -49599 -49063
-49639 -49409 -49547 -49439
-49629 -49070 -49595 -49043
-49616 -49830 -49532 -49887
-49616 -49438 -49544 -49519
-49615 -49093 -49574 -49097
-49610 -49583 -49587 -49491
-49608 -49736 -49550 -49812
-49599 -49403 -49520 -49408
-49588 -49924 -49576 -49918
-49579 -50064 -49483 -49994
-49559 -49800 -49506 -49870
-49552 -49543 -49534 -49570
-49541 -49937 -49469 -49920
-49513 -49782 -49494 -49816
-49509 -49996 -49472 -49955
-49509 -49892 -49452 -49796
-49482 -50051 -49468 -49972
-49419 -49411 -49371 -49424
-49411 -49494 -49328 -49567
-49407 -49584 -49358 -49547
-49399 -49236 -49341 -49168
-49370 -49205 -49284 -49121
-49364 -49032 -49355 -49119
-49363 -49029 -49302 -49048
-49340 -49544 -49242 -49472
-49335 -49800 -49325 -49866
-49334 -49786 -49282 -49798
-49323 -49685 -49229 -49765
-49311 -49777 -49247 -49789
-49305 -49633 -49283 -49631
-49300 -49574 -49261 -49601
-49295 -49245 -49286 -49278
-49275 -49861 -49223 -49778
-49261 -49054 -49235 -49029
-49260 -49355 -49228 -49436
-49248 -49991 -49198 -49926
-49236 -49618 -49152 -49637
-49225 -50042 -49179 -49961
-49190 -49250 -49108 -49308
-49189 -49609 -49168 -49578
-49164 -49936 -49133 -49922
-49124 -49640 -49057 -49695
-49117 -49169 -49041 -49213
-49111 -49802 -49099 -49894
-49105 -49697 -49053 -49738
-49082 -49532 -49031 -49559
-49063 -49321 -49007 -49234
-49032 -49130 -48987 -49200
-49025 -49437 -48988 -49427
-49008 -49119 -48968 -49214
"#;
        let mut vb = VB::Builder::<I, F>::default();
        let br = BufReader::new(Cursor::new(input));
        let (points, segments) = FR::read_boost_input_buffer::<I, _>(br)?;
        vb.with_vertices(points.iter())?;
        vb.with_segments(segments.iter())?;
        vb.build()?.into()
    };
    assert_eq!(output.cells().len(), 243);
    let cell = output.cells()[0];
    assert_eq!(cell.id().0, 0);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[1];
    assert_eq!(cell.id().0, 1);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[2];
    assert_eq!(cell.id().0, 2);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[3];
    assert_eq!(cell.id().0, 3);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[4];
    assert_eq!(cell.id().0, 4);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[5];
    assert_eq!(cell.id().0, 5);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[6];
    assert_eq!(cell.id().0, 6);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[7];
    assert_eq!(cell.id().0, 7);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[8];
    assert_eq!(cell.id().0, 8);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[9];
    assert_eq!(cell.id().0, 9);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[10];
    assert_eq!(cell.id().0, 10);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[11];
    assert_eq!(cell.id().0, 11);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[12];
    assert_eq!(cell.id().0, 12);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[13];
    assert_eq!(cell.id().0, 13);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[14];
    assert_eq!(cell.id().0, 14);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[15];
    assert_eq!(cell.id().0, 15);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[16];
    assert_eq!(cell.id().0, 16);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[17];
    assert_eq!(cell.id().0, 17);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[18];
    assert_eq!(cell.id().0, 18);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[19];
    assert_eq!(cell.id().0, 19);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[20];
    assert_eq!(cell.id().0, 20);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[21];
    assert_eq!(cell.id().0, 21);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[22];
    assert_eq!(cell.id().0, 22);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[23];
    assert_eq!(cell.id().0, 23);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[24];
    assert_eq!(cell.id().0, 24);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[25];
    assert_eq!(cell.id().0, 25);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[26];
    assert_eq!(cell.id().0, 26);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[27];
    assert_eq!(cell.id().0, 27);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[28];
    assert_eq!(cell.id().0, 28);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[29];
    assert_eq!(cell.id().0, 29);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[30];
    assert_eq!(cell.id().0, 30);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[31];
    assert_eq!(cell.id().0, 31);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[32];
    assert_eq!(cell.id().0, 32);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[33];
    assert_eq!(cell.id().0, 33);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[34];
    assert_eq!(cell.id().0, 34);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[35];
    assert_eq!(cell.id().0, 35);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[36];
    assert_eq!(cell.id().0, 36);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[37];
    assert_eq!(cell.id().0, 37);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[38];
    assert_eq!(cell.id().0, 38);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[39];
    assert_eq!(cell.id().0, 39);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[40];
    assert_eq!(cell.id().0, 40);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[41];
    assert_eq!(cell.id().0, 41);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[42];
    assert_eq!(cell.id().0, 42);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[43];
    assert_eq!(cell.id().0, 43);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[44];
    assert_eq!(cell.id().0, 44);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[45];
    assert_eq!(cell.id().0, 45);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[46];
    assert_eq!(cell.id().0, 46);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[47];
    assert_eq!(cell.id().0, 47);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[48];
    assert_eq!(cell.id().0, 48);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[49];
    assert_eq!(cell.id().0, 49);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[50];
    assert_eq!(cell.id().0, 50);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[51];
    assert_eq!(cell.id().0, 51);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[52];
    assert_eq!(cell.id().0, 52);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[53];
    assert_eq!(cell.id().0, 53);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[54];
    assert_eq!(cell.id().0, 54);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[55];
    assert_eq!(cell.id().0, 55);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[56];
    assert_eq!(cell.id().0, 56);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[57];
    assert_eq!(cell.id().0, 57);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[58];
    assert_eq!(cell.id().0, 58);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[59];
    assert_eq!(cell.id().0, 59);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[60];
    assert_eq!(cell.id().0, 60);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[61];
    assert_eq!(cell.id().0, 61);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[62];
    assert_eq!(cell.id().0, 62);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[63];
    assert_eq!(cell.id().0, 63);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[64];
    assert_eq!(cell.id().0, 64);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[65];
    assert_eq!(cell.id().0, 65);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[66];
    assert_eq!(cell.id().0, 66);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[67];
    assert_eq!(cell.id().0, 67);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[68];
    assert_eq!(cell.id().0, 68);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[69];
    assert_eq!(cell.id().0, 69);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[70];
    assert_eq!(cell.id().0, 70);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[71];
    assert_eq!(cell.id().0, 71);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[72];
    assert_eq!(cell.id().0, 72);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[73];
    assert_eq!(cell.id().0, 73);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[74];
    assert_eq!(cell.id().0, 74);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[75];
    assert_eq!(cell.id().0, 75);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[76];
    assert_eq!(cell.id().0, 76);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[77];
    assert_eq!(cell.id().0, 77);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[78];
    assert_eq!(cell.id().0, 78);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[79];
    assert_eq!(cell.id().0, 79);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[80];
    assert_eq!(cell.id().0, 80);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[81];
    assert_eq!(cell.id().0, 81);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[82];
    assert_eq!(cell.id().0, 82);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[83];
    assert_eq!(cell.id().0, 83);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[84];
    assert_eq!(cell.id().0, 84);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[85];
    assert_eq!(cell.id().0, 85);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[86];
    assert_eq!(cell.id().0, 86);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[87];
    assert_eq!(cell.id().0, 87);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[88];
    assert_eq!(cell.id().0, 88);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[89];
    assert_eq!(cell.id().0, 89);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[90];
    assert_eq!(cell.id().0, 90);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[91];
    assert_eq!(cell.id().0, 91);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[92];
    assert_eq!(cell.id().0, 92);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[93];
    assert_eq!(cell.id().0, 93);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[94];
    assert_eq!(cell.id().0, 94);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[95];
    assert_eq!(cell.id().0, 95);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[96];
    assert_eq!(cell.id().0, 96);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[97];
    assert_eq!(cell.id().0, 97);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[98];
    assert_eq!(cell.id().0, 98);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[99];
    assert_eq!(cell.id().0, 99);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[100];
    assert_eq!(cell.id().0, 100);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[101];
    assert_eq!(cell.id().0, 101);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[102];
    assert_eq!(cell.id().0, 102);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[103];
    assert_eq!(cell.id().0, 103);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[104];
    assert_eq!(cell.id().0, 104);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[105];
    assert_eq!(cell.id().0, 105);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[106];
    assert_eq!(cell.id().0, 106);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[107];
    assert_eq!(cell.id().0, 107);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[108];
    assert_eq!(cell.id().0, 108);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[109];
    assert_eq!(cell.id().0, 109);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[110];
    assert_eq!(cell.id().0, 110);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[111];
    assert_eq!(cell.id().0, 111);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[112];
    assert_eq!(cell.id().0, 112);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[113];
    assert_eq!(cell.id().0, 113);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[114];
    assert_eq!(cell.id().0, 114);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[115];
    assert_eq!(cell.id().0, 115);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[116];
    assert_eq!(cell.id().0, 116);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[117];
    assert_eq!(cell.id().0, 117);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[118];
    assert_eq!(cell.id().0, 118);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[119];
    assert_eq!(cell.id().0, 119);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[120];
    assert_eq!(cell.id().0, 120);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[121];
    assert_eq!(cell.id().0, 121);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[122];
    assert_eq!(cell.id().0, 122);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[123];
    assert_eq!(cell.id().0, 123);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[124];
    assert_eq!(cell.id().0, 124);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[125];
    assert_eq!(cell.id().0, 125);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[126];
    assert_eq!(cell.id().0, 126);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[127];
    assert_eq!(cell.id().0, 127);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[128];
    assert_eq!(cell.id().0, 128);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[129];
    assert_eq!(cell.id().0, 129);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[130];
    assert_eq!(cell.id().0, 130);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[131];
    assert_eq!(cell.id().0, 131);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[132];
    assert_eq!(cell.id().0, 132);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[133];
    assert_eq!(cell.id().0, 133);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[134];
    assert_eq!(cell.id().0, 134);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[135];
    assert_eq!(cell.id().0, 135);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[136];
    assert_eq!(cell.id().0, 136);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[137];
    assert_eq!(cell.id().0, 137);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[138];
    assert_eq!(cell.id().0, 138);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[139];
    assert_eq!(cell.id().0, 139);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[140];
    assert_eq!(cell.id().0, 140);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[141];
    assert_eq!(cell.id().0, 141);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[142];
    assert_eq!(cell.id().0, 142);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[143];
    assert_eq!(cell.id().0, 143);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[144];
    assert_eq!(cell.id().0, 144);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[145];
    assert_eq!(cell.id().0, 145);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[146];
    assert_eq!(cell.id().0, 146);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[147];
    assert_eq!(cell.id().0, 147);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[148];
    assert_eq!(cell.id().0, 148);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[149];
    assert_eq!(cell.id().0, 149);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[150];
    assert_eq!(cell.id().0, 150);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[151];
    assert_eq!(cell.id().0, 151);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[152];
    assert_eq!(cell.id().0, 152);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[153];
    assert_eq!(cell.id().0, 153);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[154];
    assert_eq!(cell.id().0, 154);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[155];
    assert_eq!(cell.id().0, 155);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[156];
    assert_eq!(cell.id().0, 156);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[157];
    assert_eq!(cell.id().0, 157);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[158];
    assert_eq!(cell.id().0, 158);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[159];
    assert_eq!(cell.id().0, 159);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[160];
    assert_eq!(cell.id().0, 160);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[161];
    assert_eq!(cell.id().0, 161);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[162];
    assert_eq!(cell.id().0, 162);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[163];
    assert_eq!(cell.id().0, 163);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[164];
    assert_eq!(cell.id().0, 164);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[165];
    assert_eq!(cell.id().0, 165);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[166];
    assert_eq!(cell.id().0, 166);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[167];
    assert_eq!(cell.id().0, 167);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[168];
    assert_eq!(cell.id().0, 168);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[169];
    assert_eq!(cell.id().0, 169);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[170];
    assert_eq!(cell.id().0, 170);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[171];
    assert_eq!(cell.id().0, 171);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[172];
    assert_eq!(cell.id().0, 172);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[173];
    assert_eq!(cell.id().0, 173);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[174];
    assert_eq!(cell.id().0, 174);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[175];
    assert_eq!(cell.id().0, 175);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[176];
    assert_eq!(cell.id().0, 176);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[177];
    assert_eq!(cell.id().0, 177);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[178];
    assert_eq!(cell.id().0, 178);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[179];
    assert_eq!(cell.id().0, 179);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[180];
    assert_eq!(cell.id().0, 180);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[181];
    assert_eq!(cell.id().0, 181);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[182];
    assert_eq!(cell.id().0, 182);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[183];
    assert_eq!(cell.id().0, 183);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[184];
    assert_eq!(cell.id().0, 184);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[185];
    assert_eq!(cell.id().0, 185);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[186];
    assert_eq!(cell.id().0, 186);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[187];
    assert_eq!(cell.id().0, 187);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[188];
    assert_eq!(cell.id().0, 188);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[189];
    assert_eq!(cell.id().0, 189);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[190];
    assert_eq!(cell.id().0, 190);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[191];
    assert_eq!(cell.id().0, 191);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[192];
    assert_eq!(cell.id().0, 192);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[193];
    assert_eq!(cell.id().0, 193);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[194];
    assert_eq!(cell.id().0, 194);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[195];
    assert_eq!(cell.id().0, 195);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[196];
    assert_eq!(cell.id().0, 196);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[197];
    assert_eq!(cell.id().0, 197);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[198];
    assert_eq!(cell.id().0, 198);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[199];
    assert_eq!(cell.id().0, 199);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[200];
    assert_eq!(cell.id().0, 200);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[201];
    assert_eq!(cell.id().0, 201);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[202];
    assert_eq!(cell.id().0, 202);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[203];
    assert_eq!(cell.id().0, 203);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[204];
    assert_eq!(cell.id().0, 204);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[205];
    assert_eq!(cell.id().0, 205);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[206];
    assert_eq!(cell.id().0, 206);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[207];
    assert_eq!(cell.id().0, 207);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[208];
    assert_eq!(cell.id().0, 208);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[209];
    assert_eq!(cell.id().0, 209);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[210];
    assert_eq!(cell.id().0, 210);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[211];
    assert_eq!(cell.id().0, 211);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[212];
    assert_eq!(cell.id().0, 212);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[213];
    assert_eq!(cell.id().0, 213);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[214];
    assert_eq!(cell.id().0, 214);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[215];
    assert_eq!(cell.id().0, 215);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[216];
    assert_eq!(cell.id().0, 216);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[217];
    assert_eq!(cell.id().0, 217);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[218];
    assert_eq!(cell.id().0, 218);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[219];
    assert_eq!(cell.id().0, 219);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[220];
    assert_eq!(cell.id().0, 220);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[221];
    assert_eq!(cell.id().0, 221);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[222];
    assert_eq!(cell.id().0, 222);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[223];
    assert_eq!(cell.id().0, 223);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[224];
    assert_eq!(cell.id().0, 224);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[225];
    assert_eq!(cell.id().0, 225);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[226];
    assert_eq!(cell.id().0, 226);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[227];
    assert_eq!(cell.id().0, 227);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[228];
    assert_eq!(cell.id().0, 228);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[229];
    assert_eq!(cell.id().0, 229);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[230];
    assert_eq!(cell.id().0, 230);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[231];
    assert_eq!(cell.id().0, 231);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[232];
    assert_eq!(cell.id().0, 232);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[233];
    assert_eq!(cell.id().0, 233);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[234];
    assert_eq!(cell.id().0, 234);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[235];
    assert_eq!(cell.id().0, 235);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[236];
    assert_eq!(cell.id().0, 236);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[237];
    assert_eq!(cell.id().0, 237);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[238];
    assert_eq!(cell.id().0, 238);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[239];
    assert_eq!(cell.id().0, 239);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[240];
    assert_eq!(cell.id().0, 240);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[241];
    assert_eq!(cell.id().0, 241);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[242];
    assert_eq!(cell.id().0, 242);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    assert_eq!(output.vertices().len(), 472);
    assert_eq!(output.edges().len(), 1428);
    let v = output.vertices()[0];
    assert!(almost_equal(v.x(), -50163.1355270, v.y(), -49798.6034579));
    assert_eq!(v.get_incident_edge().unwrap().0, 7);
    let v = output.vertices()[1];
    assert!(almost_equal(v.x(), -50044.8667992, v.y(), -49757.6620278));
    assert_eq!(v.get_incident_edge().unwrap().0, 15);
    let v = output.vertices()[2];
    assert!(almost_equal(v.x(), -50003.4832846, v.y(), -49661.4770195));
    assert_eq!(v.get_incident_edge().unwrap().0, 21);
    let v = output.vertices()[3];
    assert!(almost_equal(v.x(), -49976.6467736, v.y(), -49696.0034718));
    assert_eq!(v.get_incident_edge().unwrap().0, 23);
    let v = output.vertices()[4];
    assert!(almost_equal(v.x(), -49980.3850116, v.y(), -49709.8061967));
    assert_eq!(v.get_incident_edge().unwrap().0, 25);
    let v = output.vertices()[5];
    assert!(almost_equal(v.x(), -50003.1320143, v.y(), -49727.4868487));
    assert_eq!(v.get_incident_edge().unwrap().0, 33);
    let v = output.vertices()[6];
    assert!(almost_equal(v.x(), -50005.7963973, v.y(), -49729.6241652));
    assert_eq!(v.get_incident_edge().unwrap().0, 35);
    let v = output.vertices()[7];
    assert!(almost_equal(v.x(), -50057.6140469, v.y(), -49769.4987579));
    assert_eq!(v.get_incident_edge().unwrap().0, 37);
    let v = output.vertices()[8];
    assert!(almost_equal(v.x(), -50014.1418919, v.y(), -49644.2162162));
    assert_eq!(v.get_incident_edge().unwrap().0, 41);
    let v = output.vertices()[9];
    assert!(almost_equal(v.x(), -49977.7985817, v.y(), -49836.6231761));
    assert_eq!(v.get_incident_edge().unwrap().0, 43);
    let v = output.vertices()[10];
    assert!(almost_equal(v.x(), -50007.3987936, v.y(), -49767.7235255));
    assert_eq!(v.get_incident_edge().unwrap().0, 49);
    let v = output.vertices()[11];
    assert!(almost_equal(v.x(), -49973.5488266, v.y(), -49735.3149129));
    assert_eq!(v.get_incident_edge().unwrap().0, 51);
    let v = output.vertices()[12];
    assert!(almost_equal(v.x(), -49975.0110117, v.y(), -49763.4619752));
    assert_eq!(v.get_incident_edge().unwrap().0, 53);
    let v = output.vertices()[13];
    assert!(almost_equal(v.x(), -49975.7342908, v.y(), -49868.2782765));
    assert_eq!(v.get_incident_edge().unwrap().0, 55);
    let v = output.vertices()[14];
    assert!(almost_equal(v.x(), -49958.1743169, v.y(), -49677.7012184));
    assert_eq!(v.get_incident_edge().unwrap().0, 61);
    let v = output.vertices()[15];
    assert!(almost_equal(v.x(), -49957.8025594, v.y(), -49660.7848873));
    assert_eq!(v.get_incident_edge().unwrap().0, 63);
    let v = output.vertices()[16];
    assert!(almost_equal(v.x(), -50248.7027110, v.y(), -49470.4858287));
    assert_eq!(v.get_incident_edge().unwrap().0, 69);
    let v = output.vertices()[17];
    assert!(almost_equal(v.x(), -50470.3888889, v.y(), -49161.2783688));
    assert_eq!(v.get_incident_edge().unwrap().0, 75);
    let v = output.vertices()[18];
    assert!(almost_equal(v.x(), -49952.6390094, v.y(), -49726.7218032));
    assert_eq!(v.get_incident_edge().unwrap().0, 77);
    let v = output.vertices()[19];
    assert!(almost_equal(v.x(), -50056.8472222, v.y(), -49200.8727837));
    assert_eq!(v.get_incident_edge().unwrap().0, 85);
    let v = output.vertices()[20];
    assert!(almost_equal(v.x(), -50055.9210526, v.y(), -49560.6578947));
    assert_eq!(v.get_incident_edge().unwrap().0, 87);
    let v = output.vertices()[21];
    assert!(almost_equal(v.x(), -51965.5789474, v.y(), -48922.6842105));
    assert_eq!(v.get_incident_edge().unwrap().0, 93);
    let v = output.vertices()[22];
    assert!(almost_equal(v.x(), -49957.2715003, v.y(), -49629.5301318));
    assert_eq!(v.get_incident_edge().unwrap().0, 95);
    let v = output.vertices()[23];
    assert!(almost_equal(v.x(), -50040.0161290, v.y(), -49563.9589166));
    assert_eq!(v.get_incident_edge().unwrap().0, 97);
    let v = output.vertices()[24];
    assert!(almost_equal(v.x(), -49945.0005184, v.y(), -49790.9228842));
    assert_eq!(v.get_incident_edge().unwrap().0, 99);
    let v = output.vertices()[25];
    assert!(almost_equal(v.x(), -50040.6978852, v.y(), -49434.9728097));
    assert_eq!(v.get_incident_edge().unwrap().0, 105);
    let v = output.vertices()[26];
    assert!(almost_equal(v.x(), -49944.8951539, v.y(), -49719.2574381));
    assert_eq!(v.get_incident_edge().unwrap().0, 107);
    let v = output.vertices()[27];
    assert!(almost_equal(v.x(), -50006.9107627, v.y(), -49561.1480836));
    assert_eq!(v.get_incident_edge().unwrap().0, 109);
    let v = output.vertices()[28];
    assert!(almost_equal(v.x(), -49949.2409805, v.y(), -49637.0027236));
    assert_eq!(v.get_incident_edge().unwrap().0, 111);
    let v = output.vertices()[29];
    assert!(almost_equal(v.x(), -50095.7727273, v.y(), -49051.6363636));
    assert_eq!(v.get_incident_edge().unwrap().0, 117);
    let v = output.vertices()[30];
    assert!(almost_equal(v.x(), -50076.0538922, v.y(), -49047.4850299));
    assert_eq!(v.get_incident_edge().unwrap().0, 119);
    let v = output.vertices()[31];
    assert!(almost_equal(v.x(), -49977.6388197, v.y(), -49021.2938681));
    assert_eq!(v.get_incident_edge().unwrap().0, 121);
    let v = output.vertices()[32];
    assert!(almost_equal(v.x(), -49956.2772966, v.y(), -49013.6952082));
    assert_eq!(v.get_incident_edge().unwrap().0, 125);
    let v = output.vertices()[33];
    assert!(almost_equal(v.x(), -49937.8016296, v.y(), -49777.4511885));
    assert_eq!(v.get_incident_edge().unwrap().0, 127);
    let v = output.vertices()[34];
    assert!(almost_equal(v.x(), -49934.9500725, v.y(), -49349.2515703));
    assert_eq!(v.get_incident_edge().unwrap().0, 135);
    let v = output.vertices()[35];
    assert!(almost_equal(v.x(), -49972.0000000, v.y(), -49087.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 137);
    let v = output.vertices()[36];
    assert!(almost_equal(v.x(), -49925.6677965, v.y(), -49655.2980760));
    assert_eq!(v.get_incident_edge().unwrap().0, 139);
    let v = output.vertices()[37];
    assert!(almost_equal(v.x(), -49903.6658682, v.y(), -49721.4792335));
    assert_eq!(v.get_incident_edge().unwrap().0, 145);
    let v = output.vertices()[38];
    assert!(almost_equal(v.x(), -56882.5844097, v.y(), -48352.2029138));
    assert_eq!(v.get_incident_edge().unwrap().0, 147);
    let v = output.vertices()[39];
    assert!(almost_equal(v.x(), -49898.0702112, v.y(), -49745.4267672));
    assert_eq!(v.get_incident_edge().unwrap().0, 153);
    let v = output.vertices()[40];
    assert!(almost_equal(v.x(), -49896.8910256, v.y(), -49745.9022436));
    assert_eq!(v.get_incident_edge().unwrap().0, 157);
    let v = output.vertices()[41];
    assert!(almost_equal(v.x(), -49900.7897371, v.y(), -49015.1948720));
    assert_eq!(v.get_incident_edge().unwrap().0, 159);
    let v = output.vertices()[42];
    assert!(almost_equal(v.x(), -49916.8994585, v.y(), -49770.6417386));
    assert_eq!(v.get_incident_edge().unwrap().0, 161);
    let v = output.vertices()[43];
    assert!(almost_equal(v.x(), -49940.5526860, v.y(), -49382.7231405));
    assert_eq!(v.get_incident_edge().unwrap().0, 163);
    let v = output.vertices()[44];
    assert!(almost_equal(v.x(), -49896.7561012, v.y(), -49026.5343196));
    assert_eq!(v.get_incident_edge().unwrap().0, 167);
    let v = output.vertices()[45];
    assert!(almost_equal(v.x(), -49936.3890154, v.y(), -49380.4286672));
    assert_eq!(v.get_incident_edge().unwrap().0, 173);
    let v = output.vertices()[46];
    assert!(almost_equal(v.x(), -49889.1483241, v.y(), -49739.2778990));
    assert_eq!(v.get_incident_edge().unwrap().0, 175);
    let v = output.vertices()[47];
    assert!(almost_equal(v.x(), -49906.1700068, v.y(), -49681.7131633));
    assert_eq!(v.get_incident_edge().unwrap().0, 177);
    let v = output.vertices()[48];
    assert!(almost_equal(v.x(), -49898.0739692, v.y(), -49768.6739062));
    assert_eq!(v.get_incident_edge().unwrap().0, 183);
    let v = output.vertices()[49];
    assert!(almost_equal(v.x(), -49908.7690670, v.y(), -49584.7020906));
    assert_eq!(v.get_incident_edge().unwrap().0, 185);
    let v = output.vertices()[50];
    assert!(almost_equal(v.x(), -49919.4079991, v.y(), -49381.6186646));
    assert_eq!(v.get_incident_edge().unwrap().0, 187);
    let v = output.vertices()[51];
    assert!(almost_equal(v.x(), -49905.1620462, v.y(), -49588.3910891));
    assert_eq!(v.get_incident_edge().unwrap().0, 191);
    let v = output.vertices()[52];
    assert!(almost_equal(v.x(), -49912.7501913, v.y(), -49302.9209721));
    assert_eq!(v.get_incident_edge().unwrap().0, 193);
    let v = output.vertices()[53];
    assert!(almost_equal(v.x(), -49885.0363419, v.y(), -49748.8659145));
    assert_eq!(v.get_incident_edge().unwrap().0, 195);
    let v = output.vertices()[54];
    assert!(almost_equal(v.x(), -49899.8283703, v.y(), -49688.8923618));
    assert_eq!(v.get_incident_edge().unwrap().0, 197);
    let v = output.vertices()[55];
    assert!(almost_equal(v.x(), -49911.0815253, v.y(), -49381.5235591));
    assert_eq!(v.get_incident_edge().unwrap().0, 199);
    let v = output.vertices()[56];
    assert!(almost_equal(v.x(), -49900.1919806, v.y(), -49255.8687728));
    assert_eq!(v.get_incident_edge().unwrap().0, 201);
    let v = output.vertices()[57];
    assert!(almost_equal(v.x(), -49898.8258362, v.y(), -49253.9561707));
    assert_eq!(v.get_incident_edge().unwrap().0, 203);
    let v = output.vertices()[58];
    assert!(almost_equal(v.x(), -49905.2433697, v.y(), -49472.5990640));
    assert_eq!(v.get_incident_edge().unwrap().0, 205);
    let v = output.vertices()[59];
    assert!(almost_equal(v.x(), -49889.8184713, v.y(), -49312.5764331));
    assert_eq!(v.get_incident_edge().unwrap().0, 207);
    let v = output.vertices()[60];
    assert!(almost_equal(v.x(), -49911.3176539, v.y(), -49196.2282230));
    assert_eq!(v.get_incident_edge().unwrap().0, 209);
    let v = output.vertices()[61];
    assert!(almost_equal(v.x(), -49883.7890861, v.y(), -49770.4159651));
    assert_eq!(v.get_incident_edge().unwrap().0, 211);
    let v = output.vertices()[62];
    assert!(almost_equal(v.x(), -49891.5000000, v.y(), -49083.5000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 213);
    let v = output.vertices()[63];
    assert!(almost_equal(v.x(), -49882.6146497, v.y(), -49315.1008493));
    assert_eq!(v.get_incident_edge().unwrap().0, 215);
    let v = output.vertices()[64];
    assert!(almost_equal(v.x(), -49898.8633911, v.y(), -49474.1706260));
    assert_eq!(v.get_incident_edge().unwrap().0, 219);
    let v = output.vertices()[65];
    assert!(almost_equal(v.x(), -49876.3153942, v.y(), -49695.2227785));
    assert_eq!(v.get_incident_edge().unwrap().0, 221);
    let v = output.vertices()[66];
    assert!(almost_equal(v.x(), -49888.1229640, v.y(), -49233.8444598));
    assert_eq!(v.get_incident_edge().unwrap().0, 227);
    let v = output.vertices()[67];
    assert!(almost_equal(v.x(), -49840.5000738, v.y(), -49252.6581718));
    assert_eq!(v.get_incident_edge().unwrap().0, 229);
    let v = output.vertices()[68];
    assert!(almost_equal(v.x(), -49887.0216867, v.y(), -49217.3253012));
    assert_eq!(v.get_incident_edge().unwrap().0, 231);
    let v = output.vertices()[69];
    assert!(almost_equal(v.x(), -49870.9556452, v.y(), -49053.0282258));
    assert_eq!(v.get_incident_edge().unwrap().0, 233);
    let v = output.vertices()[70];
    assert!(almost_equal(v.x(), -49866.0657328, v.y(), -49365.9525862));
    assert_eq!(v.get_incident_edge().unwrap().0, 235);
    let v = output.vertices()[71];
    assert!(almost_equal(v.x(), -49887.0665032, v.y(), -49197.6875723));
    assert_eq!(v.get_incident_edge().unwrap().0, 239);
    let v = output.vertices()[72];
    assert!(almost_equal(v.x(), -49866.0599002, v.y(), -49829.4975042));
    assert_eq!(v.get_incident_edge().unwrap().0, 241);
    let v = output.vertices()[73];
    assert!(almost_equal(v.x(), -49866.2042206, v.y(), -49833.1055140));
    assert_eq!(v.get_incident_edge().unwrap().0, 243);
    let v = output.vertices()[74];
    assert!(almost_equal(v.x(), -49850.0446618, v.y(), -49324.3290125));
    assert_eq!(v.get_incident_edge().unwrap().0, 245);
    let v = output.vertices()[75];
    assert!(almost_equal(v.x(), -49854.9661017, v.y(), -49063.2033898));
    assert_eq!(v.get_incident_edge().unwrap().0, 247);
    let v = output.vertices()[76];
    assert!(almost_equal(v.x(), -49847.2409187, v.y(), -49163.1663463));
    assert_eq!(v.get_incident_edge().unwrap().0, 249);
    let v = output.vertices()[77];
    assert!(almost_equal(v.x(), -49797.7894857, v.y(), -49211.7627091));
    assert_eq!(v.get_incident_edge().unwrap().0, 257);
    let v = output.vertices()[78];
    assert!(almost_equal(v.x(), -49826.3898758, v.y(), -49138.8051576));
    assert_eq!(v.get_incident_edge().unwrap().0, 265);
    let v = output.vertices()[79];
    assert!(almost_equal(v.x(), -49829.2697526, v.y(), -49851.7262570));
    assert_eq!(v.get_incident_edge().unwrap().0, 271);
    let v = output.vertices()[80];
    assert!(almost_equal(v.x(), -49829.6706161, v.y(), -49336.8364929));
    assert_eq!(v.get_incident_edge().unwrap().0, 281);
    let v = output.vertices()[81];
    assert!(almost_equal(v.x(), -49798.9735740, v.y(), -49272.5044043));
    assert_eq!(v.get_incident_edge().unwrap().0, 283);
    let v = output.vertices()[82];
    assert!(almost_equal(v.x(), -49828.1268035, v.y(), -49332.0798809));
    assert_eq!(v.get_incident_edge().unwrap().0, 285);
    let v = output.vertices()[83];
    assert!(almost_equal(v.x(), -49783.7758114, v.y(), -49249.7948212));
    assert_eq!(v.get_incident_edge().unwrap().0, 287);
    let v = output.vertices()[84];
    assert!(almost_equal(v.x(), -49823.5523356, v.y(), -49328.4925235));
    assert_eq!(v.get_incident_edge().unwrap().0, 289);
    let v = output.vertices()[85];
    assert!(almost_equal(v.x(), -49813.8663968, v.y(), -49118.6781377));
    assert_eq!(v.get_incident_edge().unwrap().0, 291);
    let v = output.vertices()[86];
    assert!(almost_equal(v.x(), -49796.0497336, v.y(), -49762.2877442));
    assert_eq!(v.get_incident_edge().unwrap().0, 293);
    let v = output.vertices()[87];
    assert!(almost_equal(v.x(), -49776.2438322, v.y(), -49242.8449633));
    assert_eq!(v.get_incident_edge().unwrap().0, 295);
    let v = output.vertices()[88];
    assert!(almost_equal(v.x(), -49762.6008584, v.y(), -49278.5665236));
    assert_eq!(v.get_incident_edge().unwrap().0, 297);
    let v = output.vertices()[89];
    assert!(almost_equal(v.x(), -49788.5390335, v.y(), -49333.4944238));
    assert_eq!(v.get_incident_edge().unwrap().0, 299);
    let v = output.vertices()[90];
    assert!(almost_equal(v.x(), -49821.0333333, v.y(), -49614.5000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 301);
    let v = output.vertices()[91];
    assert!(almost_equal(v.x(), -49753.4631313, v.y(), -49278.6541958));
    assert_eq!(v.get_incident_edge().unwrap().0, 307);
    let v = output.vertices()[92];
    assert!(almost_equal(v.x(), -49752.0752490, v.y(), -49275.7716710));
    assert_eq!(v.get_incident_edge().unwrap().0, 309);
    let v = output.vertices()[93];
    assert!(almost_equal(v.x(), -49782.7220980, v.y(), -49334.5033537));
    assert_eq!(v.get_incident_edge().unwrap().0, 311);
    let v = output.vertices()[94];
    assert!(almost_equal(v.x(), -49762.2169118, v.y(), -49187.5441176));
    assert_eq!(v.get_incident_edge().unwrap().0, 313);
    let v = output.vertices()[95];
    assert!(almost_equal(v.x(), -49804.6104568, v.y(), -49098.5907785));
    assert_eq!(v.get_incident_edge().unwrap().0, 315);
    let v = output.vertices()[96];
    assert!(almost_equal(v.x(), -49747.3843437, v.y(), -49268.7471037));
    assert_eq!(v.get_incident_edge().unwrap().0, 317);
    let v = output.vertices()[97];
    assert!(almost_equal(v.x(), -49856.7365218, v.y(), -49914.0469592));
    assert_eq!(v.get_incident_edge().unwrap().0, 319);
    let v = output.vertices()[98];
    assert!(almost_equal(v.x(), -49759.4285144, v.y(), -49200.9596617));
    assert_eq!(v.get_incident_edge().unwrap().0, 323);
    let v = output.vertices()[99];
    assert!(almost_equal(v.x(), -49755.5413014, v.y(), -49196.0331036));
    assert_eq!(v.get_incident_edge().unwrap().0, 325);
    let v = output.vertices()[100];
    assert!(almost_equal(v.x(), -49754.8010537, v.y(), -49195.4423956));
    assert_eq!(v.get_incident_edge().unwrap().0, 327);
    let v = output.vertices()[101];
    assert!(almost_equal(v.x(), -49724.6093820, v.y(), -49291.2337748));
    assert_eq!(v.get_incident_edge().unwrap().0, 331);
    let v = output.vertices()[102];
    assert!(almost_equal(v.x(), -49807.7345637, v.y(), -49493.9048280));
    assert_eq!(v.get_incident_edge().unwrap().0, 335);
    let v = output.vertices()[103];
    assert!(almost_equal(v.x(), -49805.1478490, v.y(), -49494.4513523));
    assert_eq!(v.get_incident_edge().unwrap().0, 343);
    let v = output.vertices()[104];
    assert!(almost_equal(v.x(), -49728.8407325, v.y(), -49174.4638442));
    assert_eq!(v.get_incident_edge().unwrap().0, 349);
    let v = output.vertices()[105];
    assert!(almost_equal(v.x(), -49725.9600000, v.y(), -49407.0800000));
    assert_eq!(v.get_incident_edge().unwrap().0, 351);
    let v = output.vertices()[106];
    assert!(almost_equal(v.x(), -49728.2345724, v.y(), -49345.6665450));
    assert_eq!(v.get_incident_edge().unwrap().0, 355);
    let v = output.vertices()[107];
    assert!(almost_equal(v.x(), -49777.3855523, v.y(), -49722.0285882));
    assert_eq!(v.get_incident_edge().unwrap().0, 361);
    let v = output.vertices()[108];
    assert!(almost_equal(v.x(), -49789.8536798, v.y(), -49689.0011741));
    assert_eq!(v.get_incident_edge().unwrap().0, 363);
    let v = output.vertices()[109];
    assert!(almost_equal(v.x(), -49788.6494321, v.y(), -49683.8898118));
    assert_eq!(v.get_incident_edge().unwrap().0, 365);
    let v = output.vertices()[110];
    assert!(almost_equal(v.x(), -49760.1111519, v.y(), -49739.8484960));
    assert_eq!(v.get_incident_edge().unwrap().0, 371);
    let v = output.vertices()[111];
    assert!(almost_equal(v.x(), -49789.5544202, v.y(), -49053.6641863));
    assert_eq!(v.get_incident_edge().unwrap().0, 373);
    let v = output.vertices()[112];
    assert!(almost_equal(v.x(), -49750.7131679, v.y(), -49753.2374046));
    assert_eq!(v.get_incident_edge().unwrap().0, 375);
    let v = output.vertices()[113];
    assert!(almost_equal(v.x(), -49698.8452695, v.y(), -49860.5224274));
    assert_eq!(v.get_incident_edge().unwrap().0, 377);
    let v = output.vertices()[114];
    assert!(almost_equal(v.x(), -49720.3663628, v.y(), -49811.2544667));
    assert_eq!(v.get_incident_edge().unwrap().0, 383);
    let v = output.vertices()[115];
    assert!(almost_equal(v.x(), -49704.1485149, v.y(), -49390.1155116));
    assert_eq!(v.get_incident_edge().unwrap().0, 389);
    let v = output.vertices()[116];
    assert!(almost_equal(v.x(), -49709.5145592, v.y(), -49352.6190666));
    assert_eq!(v.get_incident_edge().unwrap().0, 391);
    let v = output.vertices()[117];
    assert!(almost_equal(v.x(), -49784.2510614, v.y(), -49047.3562721));
    assert_eq!(v.get_incident_edge().unwrap().0, 393);
    let v = output.vertices()[118];
    assert!(almost_equal(v.x(), -49856.4306187, v.y(), -49986.1658448));
    assert_eq!(v.get_incident_edge().unwrap().0, 397);
    let v = output.vertices()[119];
    assert!(almost_equal(v.x(), -49770.3996632, v.y(), -49052.6748526));
    assert_eq!(v.get_incident_edge().unwrap().0, 399);
    let v = output.vertices()[120];
    assert!(almost_equal(v.x(), -49691.8568249, v.y(), -49862.4039318));
    assert_eq!(v.get_incident_edge().unwrap().0, 401);
    let v = output.vertices()[121];
    assert!(almost_equal(v.x(), -49678.0663589, v.y(), -49415.9316626));
    assert_eq!(v.get_incident_edge().unwrap().0, 403);
    let v = output.vertices()[122];
    assert!(almost_equal(v.x(), -49694.5510204, v.y(), -49436.5034014));
    assert_eq!(v.get_incident_edge().unwrap().0, 405);
    let v = output.vertices()[123];
    assert!(almost_equal(v.x(), -49690.6797501, v.y(), -49205.7346492));
    assert_eq!(v.get_incident_edge().unwrap().0, 407);
    let v = output.vertices()[124];
    assert!(almost_equal(v.x(), -49784.4121937, v.y(), -49614.5000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 409);
    let v = output.vertices()[125];
    assert!(almost_equal(v.x(), -49788.2454745, v.y(), -49497.9342659));
    assert_eq!(v.get_incident_edge().unwrap().0, 411);
    let v = output.vertices()[126];
    assert!(almost_equal(v.x(), -49777.0600000, v.y(), -49487.3800000));
    assert_eq!(v.get_incident_edge().unwrap().0, 417);
    let v = output.vertices()[127];
    assert!(almost_equal(v.x(), -49746.3653461, v.y(), -49061.2223389));
    assert_eq!(v.get_incident_edge().unwrap().0, 423);
    let v = output.vertices()[128];
    assert!(almost_equal(v.x(), -49777.4604145, v.y(), -49593.3557477));
    assert_eq!(v.get_incident_edge().unwrap().0, 425);
    let v = output.vertices()[129];
    assert!(almost_equal(v.x(), -49733.1430063, v.y(), -49065.4441545));
    assert_eq!(v.get_incident_edge().unwrap().0, 427);
    let v = output.vertices()[130];
    assert!(almost_equal(v.x(), -49725.2369717, v.y(), -49780.2701485));
    assert_eq!(v.get_incident_edge().unwrap().0, 429);
    let v = output.vertices()[131];
    assert!(almost_equal(v.x(), -49782.7347104, v.y(), -49501.0660662));
    assert_eq!(v.get_incident_edge().unwrap().0, 431);
    let v = output.vertices()[132];
    assert!(almost_equal(v.x(), -49668.5872894, v.y(), -49415.8281012));
    assert_eq!(v.get_incident_edge().unwrap().0, 433);
    let v = output.vertices()[133];
    assert!(almost_equal(v.x(), -49676.4539474, v.y(), -49717.0328947));
    assert_eq!(v.get_incident_edge().unwrap().0, 439);
    let v = output.vertices()[134];
    assert!(almost_equal(v.x(), -49718.2595165, v.y(), -49785.7435471));
    assert_eq!(v.get_incident_edge().unwrap().0, 441);
    let v = output.vertices()[135];
    assert!(almost_equal(v.x(), -49787.0477165, v.y(), -48932.1340816));
    assert_eq!(v.get_incident_edge().unwrap().0, 443);
    let v = output.vertices()[136];
    assert!(almost_equal(v.x(), -49688.5000000, v.y(), -49094.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 445);
    let v = output.vertices()[137];
    assert!(almost_equal(v.x(), -49679.7225744, v.y(), -49179.5397367));
    assert_eq!(v.get_incident_edge().unwrap().0, 449);
    let v = output.vertices()[138];
    assert!(almost_equal(v.x(), -49672.6744264, v.y(), -49715.8829169));
    assert_eq!(v.get_incident_edge().unwrap().0, 451);
    let v = output.vertices()[139];
    assert!(almost_equal(v.x(), -49684.2028689, v.y(), -49160.6055328));
    assert_eq!(v.get_incident_edge().unwrap().0, 453);
    let v = output.vertices()[140];
    assert!(almost_equal(v.x(), -49684.1868033, v.y(), -49160.8521241));
    assert_eq!(v.get_incident_edge().unwrap().0, 455);
    let v = output.vertices()[141];
    assert!(almost_equal(v.x(), -49690.3021604, v.y(), -49077.4931424));
    assert_eq!(v.get_incident_edge().unwrap().0, 463);
    let v = output.vertices()[142];
    assert!(almost_equal(v.x(), -49684.4335279, v.y(), -49366.7556479));
    assert_eq!(v.get_incident_edge().unwrap().0, 465);
    let v = output.vertices()[143];
    assert!(almost_equal(v.x(), -49654.5000000, v.y(), -49409.7346729));
    assert_eq!(v.get_incident_edge().unwrap().0, 473);
    let v = output.vertices()[144];
    assert!(almost_equal(v.x(), -49653.2463127, v.y(), -49103.3281711));
    assert_eq!(v.get_incident_edge().unwrap().0, 475);
    let v = output.vertices()[145];
    assert!(almost_equal(v.x(), -49676.0540585, v.y(), -49788.8848906));
    assert_eq!(v.get_incident_edge().unwrap().0, 477);
    let v = output.vertices()[146];
    assert!(almost_equal(v.x(), -49677.6962617, v.y(), -49164.0046729));
    assert_eq!(v.get_incident_edge().unwrap().0, 479);
    let v = output.vertices()[147];
    assert!(almost_equal(v.x(), -49682.6916090, v.y(), -49076.3858131));
    assert_eq!(v.get_incident_edge().unwrap().0, 481);
    let v = output.vertices()[148];
    assert!(almost_equal(v.x(), -49679.0434948, v.y(), -49365.2897749));
    assert_eq!(v.get_incident_edge().unwrap().0, 483);
    let v = output.vertices()[149];
    assert!(almost_equal(v.x(), -49641.9657312, v.y(), -49418.0949090));
    assert_eq!(v.get_incident_edge().unwrap().0, 485);
    let v = output.vertices()[150];
    assert!(almost_equal(v.x(), -49676.5036853, v.y(), -49282.7136865));
    assert_eq!(v.get_incident_edge().unwrap().0, 487);
    let v = output.vertices()[151];
    assert!(almost_equal(v.x(), -49759.9422563, v.y(), -49529.5183915));
    assert_eq!(v.get_incident_edge().unwrap().0, 489);
    let v = output.vertices()[152];
    assert!(almost_equal(v.x(), -49639.3000773, v.y(), -49418.7718480));
    assert_eq!(v.get_incident_edge().unwrap().0, 491);
    let v = output.vertices()[153];
    assert!(almost_equal(v.x(), -49673.4693635, v.y(), -49074.6897680));
    assert_eq!(v.get_incident_edge().unwrap().0, 497);
    let v = output.vertices()[154];
    assert!(almost_equal(v.x(), -49661.4516266, v.y(), -49082.3606639));
    assert_eq!(v.get_incident_edge().unwrap().0, 499);
    let v = output.vertices()[155];
    assert!(almost_equal(v.x(), -49653.7757823, v.y(), -49787.2846259));
    assert_eq!(v.get_incident_edge().unwrap().0, 501);
    let v = output.vertices()[156];
    assert!(almost_equal(v.x(), -49648.2797848, v.y(), -49045.7217525));
    assert_eq!(v.get_incident_edge().unwrap().0, 503);
    let v = output.vertices()[157];
    assert!(almost_equal(v.x(), -49860.5163160, v.y(), -50063.2144858));
    assert_eq!(v.get_incident_edge().unwrap().0, 513);
    let v = output.vertices()[158];
    assert!(almost_equal(v.x(), -49632.0062696, v.y(), -49826.3416928));
    assert_eq!(v.get_incident_edge().unwrap().0, 515);
    let v = output.vertices()[159];
    assert!(almost_equal(v.x(), -49643.9647887, v.y(), -49142.7746479));
    assert_eq!(v.get_incident_edge().unwrap().0, 521);
    let v = output.vertices()[160];
    assert!(almost_equal(v.x(), -49623.2956990, v.y(), -49077.1831939));
    assert_eq!(v.get_incident_edge().unwrap().0, 523);
    let v = output.vertices()[161];
    assert!(almost_equal(v.x(), -49626.5848835, v.y(), -49428.2122088));
    assert_eq!(v.get_incident_edge().unwrap().0, 525);
    let v = output.vertices()[162];
    assert!(almost_equal(v.x(), -49614.5420199, v.y(), -49088.3057042));
    assert_eq!(v.get_incident_edge().unwrap().0, 531);
    let v = output.vertices()[163];
    assert!(almost_equal(v.x(), -49720.8191169, v.y(), -49524.0637230));
    assert_eq!(v.get_incident_edge().unwrap().0, 533);
    let v = output.vertices()[164];
    assert!(almost_equal(v.x(), -49650.1967871, v.y(), -49159.6626506));
    assert_eq!(v.get_incident_edge().unwrap().0, 535);
    let v = output.vertices()[165];
    assert!(almost_equal(v.x(), -49662.0000000, v.y(), -49570.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 537);
    let v = output.vertices()[166];
    assert!(almost_equal(v.x(), -49638.5000000, v.y(), -49458.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 539);
    let v = output.vertices()[167];
    assert!(almost_equal(v.x(), -49861.5956354, v.y(), -50073.7278562));
    assert_eq!(v.get_incident_edge().unwrap().0, 541);
    let v = output.vertices()[168];
    assert!(almost_equal(v.x(), -49635.9311881, v.y(), -49736.8737624));
    assert_eq!(v.get_incident_edge().unwrap().0, 547);
    let v = output.vertices()[169];
    assert!(almost_equal(v.x(), -49628.4148086, v.y(), -49751.5797223));
    assert_eq!(v.get_incident_edge().unwrap().0, 549);
    let v = output.vertices()[170];
    assert!(almost_equal(v.x(), -49654.5000000, v.y(), -49357.7823529));
    assert_eq!(v.get_incident_edge().unwrap().0, 551);
    let v = output.vertices()[171];
    assert!(almost_equal(v.x(), -49616.6989377, v.y(), -49110.4141110));
    assert_eq!(v.get_incident_edge().unwrap().0, 553);
    let v = output.vertices()[172];
    assert!(almost_equal(v.x(), -49703.4934797, v.y(), -49523.2917887));
    assert_eq!(v.get_incident_edge().unwrap().0, 561);
    let v = output.vertices()[173];
    assert!(almost_equal(v.x(), -49644.3846362, v.y(), -49706.4364377));
    assert_eq!(v.get_incident_edge().unwrap().0, 563);
    let v = output.vertices()[174];
    assert!(almost_equal(v.x(), -49604.0394776, v.y(), -49058.6660493));
    assert_eq!(v.get_incident_edge().unwrap().0, 565);
    let v = output.vertices()[175];
    assert!(almost_equal(v.x(), -49697.1083501, v.y(), -49519.3024721));
    assert_eq!(v.get_incident_edge().unwrap().0, 567);
    let v = output.vertices()[176];
    assert!(almost_equal(v.x(), -49607.8009916, v.y(), -49430.7119925));
    assert_eq!(v.get_incident_edge().unwrap().0, 569);
    let v = output.vertices()[177];
    assert!(almost_equal(v.x(), -49651.3276894, v.y(), -49484.0645993));
    assert_eq!(v.get_incident_edge().unwrap().0, 573);
    let v = output.vertices()[178];
    assert!(almost_equal(v.x(), -49643.9505442, v.y(), -49871.1902757));
    assert_eq!(v.get_incident_edge().unwrap().0, 575);
    let v = output.vertices()[179];
    assert!(almost_equal(v.x(), -49639.6966038, v.y(), -49158.6110085));
    assert_eq!(v.get_incident_edge().unwrap().0, 577);
    let v = output.vertices()[180];
    assert!(almost_equal(v.x(), -49781.1761031, v.y(), -48852.2445538));
    assert_eq!(v.get_incident_edge().unwrap().0, 579);
    let v = output.vertices()[181];
    assert!(almost_equal(v.x(), -49599.5809809, v.y(), -49412.1794976));
    assert_eq!(v.get_incident_edge().unwrap().0, 581);
    let v = output.vertices()[182];
    assert!(almost_equal(v.x(), -49656.4344844, v.y(), -49493.1269727));
    assert_eq!(v.get_incident_edge().unwrap().0, 583);
    let v = output.vertices()[183];
    assert!(almost_equal(v.x(), -49617.1171896, v.y(), -49483.4707026));
    assert_eq!(v.get_incident_edge().unwrap().0, 591);
    let v = output.vertices()[184];
    assert!(almost_equal(v.x(), -49621.7847739, v.y(), -49779.4377747));
    assert_eq!(v.get_incident_edge().unwrap().0, 593);
    let v = output.vertices()[185];
    assert!(almost_equal(v.x(), -49631.2798417, v.y(), -49010.8724307));
    assert_eq!(v.get_incident_edge().unwrap().0, 595);
    let v = output.vertices()[186];
    assert!(almost_equal(v.x(), -49624.9931507, v.y(), -49366.0456621));
    assert_eq!(v.get_incident_edge().unwrap().0, 597);
    let v = output.vertices()[187];
    assert!(almost_equal(v.x(), -49644.7710373, v.y(), -49889.8875450));
    assert_eq!(v.get_incident_edge().unwrap().0, 599);
    let v = output.vertices()[188];
    assert!(almost_equal(v.x(), -49644.6404553, v.y(), -49890.2472045));
    assert_eq!(v.get_incident_edge().unwrap().0, 605);
    let v = output.vertices()[189];
    assert!(almost_equal(v.x(), -49628.7132072, v.y(), -49000.5463317));
    assert_eq!(v.get_incident_edge().unwrap().0, 609);
    let v = output.vertices()[190];
    assert!(almost_equal(v.x(), -49863.3324097, v.y(), -50115.6999002));
    assert_eq!(v.get_incident_edge().unwrap().0, 611);
    let v = output.vertices()[191];
    assert!(almost_equal(v.x(), -49608.3727865, v.y(), -49794.8303820));
    assert_eq!(v.get_incident_edge().unwrap().0, 615);
    let v = output.vertices()[192];
    assert!(almost_equal(v.x(), -49578.6032150, v.y(), -49493.0991963));
    assert_eq!(v.get_incident_edge().unwrap().0, 617);
    let v = output.vertices()[193];
    assert!(almost_equal(v.x(), -49585.1818182, v.y(), -49055.3636364));
    assert_eq!(v.get_incident_edge().unwrap().0, 619);
    let v = output.vertices()[194];
    assert!(almost_equal(v.x(), -49597.7325419, v.y(), -49803.0795355));
    assert_eq!(v.get_incident_edge().unwrap().0, 621);
    let v = output.vertices()[195];
    assert!(almost_equal(v.x(), -49606.6395391, v.y(), -49886.7209217));
    assert_eq!(v.get_incident_edge().unwrap().0, 623);
    let v = output.vertices()[196];
    assert!(almost_equal(v.x(), -49583.8819840, v.y(), -49076.0014937));
    assert_eq!(v.get_incident_edge().unwrap().0, 625);
    let v = output.vertices()[197];
    assert!(almost_equal(v.x(), -49625.4146341, v.y(), -48989.2439024));
    assert_eq!(v.get_incident_edge().unwrap().0, 627);
    let v = output.vertices()[198];
    assert!(almost_equal(v.x(), -49692.6403380, v.y(), -50001.0161646));
    assert_eq!(v.get_incident_edge().unwrap().0, 629);
    let v = output.vertices()[199];
    assert!(almost_equal(v.x(), -49616.9014085, v.y(), -49637.6478873));
    assert_eq!(v.get_incident_edge().unwrap().0, 631);
    let v = output.vertices()[200];
    assert!(almost_equal(v.x(), -49628.8161411, v.y(), -49340.5590593));
    assert_eq!(v.get_incident_edge().unwrap().0, 633);
    let v = output.vertices()[201];
    assert!(almost_equal(v.x(), -49559.0500356, v.y(), -49800.0378841));
    assert_eq!(v.get_incident_edge().unwrap().0, 639);
    let v = output.vertices()[202];
    assert!(almost_equal(v.x(), -49591.1246546, v.y(), -49887.7506908));
    assert_eq!(v.get_incident_edge().unwrap().0, 641);
    let v = output.vertices()[203];
    assert!(almost_equal(v.x(), -49549.9728378, v.y(), -49811.9792710));
    assert_eq!(v.get_incident_edge().unwrap().0, 649);
    let v = output.vertices()[204];
    assert!(almost_equal(v.x(), -49601.7802513, v.y(), -49147.5889839));
    assert_eq!(v.get_incident_edge().unwrap().0, 651);
    let v = output.vertices()[205];
    assert!(almost_equal(v.x(), -49575.1640959, v.y(), -49558.4427306));
    assert_eq!(v.get_incident_edge().unwrap().0, 653);
    let v = output.vertices()[206];
    assert!(almost_equal(v.x(), -49570.3999829, v.y(), -49526.5001771));
    assert_eq!(v.get_incident_edge().unwrap().0, 657);
    let v = output.vertices()[207];
    assert!(almost_equal(v.x(), -49627.9577114, v.y(), -49996.8579957));
    assert_eq!(v.get_incident_edge().unwrap().0, 659);
    let v = output.vertices()[208];
    assert!(almost_equal(v.x(), -49571.1974860, v.y(), -49828.1770288));
    assert_eq!(v.get_incident_edge().unwrap().0, 661);
    let v = output.vertices()[209];
    assert!(almost_equal(v.x(), -49569.8490418, v.y(), -49829.4096830));
    assert_eq!(v.get_incident_edge().unwrap().0, 665);
    let v = output.vertices()[210];
    assert!(almost_equal(v.x(), -49571.2472952, v.y(), -49068.7847759));
    assert_eq!(v.get_incident_edge().unwrap().0, 667);
    let v = output.vertices()[211];
    assert!(almost_equal(v.x(), -49565.1369863, v.y(), -49939.7260274));
    assert_eq!(v.get_incident_edge().unwrap().0, 673);
    let v = output.vertices()[212];
    assert!(almost_equal(v.x(), -49554.9090909, v.y(), -49528.6969697));
    assert_eq!(v.get_incident_edge().unwrap().0, 675);
    let v = output.vertices()[213];
    assert!(almost_equal(v.x(), -49571.7123288, v.y(), -49956.5753425));
    assert_eq!(v.get_incident_edge().unwrap().0, 677);
    let v = output.vertices()[214];
    assert!(almost_equal(v.x(), -49555.4698216, v.y(), -49464.9741196));
    assert_eq!(v.get_incident_edge().unwrap().0, 683);
    let v = output.vertices()[215];
    assert!(almost_equal(v.x(), -49569.1411087, v.y(), -49593.2147228));
    assert_eq!(v.get_incident_edge().unwrap().0, 685);
    let v = output.vertices()[216];
    assert!(almost_equal(v.x(), -49569.1082677, v.y(), -49593.4055118));
    assert_eq!(v.get_incident_edge().unwrap().0, 687);
    let v = output.vertices()[217];
    assert!(almost_equal(v.x(), -49541.8587173, v.y(), -49423.2333997));
    assert_eq!(v.get_incident_edge().unwrap().0, 689);
    let v = output.vertices()[218];
    assert!(almost_equal(v.x(), -49600.1425951, v.y(), -49659.3842169));
    assert_eq!(v.get_incident_edge().unwrap().0, 691);
    let v = output.vertices()[219];
    assert!(almost_equal(v.x(), -49538.6666667, v.y(), -49534.1111111));
    assert_eq!(v.get_incident_edge().unwrap().0, 693);
    let v = output.vertices()[220];
    assert!(almost_equal(v.x(), -49594.7136801, v.y(), -49167.4007751));
    assert_eq!(v.get_incident_edge().unwrap().0, 695);
    let v = output.vertices()[221];
    assert!(almost_equal(v.x(), -49559.8773585, v.y(), -49060.4245283));
    assert_eq!(v.get_incident_edge().unwrap().0, 697);
    let v = output.vertices()[222];
    assert!(almost_equal(v.x(), -49548.8832141, v.y(), -49909.7848680));
    assert_eq!(v.get_incident_edge().unwrap().0, 699);
    let v = output.vertices()[223];
    assert!(almost_equal(v.x(), -49547.6075748, v.y(), -49910.0006365));
    assert_eq!(v.get_incident_edge().unwrap().0, 703);
    let v = output.vertices()[224];
    assert!(almost_equal(v.x(), -49547.3645199, v.y(), -49910.0443864));
    assert_eq!(v.get_incident_edge().unwrap().0, 705);
    let v = output.vertices()[225];
    assert!(almost_equal(v.x(), -49580.0225508, v.y(), -49986.6199913));
    assert_eq!(v.get_incident_edge().unwrap().0, 707);
    let v = output.vertices()[226];
    assert!(almost_equal(v.x(), -49539.5545906, v.y(), -49781.9160463));
    assert_eq!(v.get_incident_edge().unwrap().0, 713);
    let v = output.vertices()[227];
    assert!(almost_equal(v.x(), -49538.5400096, v.y(), -49784.5088644));
    assert_eq!(v.get_incident_edge().unwrap().0, 715);
    let v = output.vertices()[228];
    assert!(almost_equal(v.x(), -49533.8493654, v.y(), -49793.6511159));
    assert_eq!(v.get_incident_edge().unwrap().0, 717);
    let v = output.vertices()[229];
    assert!(almost_equal(v.x(), -49560.2429007, v.y(), -49985.6147936));
    assert_eq!(v.get_incident_edge().unwrap().0, 727);
    let v = output.vertices()[230];
    assert!(almost_equal(v.x(), -49519.6666667, v.y(), -49885.6666667));
    assert_eq!(v.get_incident_edge().unwrap().0, 729);
    let v = output.vertices()[231];
    assert!(almost_equal(v.x(), -49523.0632577, v.y(), -49873.8300640));
    assert_eq!(v.get_incident_edge().unwrap().0, 731);
    let v = output.vertices()[232];
    assert!(almost_equal(v.x(), -49519.0244624, v.y(), -49878.4973689));
    assert_eq!(v.get_incident_edge().unwrap().0, 735);
    let v = output.vertices()[233];
    assert!(almost_equal(v.x(), -49517.0785486, v.y(), -49878.3880439));
    assert_eq!(v.get_incident_edge().unwrap().0, 737);
    let v = output.vertices()[234];
    assert!(almost_equal(v.x(), -49524.7808849, v.y(), -49909.1920707));
    assert_eq!(v.get_incident_edge().unwrap().0, 739);
    let v = output.vertices()[235];
    assert!(almost_equal(v.x(), -49581.8167872, v.y(), -49177.1220688));
    assert_eq!(v.get_incident_edge().unwrap().0, 741);
    let v = output.vertices()[236];
    assert!(almost_equal(v.x(), -49535.4241774, v.y(), -49972.1537911));
    assert_eq!(v.get_incident_edge().unwrap().0, 743);
    let v = output.vertices()[237];
    assert!(almost_equal(v.x(), -49533.0212594, v.y(), -49970.7923131));
    assert_eq!(v.get_incident_edge().unwrap().0, 745);
    let v = output.vertices()[238];
    assert!(almost_equal(v.x(), -49503.8590290, v.y(), -50000.6394129));
    assert_eq!(v.get_incident_edge().unwrap().0, 747);
    let v = output.vertices()[239];
    assert!(almost_equal(v.x(), -49521.6350619, v.y(), -49433.8339783));
    assert_eq!(v.get_incident_edge().unwrap().0, 749);
    let v = output.vertices()[240];
    assert!(almost_equal(v.x(), -49500.9917372, v.y(), -49866.2080296));
    assert_eq!(v.get_incident_edge().unwrap().0, 751);
    let v = output.vertices()[241];
    assert!(almost_equal(v.x(), -49512.4916988, v.y(), -49826.3335964));
    assert_eq!(v.get_incident_edge().unwrap().0, 755);
    let v = output.vertices()[242];
    assert!(almost_equal(v.x(), -49592.2027450, v.y(), -49295.6033704));
    assert_eq!(v.get_incident_edge().unwrap().0, 757);
    let v = output.vertices()[243];
    assert!(almost_equal(v.x(), -49508.6859726, v.y(), -50046.2708403));
    assert_eq!(v.get_incident_edge().unwrap().0, 765);
    let v = output.vertices()[244];
    assert!(almost_equal(v.x(), -49502.6365580, v.y(), -49836.9312869));
    assert_eq!(v.get_incident_edge().unwrap().0, 767);
    let v = output.vertices()[245];
    assert!(almost_equal(v.x(), -49488.3483105, v.y(), -49986.6651742));
    assert_eq!(v.get_incident_edge().unwrap().0, 769);
    let v = output.vertices()[246];
    assert!(almost_equal(v.x(), -49587.4214907, v.y(), -49289.6105647));
    assert_eq!(v.get_incident_edge().unwrap().0, 771);
    let v = output.vertices()[247];
    assert!(almost_equal(v.x(), -49479.2581441, v.y(), -49999.1316881));
    assert_eq!(v.get_incident_edge().unwrap().0, 773);
    let v = output.vertices()[248];
    assert!(almost_equal(v.x(), -49481.5180461, v.y(), -49982.6427492));
    assert_eq!(v.get_incident_edge().unwrap().0, 777);
    let v = output.vertices()[249];
    assert!(almost_equal(v.x(), -49490.5073271, v.y(), -49902.9800245));
    assert_eq!(v.get_incident_edge().unwrap().0, 779);
    let v = output.vertices()[250];
    assert!(almost_equal(v.x(), -49475.6511708, v.y(), -49970.6440963));
    assert_eq!(v.get_incident_edge().unwrap().0, 785);
    let v = output.vertices()[251];
    assert!(almost_equal(v.x(), -49485.5938724, v.y(), -49942.7323590));
    assert_eq!(v.get_incident_edge().unwrap().0, 787);
    let v = output.vertices()[252];
    assert!(almost_equal(v.x(), -49557.9168526, v.y(), -49658.8322464));
    assert_eq!(v.get_incident_edge().unwrap().0, 789);
    let v = output.vertices()[253];
    assert!(almost_equal(v.x(), -49478.9409477, v.y(), -49807.5846473));
    assert_eq!(v.get_incident_edge().unwrap().0, 791);
    let v = output.vertices()[254];
    assert!(almost_equal(v.x(), -49573.5843077, v.y(), -49284.3344810));
    assert_eq!(v.get_incident_edge().unwrap().0, 793);
    let v = output.vertices()[255];
    assert!(almost_equal(v.x(), -49464.1153846, v.y(), -49962.1153846));
    assert_eq!(v.get_incident_edge().unwrap().0, 795);
    let v = output.vertices()[256];
    assert!(almost_equal(v.x(), -49478.5821494, v.y(), -49780.2168488));
    assert_eq!(v.get_incident_edge().unwrap().0, 799);
    let v = output.vertices()[257];
    assert!(almost_equal(v.x(), -49475.6029337, v.y(), -49892.0346338));
    assert_eq!(v.get_incident_edge().unwrap().0, 801);
    let v = output.vertices()[258];
    assert!(almost_equal(v.x(), -49464.7517213, v.y(), -49937.9927096));
    assert_eq!(v.get_incident_edge().unwrap().0, 803);
    let v = output.vertices()[259];
    assert!(almost_equal(v.x(), -49496.9518760, v.y(), -49477.1794454));
    assert_eq!(v.get_incident_edge().unwrap().0, 805);
    let v = output.vertices()[260];
    assert!(almost_equal(v.x(), -49476.1698009, v.y(), -49761.4184181));
    assert_eq!(v.get_incident_edge().unwrap().0, 807);
    let v = output.vertices()[261];
    assert!(almost_equal(v.x(), -49525.7015473, v.y(), -49673.1932861));
    assert_eq!(v.get_incident_edge().unwrap().0, 809);
    let v = output.vertices()[262];
    assert!(almost_equal(v.x(), -49518.5110069, v.y(), -50146.9563335));
    assert_eq!(v.get_incident_edge().unwrap().0, 815);
    let v = output.vertices()[263];
    assert!(almost_equal(v.x(), -49519.9873618, v.y(), -49675.6520500));
    assert_eq!(v.get_incident_edge().unwrap().0, 817);
    let v = output.vertices()[264];
    assert!(almost_equal(v.x(), -49477.7291667, v.y(), -49532.4861111));
    assert_eq!(v.get_incident_edge().unwrap().0, 823);
    let v = output.vertices()[265];
    assert!(almost_equal(v.x(), -49483.1089594, v.y(), -49476.6603360));
    assert_eq!(v.get_incident_edge().unwrap().0, 825);
    let v = output.vertices()[266];
    assert!(almost_equal(v.x(), -49480.3361691, v.y(), -49469.8039818));
    assert_eq!(v.get_incident_edge().unwrap().0, 831);
    let v = output.vertices()[267];
    assert!(almost_equal(v.x(), -49470.9373729, v.y(), -49457.8915540));
    assert_eq!(v.get_incident_edge().unwrap().0, 833);
    let v = output.vertices()[268];
    assert!(almost_equal(v.x(), -49472.7971705, v.y(), -49531.5190530));
    assert_eq!(v.get_incident_edge().unwrap().0, 835);
    let v = output.vertices()[269];
    assert!(almost_equal(v.x(), -49529.3365952, v.y(), -49220.4309651));
    assert_eq!(v.get_incident_edge().unwrap().0, 841);
    let v = output.vertices()[270];
    assert!(almost_equal(v.x(), -49535.7305104, v.y(), -49266.6092415));
    assert_eq!(v.get_incident_edge().unwrap().0, 843);
    let v = output.vertices()[271];
    assert!(almost_equal(v.x(), -49466.5931950, v.y(), -49541.5596976));
    assert_eq!(v.get_incident_edge().unwrap().0, 845);
    let v = output.vertices()[272];
    assert!(almost_equal(v.x(), -49452.2697130, v.y(), -49540.9230984));
    assert_eq!(v.get_incident_edge().unwrap().0, 847);
    let v = output.vertices()[273];
    assert!(almost_equal(v.x(), -49512.2100683, v.y(), -49284.9190798));
    assert_eq!(v.get_incident_edge().unwrap().0, 849);
    let v = output.vertices()[274];
    assert!(almost_equal(v.x(), -49430.6480670, v.y(), -49454.0082474));
    assert_eq!(v.get_incident_edge().unwrap().0, 851);
    let v = output.vertices()[275];
    assert!(almost_equal(v.x(), -49439.5756926, v.y(), -49540.8592180));
    assert_eq!(v.get_incident_edge().unwrap().0, 853);
    let v = output.vertices()[276];
    assert!(almost_equal(v.x(), -49484.0698327, v.y(), -49163.4404368));
    assert_eq!(v.get_incident_edge().unwrap().0, 855);
    let v = output.vertices()[277];
    assert!(almost_equal(v.x(), -49370.6815759, v.y(), -49204.3021961));
    assert_eq!(v.get_incident_edge().unwrap().0, 863);
    let v = output.vertices()[278];
    assert!(almost_equal(v.x(), -49480.9487420, v.y(), -49671.7850169));
    assert_eq!(v.get_incident_edge().unwrap().0, 865);
    let v = output.vertices()[279];
    assert!(almost_equal(v.x(), -49479.8896104, v.y(), -49029.3181818));
    assert_eq!(v.get_incident_edge().unwrap().0, 871);
    let v = output.vertices()[280];
    assert!(almost_equal(v.x(), -49475.4896579, v.y(), -49043.5334129));
    assert_eq!(v.get_incident_edge().unwrap().0, 873);
    let v = output.vertices()[281];
    assert!(almost_equal(v.x(), -49363.4702970, v.y(), -49030.5099010));
    assert_eq!(v.get_incident_edge().unwrap().0, 879);
    let v = output.vertices()[282];
    assert!(almost_equal(v.x(), -49466.7495066, v.y(), -49316.9000564));
    assert_eq!(v.get_incident_edge().unwrap().0, 881);
    let v = output.vertices()[283];
    assert!(almost_equal(v.x(), -49361.7440711, v.y(), -49031.7666280));
    assert_eq!(v.get_incident_edge().unwrap().0, 883);
    let v = output.vertices()[284];
    assert!(almost_equal(v.x(), -49481.7096774, v.y(), -48991.0967742));
    assert_eq!(v.get_incident_edge().unwrap().0, 887);
    let v = output.vertices()[285];
    assert!(almost_equal(v.x(), -49360.4052824, v.y(), -49543.8146260));
    assert_eq!(v.get_incident_edge().unwrap().0, 889);
    let v = output.vertices()[286];
    assert!(almost_equal(v.x(), -49466.7665076, v.y(), -49130.5620525));
    assert_eq!(v.get_incident_edge().unwrap().0, 893);
    let v = output.vertices()[287];
    assert!(almost_equal(v.x(), -49467.7058012, v.y(), -49139.9122935));
    assert_eq!(v.get_incident_edge().unwrap().0, 895);
    let v = output.vertices()[288];
    assert!(almost_equal(v.x(), -49354.3825121, v.y(), -49551.7907273));
    assert_eq!(v.get_incident_edge().unwrap().0, 897);
    let v = output.vertices()[289];
    assert!(almost_equal(v.x(), -49458.3721428, v.y(), -49683.8714791));
    assert_eq!(v.get_incident_edge().unwrap().0, 899);
    let v = output.vertices()[290];
    assert!(almost_equal(v.x(), -49383.8432712, v.y(), -49463.1231714));
    assert_eq!(v.get_incident_edge().unwrap().0, 901);
    let v = output.vertices()[291];
    assert!(almost_equal(v.x(), -49381.8812920, v.y(), -49464.1770782));
    assert_eq!(v.get_incident_edge().unwrap().0, 905);
    let v = output.vertices()[292];
    assert!(almost_equal(v.x(), -49338.6344430, v.y(), -49170.0176810));
    assert_eq!(v.get_incident_edge().unwrap().0, 911);
    let v = output.vertices()[293];
    assert!(almost_equal(v.x(), -49347.0407029, v.y(), -49534.4168211));
    assert_eq!(v.get_incident_edge().unwrap().0, 913);
    let v = output.vertices()[294];
    assert!(almost_equal(v.x(), -49393.8733403, v.y(), -49808.9202031));
    assert_eq!(v.get_incident_edge().unwrap().0, 919);
    let v = output.vertices()[295];
    assert!(almost_equal(v.x(), -49393.1857491, v.y(), -49788.8081608));
    assert_eq!(v.get_incident_edge().unwrap().0, 925);
    let v = output.vertices()[296];
    assert!(almost_equal(v.x(), -49364.2730627, v.y(), -49148.1494465));
    assert_eq!(v.get_incident_edge().unwrap().0, 927);
    let v = output.vertices()[297];
    assert!(almost_equal(v.x(), -49396.2103139, v.y(), -49829.1251261));
    assert_eq!(v.get_incident_edge().unwrap().0, 929);
    let v = output.vertices()[298];
    assert!(almost_equal(v.x(), -49335.5972973, v.y(), -49792.9216216));
    assert_eq!(v.get_incident_edge().unwrap().0, 931);
    let v = output.vertices()[299];
    assert!(almost_equal(v.x(), -49335.4425683, v.y(), -49550.2031709));
    assert_eq!(v.get_incident_edge().unwrap().0, 935);
    let v = output.vertices()[300];
    assert!(almost_equal(v.x(), -49404.7126508, v.y(), -49872.6951649));
    assert_eq!(v.get_incident_edge().unwrap().0, 939);
    let v = output.vertices()[301];
    assert!(almost_equal(v.x(), -49402.7096774, v.y(), -49877.7741935));
    assert_eq!(v.get_incident_edge().unwrap().0, 941);
    let v = output.vertices()[302];
    assert!(almost_equal(v.x(), -49431.2782848, v.y(), -49689.6225339));
    assert_eq!(v.get_incident_edge().unwrap().0, 947);
    let v = output.vertices()[303];
    assert!(almost_equal(v.x(), -49342.7932673, v.y(), -49142.0123621));
    assert_eq!(v.get_incident_edge().unwrap().0, 949);
    let v = output.vertices()[304];
    assert!(almost_equal(v.x(), -49324.9803036, v.y(), -49798.4818642));
    assert_eq!(v.get_incident_edge().unwrap().0, 951);
    let v = output.vertices()[305];
    assert!(almost_equal(v.x(), -49356.1448310, v.y(), -49599.0002873));
    assert_eq!(v.get_incident_edge().unwrap().0, 953);
    let v = output.vertices()[306];
    assert!(almost_equal(v.x(), -49398.3480691, v.y(), -49727.8927846));
    assert_eq!(v.get_incident_edge().unwrap().0, 955);
    let v = output.vertices()[307];
    assert!(almost_equal(v.x(), -49329.0806452, v.y(), -49764.6827957));
    assert_eq!(v.get_incident_edge().unwrap().0, 961);
    let v = output.vertices()[308];
    assert!(almost_equal(v.x(), -49395.7133372, v.y(), -49325.0184758));
    assert_eq!(v.get_incident_edge().unwrap().0, 963);
    let v = output.vertices()[309];
    assert!(almost_equal(v.x(), -49389.1387435, v.y(), -49944.4738220));
    assert_eq!(v.get_incident_edge().unwrap().0, 965);
    let v = output.vertices()[310];
    assert!(almost_equal(v.x(), -49312.2851619, v.y(), -49783.8541966));
    assert_eq!(v.get_incident_edge().unwrap().0, 967);
    let v = output.vertices()[311];
    assert!(almost_equal(v.x(), -49368.6368421, v.y(), -49634.8052632));
    assert_eq!(v.get_incident_edge().unwrap().0, 973);
    let v = output.vertices()[312];
    assert!(almost_equal(v.x(), -49370.8124798, v.y(), -49639.3341416));
    assert_eq!(v.get_incident_edge().unwrap().0, 975);
    let v = output.vertices()[313];
    assert!(almost_equal(v.x(), -49355.9822946, v.y(), -49613.7589814));
    assert_eq!(v.get_incident_edge().unwrap().0, 979);
    let v = output.vertices()[314];
    assert!(almost_equal(v.x(), -49362.7994987, v.y(), -49731.7644110));
    assert_eq!(v.get_incident_edge().unwrap().0, 981);
    let v = output.vertices()[315];
    assert!(almost_equal(v.x(), -49343.4441275, v.y(), -49245.5202392));
    assert_eq!(v.get_incident_edge().unwrap().0, 991);
    let v = output.vertices()[316];
    assert!(almost_equal(v.x(), -49338.8449692, v.y(), -49236.8968172));
    assert_eq!(v.get_incident_edge().unwrap().0, 993);
    let v = output.vertices()[317];
    assert!(almost_equal(v.x(), -49347.4368910, v.y(), -49259.3009703));
    assert_eq!(v.get_incident_edge().unwrap().0, 995);
    let v = output.vertices()[318];
    assert!(almost_equal(v.x(), -49314.0528228, v.y(), -49551.1422505));
    assert_eq!(v.get_incident_edge().unwrap().0, 997);
    let v = output.vertices()[319];
    assert!(almost_equal(v.x(), -49323.6877204, v.y(), -49115.7607987));
    assert_eq!(v.get_incident_edge().unwrap().0, 999);
    let v = output.vertices()[320];
    assert!(almost_equal(v.x(), -49340.2179198, v.y(), -49736.2208717));
    assert_eq!(v.get_incident_edge().unwrap().0, 1001);
    let v = output.vertices()[321];
    assert!(almost_equal(v.x(), -49351.3200554, v.y(), -49276.6681881));
    assert_eq!(v.get_incident_edge().unwrap().0, 1003);
    let v = output.vertices()[322];
    assert!(almost_equal(v.x(), -49344.4514248, v.y(), -49475.4399281));
    assert_eq!(v.get_incident_edge().unwrap().0, 1005);
    let v = output.vertices()[323];
    assert!(almost_equal(v.x(), -49321.8402490, v.y(), -49601.8609959));
    assert_eq!(v.get_incident_edge().unwrap().0, 1007);
    let v = output.vertices()[324];
    assert!(almost_equal(v.x(), -49319.4298030, v.y(), -49602.0652709));
    assert_eq!(v.get_incident_edge().unwrap().0, 1011);
    let v = output.vertices()[325];
    assert!(almost_equal(v.x(), -49309.2628980, v.y(), -49551.5515921));
    assert_eq!(v.get_incident_edge().unwrap().0, 1013);
    let v = output.vertices()[326];
    assert!(almost_equal(v.x(), -49377.7792363, v.y(), -49944.2553699));
    assert_eq!(v.get_incident_edge().unwrap().0, 1017);
    let v = output.vertices()[327];
    assert!(almost_equal(v.x(), -49357.5590868, v.y(), -49297.5161146));
    assert_eq!(v.get_incident_edge().unwrap().0, 1019);
    let v = output.vertices()[328];
    assert!(almost_equal(v.x(), -49307.6308756, v.y(), -49604.0603684));
    assert_eq!(v.get_incident_edge().unwrap().0, 1025);
    let v = output.vertices()[329];
    assert!(almost_equal(v.x(), -49300.1268657, v.y(), -49862.2313433));
    assert_eq!(v.get_incident_edge().unwrap().0, 1031);
    let v = output.vertices()[330];
    assert!(almost_equal(v.x(), -49303.8596713, v.y(), -49662.5101138));
    assert_eq!(v.get_incident_edge().unwrap().0, 1033);
    let v = output.vertices()[331];
    assert!(almost_equal(v.x(), -49368.1846862, v.y(), -49326.1040366));
    assert_eq!(v.get_incident_edge().unwrap().0, 1035);
    let v = output.vertices()[332];
    assert!(almost_equal(v.x(), -49280.2795414, v.y(), -49790.5446793));
    assert_eq!(v.get_incident_edge().unwrap().0, 1037);
    let v = output.vertices()[333];
    assert!(almost_equal(v.x(), -49302.2693244, v.y(), -49663.0374317));
    assert_eq!(v.get_incident_edge().unwrap().0, 1039);
    let v = output.vertices()[334];
    assert!(almost_equal(v.x(), -49314.8046492, v.y(), -49089.1096632));
    assert_eq!(v.get_incident_edge().unwrap().0, 1041);
    let v = output.vertices()[335];
    assert!(almost_equal(v.x(), -49304.8551109, v.y(), -49744.2272580));
    assert_eq!(v.get_incident_edge().unwrap().0, 1043);
    let v = output.vertices()[336];
    assert!(almost_equal(v.x(), -49314.3458686, v.y(), -49089.9316107));
    assert_eq!(v.get_incident_edge().unwrap().0, 1045);
    let v = output.vertices()[337];
    assert!(almost_equal(v.x(), -49314.2788261, v.y(), -49089.7468338));
    assert_eq!(v.get_incident_edge().unwrap().0, 1047);
    let v = output.vertices()[338];
    assert!(almost_equal(v.x(), -49299.1865185, v.y(), -49845.8470004));
    assert_eq!(v.get_incident_edge().unwrap().0, 1049);
    let v = output.vertices()[339];
    assert!(almost_equal(v.x(), -49284.9051168, v.y(), -49610.0437151));
    assert_eq!(v.get_incident_edge().unwrap().0, 1051);
    let v = output.vertices()[340];
    assert!(almost_equal(v.x(), -49295.8764580, v.y(), -49829.8995899));
    assert_eq!(v.get_incident_edge().unwrap().0, 1053);
    let v = output.vertices()[341];
    assert!(almost_equal(v.x(), -49348.1334278, v.y(), -49339.5695797));
    assert_eq!(v.get_incident_edge().unwrap().0, 1061);
    let v = output.vertices()[342];
    assert!(almost_equal(v.x(), -49288.6740338, v.y(), -49826.9208132));
    assert_eq!(v.get_incident_edge().unwrap().0, 1067);
    let v = output.vertices()[343];
    assert!(almost_equal(v.x(), -49345.6328692, v.y(), -49341.0253844));
    assert_eq!(v.get_incident_edge().unwrap().0, 1069);
    let v = output.vertices()[344];
    assert!(almost_equal(v.x(), -49321.7752478, v.y(), -49379.4050362));
    assert_eq!(v.get_incident_edge().unwrap().0, 1071);
    let v = output.vertices()[345];
    assert!(almost_equal(v.x(), -49271.5918367, v.y(), -49616.2993197));
    assert_eq!(v.get_incident_edge().unwrap().0, 1073);
    let v = output.vertices()[346];
    assert!(almost_equal(v.x(), -49279.1731583, v.y(), -49035.0999153));
    assert_eq!(v.get_incident_edge().unwrap().0, 1075);
    let v = output.vertices()[347];
    assert!(almost_equal(v.x(), -49358.8500793, v.y(), -49991.3430239));
    assert_eq!(v.get_incident_edge().unwrap().0, 1081);
    let v = output.vertices()[348];
    assert!(almost_equal(v.x(), -49286.1554593, v.y(), -49082.8123050));
    assert_eq!(v.get_incident_edge().unwrap().0, 1083);
    let v = output.vertices()[349];
    assert!(almost_equal(v.x(), -49357.1830664, v.y(), -49972.0407689));
    assert_eq!(v.get_incident_edge().unwrap().0, 1087);
    let v = output.vertices()[350];
    assert!(almost_equal(v.x(), -49263.8830401, v.y(), -49805.9732075));
    assert_eq!(v.get_incident_edge().unwrap().0, 1089);
    let v = output.vertices()[351];
    assert!(almost_equal(v.x(), -49279.4488174, v.y(), -49544.3149585));
    assert_eq!(v.get_incident_edge().unwrap().0, 1091);
    let v = output.vertices()[352];
    assert!(almost_equal(v.x(), -49279.4287182, v.y(), -49670.2841000));
    assert_eq!(v.get_incident_edge().unwrap().0, 1095);
    let v = output.vertices()[353];
    assert!(almost_equal(v.x(), -49248.9564524, v.y(), -49799.4344126));
    assert_eq!(v.get_incident_edge().unwrap().0, 1097);
    let v = output.vertices()[354];
    assert!(almost_equal(v.x(), -49259.2686833, v.y(), -49625.3362989));
    assert_eq!(v.get_incident_edge().unwrap().0, 1105);
    let v = output.vertices()[355];
    assert!(almost_equal(v.x(), -49303.0920181, v.y(), -49432.0312344));
    assert_eq!(v.get_incident_edge().unwrap().0, 1107);
    let v = output.vertices()[356];
    assert!(almost_equal(v.x(), -49310.2394491, v.y(), -49943.1235007));
    assert_eq!(v.get_incident_edge().unwrap().0, 1109);
    let v = output.vertices()[357];
    assert!(almost_equal(v.x(), -49244.0458493, v.y(), -49773.2445299));
    assert_eq!(v.get_incident_edge().unwrap().0, 1113);
    let v = output.vertices()[358];
    assert!(almost_equal(v.x(), -49262.8479119, v.y(), -49443.6236754));
    assert_eq!(v.get_incident_edge().unwrap().0, 1117);
    let v = output.vertices()[359];
    assert!(almost_equal(v.x(), -49254.4330709, v.y(), -49446.4426947));
    assert_eq!(v.get_incident_edge().unwrap().0, 1119);
    let v = output.vertices()[360];
    assert!(almost_equal(v.x(), -49239.4998931, v.y(), -49775.8933287));
    assert_eq!(v.get_incident_edge().unwrap().0, 1125);
    let v = output.vertices()[361];
    assert!(almost_equal(v.x(), -49238.0852620, v.y(), -49775.6751829));
    assert_eq!(v.get_incident_edge().unwrap().0, 1127);
    let v = output.vertices()[362];
    assert!(almost_equal(v.x(), -49352.6682047, v.y(), -50068.8895825));
    assert_eq!(v.get_incident_edge().unwrap().0, 1131);
    let v = output.vertices()[363];
    assert!(almost_equal(v.x(), -49230.2465928, v.y(), -49773.4599659));
    assert_eq!(v.get_incident_edge().unwrap().0, 1133);
    let v = output.vertices()[364];
    assert!(almost_equal(v.x(), -49352.5389800, v.y(), -50073.9424592));
    assert_eq!(v.get_incident_edge().unwrap().0, 1135);
    let v = output.vertices()[365];
    assert!(almost_equal(v.x(), -49293.2433206, v.y(), -49931.0667942));
    assert_eq!(v.get_incident_edge().unwrap().0, 1137);
    let v = output.vertices()[366];
    assert!(almost_equal(v.x(), -49255.1176051, v.y(), -50024.8961749));
    assert_eq!(v.get_incident_edge().unwrap().0, 1139);
    let v = output.vertices()[367];
    assert!(almost_equal(v.x(), -49283.1988638, v.y(), -48987.6384575));
    assert_eq!(v.get_incident_edge().unwrap().0, 1141);
    let v = output.vertices()[368];
    assert!(almost_equal(v.x(), -49285.9429600, v.y(), -48976.0193216));
    assert_eq!(v.get_incident_edge().unwrap().0, 1143);
    let v = output.vertices()[369];
    assert!(almost_equal(v.x(), -49230.0056063, v.y(), -50004.8418413));
    assert_eq!(v.get_incident_edge().unwrap().0, 1145);
    let v = output.vertices()[370];
    assert!(almost_equal(v.x(), -49247.3808564, v.y(), -49668.3153652));
    assert_eq!(v.get_incident_edge().unwrap().0, 1149);
    let v = output.vertices()[371];
    assert!(almost_equal(v.x(), -49245.6338474, v.y(), -49675.0212445));
    assert_eq!(v.get_incident_edge().unwrap().0, 1159);
    let v = output.vertices()[372];
    assert!(almost_equal(v.x(), -49237.7543860, v.y(), -49264.8421053));
    assert_eq!(v.get_incident_edge().unwrap().0, 1161);
    let v = output.vertices()[373];
    assert!(almost_equal(v.x(), -49237.9763780, v.y(), -49895.2489400));
    assert_eq!(v.get_incident_edge().unwrap().0, 1163);
    let v = output.vertices()[374];
    assert!(almost_equal(v.x(), -49241.6885965, v.y(), -49230.4605263));
    assert_eq!(v.get_incident_edge().unwrap().0, 1165);
    let v = output.vertices()[375];
    assert!(almost_equal(v.x(), -49226.8887723, v.y(), -49577.7187828));
    assert_eq!(v.get_incident_edge().unwrap().0, 1167);
    let v = output.vertices()[376];
    assert!(almost_equal(v.x(), -49213.9398531, v.y(), -49592.1052608));
    assert_eq!(v.get_incident_edge().unwrap().0, 1169);
    let v = output.vertices()[377];
    assert!(almost_equal(v.x(), -49231.8732985, v.y(), -49888.0191383));
    assert_eq!(v.get_incident_edge().unwrap().0, 1171);
    let v = output.vertices()[378];
    assert!(almost_equal(v.x(), -49194.9028254, v.y(), -49951.9687658));
    assert_eq!(v.get_incident_edge().unwrap().0, 1175);
    let v = output.vertices()[379];
    assert!(almost_equal(v.x(), -49222.3583884, v.y(), -49570.8100267));
    assert_eq!(v.get_incident_edge().unwrap().0, 1177);
    let v = output.vertices()[380];
    assert!(almost_equal(v.x(), -49227.1982759, v.y(), -49301.0344828));
    assert_eq!(v.get_incident_edge().unwrap().0, 1181);
    let v = output.vertices()[381];
    assert!(almost_equal(v.x(), -49182.9372694, v.y(), -49937.5867159));
    assert_eq!(v.get_incident_edge().unwrap().0, 1187);
    let v = output.vertices()[382];
    assert!(almost_equal(v.x(), -49183.8115443, v.y(), -49941.1130734));
    assert_eq!(v.get_incident_edge().unwrap().0, 1189);
    let v = output.vertices()[383];
    assert!(almost_equal(v.x(), -49177.3812303, v.y(), -49616.8707795));
    assert_eq!(v.get_incident_edge().unwrap().0, 1191);
    let v = output.vertices()[384];
    assert!(almost_equal(v.x(), -49226.4502762, v.y(), -49301.5331492));
    assert_eq!(v.get_incident_edge().unwrap().0, 1193);
    let v = output.vertices()[385];
    assert!(almost_equal(v.x(), -49224.8814351, v.y(), -49548.8287395));
    assert_eq!(v.get_incident_edge().unwrap().0, 1195);
    let v = output.vertices()[386];
    assert!(almost_equal(v.x(), -49239.6843854, v.y(), -49188.3720931));
    assert_eq!(v.get_incident_edge().unwrap().0, 1197);
    let v = output.vertices()[387];
    assert!(almost_equal(v.x(), -49225.8665476, v.y(), -49543.5623904));
    assert_eq!(v.get_incident_edge().unwrap().0, 1199);
    let v = output.vertices()[388];
    assert!(almost_equal(v.x(), -49223.0919428, v.y(), -49540.6796517));
    assert_eq!(v.get_incident_edge().unwrap().0, 1201);
    let v = output.vertices()[389];
    assert!(almost_equal(v.x(), -49175.1857506, v.y(), -49911.2315522));
    assert_eq!(v.get_incident_edge().unwrap().0, 1205);
    let v = output.vertices()[390];
    assert!(almost_equal(v.x(), -49227.6527646, v.y(), -49178.6888362));
    assert_eq!(v.get_incident_edge().unwrap().0, 1207);
    let v = output.vertices()[391];
    assert!(almost_equal(v.x(), -49203.7185034, v.y(), -49524.1053703));
    assert_eq!(v.get_incident_edge().unwrap().0, 1209);
    let v = output.vertices()[392];
    assert!(almost_equal(v.x(), -49207.2510711, v.y(), -49109.8988860));
    assert_eq!(v.get_incident_edge().unwrap().0, 1213);
    let v = output.vertices()[393];
    assert!(almost_equal(v.x(), -49153.4690265, v.y(), -49959.3185841));
    assert_eq!(v.get_incident_edge().unwrap().0, 1215);
    let v = output.vertices()[394];
    assert!(almost_equal(v.x(), -49196.1352463, v.y(), -49329.7694800));
    assert_eq!(v.get_incident_edge().unwrap().0, 1217);
    let v = output.vertices()[395];
    assert!(almost_equal(v.x(), -49205.5547825, v.y(), -49162.5864306));
    assert_eq!(v.get_incident_edge().unwrap().0, 1227);
    let v = output.vertices()[396];
    assert!(almost_equal(v.x(), -49140.2864078, v.y(), -49659.8398058));
    assert_eq!(v.get_incident_edge().unwrap().0, 1229);
    let v = output.vertices()[397];
    assert!(almost_equal(v.x(), -49144.5074289, v.y(), -49603.8749487));
    assert_eq!(v.get_incident_edge().unwrap().0, 1235);
    let v = output.vertices()[398];
    assert!(almost_equal(v.x(), -49171.2555911, v.y(), -49809.8594249));
    assert_eq!(v.get_incident_edge().unwrap().0, 1237);
    let v = output.vertices()[399];
    assert!(almost_equal(v.x(), -49171.3620382, v.y(), -49810.3514941));
    assert_eq!(v.get_incident_edge().unwrap().0, 1239);
    let v = output.vertices()[400];
    assert!(almost_equal(v.x(), -49180.5699650, v.y(), -49852.5192378));
    assert_eq!(v.get_incident_edge().unwrap().0, 1243);
    let v = output.vertices()[401];
    assert!(almost_equal(v.x(), -49180.2530937, v.y(), -49707.7223851));
    assert_eq!(v.get_incident_edge().unwrap().0, 1245);
    let v = output.vertices()[402];
    assert!(almost_equal(v.x(), -49165.3073306, v.y(), -49695.8324090));
    assert_eq!(v.get_incident_edge().unwrap().0, 1251);
    let v = output.vertices()[403];
    assert!(almost_equal(v.x(), -49179.5791731, v.y(), -49708.0615079));
    assert_eq!(v.get_incident_edge().unwrap().0, 1253);
    let v = output.vertices()[404];
    assert!(almost_equal(v.x(), -49158.2962486, v.y(), -49205.1774549));
    assert_eq!(v.get_incident_edge().unwrap().0, 1255);
    let v = output.vertices()[405];
    assert!(almost_equal(v.x(), -49169.9329768, v.y(), -49865.5408947));
    assert_eq!(v.get_incident_edge().unwrap().0, 1257);
    let v = output.vertices()[406];
    assert!(almost_equal(v.x(), -49154.6513476, v.y(), -49874.0577303));
    assert_eq!(v.get_incident_edge().unwrap().0, 1259);
    let v = output.vertices()[407];
    assert!(almost_equal(v.x(), -49143.9251046, v.y(), -49679.0829986));
    assert_eq!(v.get_incident_edge().unwrap().0, 1261);
    let v = output.vertices()[408];
    assert!(almost_equal(v.x(), -49192.0216620, v.y(), -49115.5024489));
    assert_eq!(v.get_incident_edge().unwrap().0, 1263);
    let v = output.vertices()[409];
    assert!(almost_equal(v.x(), -49124.7621247, v.y(), -49897.3602771));
    assert_eq!(v.get_incident_edge().unwrap().0, 1267);
    let v = output.vertices()[410];
    assert!(almost_equal(v.x(), -49134.6146974, v.y(), -49600.6158501));
    assert_eq!(v.get_incident_edge().unwrap().0, 1269);
    let v = output.vertices()[411];
    assert!(almost_equal(v.x(), -49133.9208229, v.y(), -49600.4276808));
    assert_eq!(v.get_incident_edge().unwrap().0, 1271);
    let v = output.vertices()[412];
    assert!(almost_equal(v.x(), -49144.1832742, v.y(), -49215.9529282));
    assert_eq!(v.get_incident_edge().unwrap().0, 1273);
    let v = output.vertices()[413];
    assert!(almost_equal(v.x(), -49161.1930693, v.y(), -49741.5891089));
    assert_eq!(v.get_incident_edge().unwrap().0, 1275);
    let v = output.vertices()[414];
    assert!(almost_equal(v.x(), -49157.7127016, v.y(), -49746.6592742));
    assert_eq!(v.get_incident_edge().unwrap().0, 1277);
    let v = output.vertices()[415];
    assert!(almost_equal(v.x(), -49299.0000000, v.y(), -48823.5263158));
    assert_eq!(v.get_incident_edge().unwrap().0, 1279);
    let v = output.vertices()[416];
    assert!(almost_equal(v.x(), -49144.7389590, v.y(), -49747.4006309));
    assert_eq!(v.get_incident_edge().unwrap().0, 1285);
    let v = output.vertices()[417];
    assert!(almost_equal(v.x(), -49171.4112291, v.y(), -49095.1323217));
    assert_eq!(v.get_incident_edge().unwrap().0, 1287);
    let v = output.vertices()[418];
    assert!(almost_equal(v.x(), -49095.0920911, v.y(), -49684.4338716));
    assert_eq!(v.get_incident_edge().unwrap().0, 1289);
    let v = output.vertices()[419];
    assert!(almost_equal(v.x(), -49503.8321168, v.y(), -48624.4963504));
    assert_eq!(v.get_incident_edge().unwrap().0, 1291);
    let v = output.vertices()[420];
    assert!(almost_equal(v.x(), -49158.4345048, v.y(), -49492.4920128));
    assert_eq!(v.get_incident_edge().unwrap().0, 1293);
    let v = output.vertices()[421];
    assert!(almost_equal(v.x(), -49157.8666192, v.y(), -49378.5010824));
    assert_eq!(v.get_incident_edge().unwrap().0, 1295);
    let v = output.vertices()[422];
    assert!(almost_equal(v.x(), -49155.5818182, v.y(), -49484.8848485));
    assert_eq!(v.get_incident_edge().unwrap().0, 1297);
    let v = output.vertices()[423];
    assert!(almost_equal(v.x(), -49083.5554598, v.y(), -49307.7688994));
    assert_eq!(v.get_incident_edge().unwrap().0, 1303);
    let v = output.vertices()[424];
    assert!(almost_equal(v.x(), -49109.8417618, v.y(), -49583.3393148));
    assert_eq!(v.get_incident_edge().unwrap().0, 1309);
    let v = output.vertices()[425];
    assert!(almost_equal(v.x(), -49066.9275969, v.y(), -49707.0936180));
    assert_eq!(v.get_incident_edge().unwrap().0, 1311);
    let v = output.vertices()[426];
    assert!(almost_equal(v.x(), -49109.2926829, v.y(), -49583.5528455));
    assert_eq!(v.get_incident_edge().unwrap().0, 1313);
    let v = output.vertices()[427];
    assert!(almost_equal(v.x(), -49137.7602779, v.y(), -49400.3497394));
    assert_eq!(v.get_incident_edge().unwrap().0, 1317);
    let v = output.vertices()[428];
    assert!(almost_equal(v.x(), -49083.1709441, v.y(), -49272.8968520));
    assert_eq!(v.get_incident_edge().unwrap().0, 1319);
    let v = output.vertices()[429];
    assert!(almost_equal(v.x(), -49079.8016123, v.y(), -49771.9922888));
    assert_eq!(v.get_incident_edge().unwrap().0, 1321);
    let v = output.vertices()[430];
    assert!(almost_equal(v.x(), -49088.2989260, v.y(), -49119.4254177));
    assert_eq!(v.get_incident_edge().unwrap().0, 1327);
    let v = output.vertices()[431];
    assert!(almost_equal(v.x(), -49087.1192222, v.y(), -49252.5459253));
    assert_eq!(v.get_incident_edge().unwrap().0, 1331);
    let v = output.vertices()[432];
    assert!(almost_equal(v.x(), -49087.1155047, v.y(), -49595.0679784));
    assert_eq!(v.get_incident_edge().unwrap().0, 1333);
    let v = output.vertices()[433];
    assert!(almost_equal(v.x(), -49065.9863079, v.y(), -49151.8483408));
    assert_eq!(v.get_incident_edge().unwrap().0, 1335);
    let v = output.vertices()[434];
    assert!(almost_equal(v.x(), -49126.5206287, v.y(), -49440.6876228));
    assert_eq!(v.get_incident_edge().unwrap().0, 1341);
    let v = output.vertices()[435];
    assert!(almost_equal(v.x(), -49060.1185994, v.y(), -49246.0230353));
    assert_eq!(v.get_incident_edge().unwrap().0, 1343);
    let v = output.vertices()[436];
    assert!(almost_equal(v.x(), -49126.3745020, v.y(), -49411.0239044));
    assert_eq!(v.get_incident_edge().unwrap().0, 1345);
    let v = output.vertices()[437];
    assert!(almost_equal(v.x(), -49110.4189590, v.y(), -49400.7579348));
    assert_eq!(v.get_incident_edge().unwrap().0, 1347);
    let v = output.vertices()[438];
    assert!(almost_equal(v.x(), -49103.7884545, v.y(), -50003.7127295));
    assert_eq!(v.get_incident_edge().unwrap().0, 1349);
    let v = output.vertices()[439];
    assert!(almost_equal(v.x(), -49020.7839080, v.y(), -49122.7896552));
    assert_eq!(v.get_incident_edge().unwrap().0, 1355);
    let v = output.vertices()[440];
    assert!(almost_equal(v.x(), -49020.1345901, v.y(), -49124.1093011));
    assert_eq!(v.get_incident_edge().unwrap().0, 1359);
    let v = output.vertices()[441];
    assert!(almost_equal(v.x(), -49034.5480212, v.y(), -49714.5974903));
    assert_eq!(v.get_incident_edge().unwrap().0, 1361);
    let v = output.vertices()[442];
    assert!(almost_equal(v.x(), -49023.8044267, v.y(), -49223.1833575));
    assert_eq!(v.get_incident_edge().unwrap().0, 1363);
    let v = output.vertices()[443];
    assert!(almost_equal(v.x(), -49119.5147255, v.y(), -49051.3909829));
    assert_eq!(v.get_incident_edge().unwrap().0, 1365);
    let v = output.vertices()[444];
    assert!(almost_equal(v.x(), -49056.0446429, v.y(), -49482.9732143));
    assert_eq!(v.get_incident_edge().unwrap().0, 1367);
    let v = output.vertices()[445];
    assert!(almost_equal(v.x(), -49027.0905850, v.y(), -49188.9746467));
    assert_eq!(v.get_incident_edge().unwrap().0, 1369);
    let v = output.vertices()[446];
    assert!(almost_equal(v.x(), -49059.7388285, v.y(), -49613.2844539));
    assert_eq!(v.get_incident_edge().unwrap().0, 1371);
    let v = output.vertices()[447];
    assert!(almost_equal(v.x(), -49054.7811271, v.y(), -49794.6671035));
    assert_eq!(v.get_incident_edge().unwrap().0, 1373);
    let v = output.vertices()[448];
    assert!(almost_equal(v.x(), -49014.3076664, v.y(), -49207.8076504));
    assert_eq!(v.get_incident_edge().unwrap().0, 1377);
    let v = output.vertices()[449];
    assert!(almost_equal(v.x(), -49005.5870307, v.y(), -49211.9488055));
    assert_eq!(v.get_incident_edge().unwrap().0, 1381);
    let v = output.vertices()[450];
    assert!(almost_equal(v.x(), -49088.5485222, v.y(), -50020.4282723));
    assert_eq!(v.get_incident_edge().unwrap().0, 1383);
    let v = output.vertices()[451];
    assert!(almost_equal(v.x(), -49040.9460616, v.y(), -49377.9995719));
    assert_eq!(v.get_incident_edge().unwrap().0, 1385);
    let v = output.vertices()[452];
    assert!(almost_equal(v.x(), -48981.8818898, v.y(), -49196.7097863));
    assert_eq!(v.get_incident_edge().unwrap().0, 1387);
    let v = output.vertices()[453];
    assert!(almost_equal(v.x(), -48988.5438394, v.y(), -49221.9742121));
    assert_eq!(v.get_incident_edge().unwrap().0, 1391);
    let v = output.vertices()[454];
    assert!(almost_equal(v.x(), -48988.2547170, v.y(), -49222.5283019));
    assert_eq!(v.get_incident_edge().unwrap().0, 1393);
    let v = output.vertices()[455];
    assert!(almost_equal(v.x(), -49008.3260838, v.y(), -49498.6934898));
    assert_eq!(v.get_incident_edge().unwrap().0, 1395);
    let v = output.vertices()[456];
    assert!(almost_equal(v.x(), -49006.0447346, v.y(), -49360.2344820));
    assert_eq!(v.get_incident_edge().unwrap().0, 1397);
    let v = output.vertices()[457];
    assert!(almost_equal(v.x(), -49004.1400770, v.y(), -49358.8868470));
    assert_eq!(v.get_incident_edge().unwrap().0, 1399);
    let v = output.vertices()[458];
    assert!(almost_equal(v.x(), -48999.5906671, v.y(), -49499.6712601));
    assert_eq!(v.get_incident_edge().unwrap().0, 1401);
    let v = output.vertices()[459];
    assert!(almost_equal(v.x(), -49085.1579343, v.y(), -48982.3372343));
    assert_eq!(v.get_incident_edge().unwrap().0, 1403);
    let v = output.vertices()[460];
    assert!(almost_equal(v.x(), -49006.9876684, v.y(), -49634.0758869));
    assert_eq!(v.get_incident_edge().unwrap().0, 1405);
    let v = output.vertices()[461];
    assert!(almost_equal(v.x(), -48970.2364716, v.y(), -49257.6638803));
    assert_eq!(v.get_incident_edge().unwrap().0, 1407);
    let v = output.vertices()[462];
    assert!(almost_equal(v.x(), -48966.3643916, v.y(), -49507.0517512));
    assert_eq!(v.get_incident_edge().unwrap().0, 1409);
    let v = output.vertices()[463];
    assert!(almost_equal(v.x(), -48922.6621998, v.y(), -49325.6960376));
    assert_eq!(v.get_incident_edge().unwrap().0, 1411);
    let v = output.vertices()[464];
    assert!(almost_equal(v.x(), -48899.6566186, v.y(), -49867.9986894));
    assert_eq!(v.get_incident_edge().unwrap().0, 1413);
    let v = output.vertices()[465];
    assert!(almost_equal(v.x(), -48845.9915010, v.y(), -50235.8674630));
    assert_eq!(v.get_incident_edge().unwrap().0, 1415);
    let v = output.vertices()[466];
    assert!(almost_equal(v.x(), -48732.6811071, v.y(), -49686.5168472));
    assert_eq!(v.get_incident_edge().unwrap().0, 1417);
    let v = output.vertices()[467];
    assert!(almost_equal(v.x(), -48774.7086521, v.y(), -50297.7210124));
    assert_eq!(v.get_incident_edge().unwrap().0, 1419);
    let v = output.vertices()[468];
    assert!(almost_equal(v.x(), -49290.3687196, v.y(), -51849.2487844));
    assert_eq!(v.get_incident_edge().unwrap().0, 1421);
    let v = output.vertices()[469];
    assert!(almost_equal(v.x(), -48223.2421239, v.y(), -49749.1294596));
    assert_eq!(v.get_incident_edge().unwrap().0, 1423);
    let v = output.vertices()[470];
    assert!(almost_equal(v.x(), -49399.8917660, v.y(), -46414.0315560));
    assert_eq!(v.get_incident_edge().unwrap().0, 1425);
    let v = output.vertices()[471];
    assert!(almost_equal(v.x(), -49299.0000000, v.y(), -45344.1981132));
    assert_eq!(v.get_incident_edge().unwrap().0, 1427);
    Ok(())
}
