use boostvoronoi::builder::{to_points, to_segments, Builder};
use boostvoronoi::diagram as VD;
use boostvoronoi::{BvError, Line, Point};

type I1 = i32;
type F1 = f64;

#[allow(dead_code)]
fn almost_equal(x1: F1, x2: F1, y1: F1, y2: F1) -> bool {
    let delta = 0.00001;
    assert!(F1::abs(x1 - x2) < delta, "{} != {}", x1, x2);
    assert!(F1::abs(y1 - y2) < delta, "{} != {}", y1, y2);

    (F1::abs(x1 - x2) < delta) && (F1::abs(y1 - y2) < delta)
}

fn retrieve_point<T>(
    point_data_: &Vec<Point<T>>,
    segment_data_: &Vec<Line<T>>,
    source: (VD::SourceIndex, VD::SourceCategory),
) -> Point<T>
where
    T: VD::InputType,
{
    match source.1 {
        VD::SourceCategory::SinglePoint => point_data_[source.0],
        VD::SourceCategory::SegmentStart => segment_data_[source.0 - point_data_.len()].start,
        VD::SourceCategory::Segment | VD::SourceCategory::SegmentEnd => {
            segment_data_[source.0 - point_data_.len()].end
        }
    }
}

//#[ignore]
#[test]
/// The rust logo
fn large_segment_1() -> Result<(), BvError> {
    let (output, _v, _s) = {
        let points: [[I1; 2]; 45] = [
            [303, 108],
            [180, 257],
            [115, 405],
            [226, 536],
            [177, 599],
            [43, 709],
            [127, 740],
            [158, 683],
            [593, 759],
            [583, 672],
            [587, 543],
            [514, 503],
            [408, 543],
            [401, 642],
            [580, 380],
            [308, 304],
            [260, 343],
            [258, 240],
            [344, 238],
            [346, 189],
            [198, 150],
            [238, 59],
            [679, 37],
            [727, 134],
            [733, 276],
            [679, 327],
            [698, 404],
            [744, 544],
            [764, 673],
            [710, 770],
            [611, 687],
            [298, 536],
            [244, 675],
            [313, 618],
            [433, 760],
            [301, 773],
            [127, 758],
            [40, 644],
            [612, 105],
            [743, 37],
            [581, 26],
            [402, 20],
            [318, 45],
            [130, 102],
            [98, 141],
        ];
        let segments: [[I1; 4]; 51] = [
            [200, 200, 200, 400],
            [200, 400, 400, 400],
            [400, 400, 400, 200],
            [400, 200, 200, 200],
            [529, 242, 367, 107],
            [667, 431, 464, 554],
            [464, 554, 230, 588],
            [230, 588, 88, 464],
            [88, 464, 80, 236],
            [80, 236, 178, 97],
            [178, 97, 463, 56],
            [463, 56, 670, 175],
            [670, 175, 732, 346],
            [732, 346, 735, 479],
            [735, 479, 512, 643],
            [512, 643, 257, 710],
            [257, 710, 100, 615],
            [100, 615, 36, 470],
            [36, 470, 53, 195],
            [53, 195, 82, 83],
            [82, 83, 211, 30],
            [379, 35, 614, 55],
            [759, 140, 784, 390],
            [734, 594, 678, 686],
            [485, 742, 203, 745],
            [103, 724, 12, 537],
            [22, 425, 38, 45],
            [43, 25, 308, 17],
            [512, 691, 629, 758],
            [629, 758, 643, 601],
            [499, 455, 618, 281],
            [618, 281, 612, 209],
            [612, 209, 486, 125],
            [462, 458, 361, 494],
            [215, 498, 148, 470],
            [453, 233, 494, 371],
            [494, 371, 560, 262],
            [560, 262, 563, 200],
            [563, 200, 451, 141],
            [451, 141, 421, 82],
            [421, 82, 243, 111],
            [243, 111, 145, 187],
            [145, 187, 144, 319],
            [144, 319, 177, 442],
            [177, 442, 266, 484],
            [266, 484, 336, 541],
            [336, 541, 433, 497],
            [433, 497, 525, 467],
            [525, 467, 594, 427],
            [594, 427, 617, 342],
            [617, 342, 675, 292],
        ];

        let _v = to_points::<I1, I1>(&points);
        let _s = to_segments::<I1, I1>(&segments);

        let mut vb = Builder::<I1, F1>::default();
        vb.with_vertices(_v.iter()).expect("large_segment_1");
        vb.with_segments(_s.iter()).expect("large_segment_1");
        (vb.construct().expect("large_segment_1"), _v, _s)
    };
    assert_eq!(output.cells().len(), 161);
    let cell = output.cells()[0].get();
    assert_eq!(cell.get_id(), 0);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 12, y: 537 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[1].get();
    assert_eq!(cell.get_id(), 1);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 12, y: 537 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[2].get();
    assert_eq!(cell.get_id(), 2);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 22, y: 425 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[3].get();
    assert_eq!(cell.get_id(), 3);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 38, y: 45 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[4].get();
    assert_eq!(cell.get_id(), 4);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 36, y: 470 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[5].get();
    assert_eq!(cell.get_id(), 5);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 36, y: 470 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[6].get();
    assert_eq!(cell.get_id(), 6);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 53, y: 195 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[7].get();
    assert_eq!(cell.get_id(), 7);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 38, y: 45 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[8].get();
    assert_eq!(cell.get_id(), 8);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 40, y: 644 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[9].get();
    assert_eq!(cell.get_id(), 9);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 43, y: 25 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[10].get();
    assert_eq!(cell.get_id(), 10);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 43, y: 709 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[11].get();
    assert_eq!(cell.get_id(), 11);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 308, y: 17 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[12].get();
    assert_eq!(cell.get_id(), 12);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 53, y: 195 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[13].get();
    assert_eq!(cell.get_id(), 13);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 82, y: 83 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[14].get();
    assert_eq!(cell.get_id(), 14);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 80, y: 236 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[15].get();
    assert_eq!(cell.get_id(), 15);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 80, y: 236 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[16].get();
    assert_eq!(cell.get_id(), 16);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 178, y: 97 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[17].get();
    assert_eq!(cell.get_id(), 17);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 82, y: 83 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[18].get();
    assert_eq!(cell.get_id(), 18);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 211, y: 30 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[19].get();
    assert_eq!(cell.get_id(), 19);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 88, y: 464 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[20].get();
    assert_eq!(cell.get_id(), 20);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 88, y: 464 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[21].get();
    assert_eq!(cell.get_id(), 21);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 98, y: 141 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[22].get();
    assert_eq!(cell.get_id(), 22);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 100, y: 615 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[23].get();
    assert_eq!(cell.get_id(), 23);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 100, y: 615 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[24].get();
    assert_eq!(cell.get_id(), 24);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 103, y: 724 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[25].get();
    assert_eq!(cell.get_id(), 25);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 115, y: 405 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[26].get();
    assert_eq!(cell.get_id(), 26);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 127, y: 740 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[27].get();
    assert_eq!(cell.get_id(), 27);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 127, y: 758 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[28].get();
    assert_eq!(cell.get_id(), 28);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 130, y: 102 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[29].get();
    assert_eq!(cell.get_id(), 29);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 144, y: 319 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[30].get();
    assert_eq!(cell.get_id(), 30);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 177, y: 442 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[31].get();
    assert_eq!(cell.get_id(), 31);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 144, y: 319 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[32].get();
    assert_eq!(cell.get_id(), 32);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 145, y: 187 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[33].get();
    assert_eq!(cell.get_id(), 33);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 145, y: 187 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[34].get();
    assert_eq!(cell.get_id(), 34);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 148, y: 470 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[35].get();
    assert_eq!(cell.get_id(), 35);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 148, y: 470 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[36].get();
    assert_eq!(cell.get_id(), 36);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 158, y: 683 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[37].get();
    assert_eq!(cell.get_id(), 37);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 177, y: 442 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[38].get();
    assert_eq!(cell.get_id(), 38);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 177, y: 599 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[39].get();
    assert_eq!(cell.get_id(), 39);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 266, y: 484 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[40].get();
    assert_eq!(cell.get_id(), 40);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 178, y: 97 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[41].get();
    assert_eq!(cell.get_id(), 41);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 463, y: 56 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[42].get();
    assert_eq!(cell.get_id(), 42);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 180, y: 257 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[43].get();
    assert_eq!(cell.get_id(), 43);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 198, y: 150 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[44].get();
    assert_eq!(cell.get_id(), 44);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 200, y: 200 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[45].get();
    assert_eq!(cell.get_id(), 45);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 200, y: 400 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[46].get();
    assert_eq!(cell.get_id(), 46);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 200, y: 400 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[47].get();
    assert_eq!(cell.get_id(), 47);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 200, y: 200 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[48].get();
    assert_eq!(cell.get_id(), 48);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 400, y: 400 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[49].get();
    assert_eq!(cell.get_id(), 49);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 203, y: 745 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[50].get();
    assert_eq!(cell.get_id(), 50);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 203, y: 745 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[51].get();
    assert_eq!(cell.get_id(), 51);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 211, y: 30 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[52].get();
    assert_eq!(cell.get_id(), 52);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 215, y: 498 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[53].get();
    assert_eq!(cell.get_id(), 53);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 226, y: 536 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[54].get();
    assert_eq!(cell.get_id(), 54);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 230, y: 588 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[55].get();
    assert_eq!(cell.get_id(), 55);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 230, y: 588 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[56].get();
    assert_eq!(cell.get_id(), 56);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 238, y: 59 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[57].get();
    assert_eq!(cell.get_id(), 57);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 243, y: 111 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[58].get();
    assert_eq!(cell.get_id(), 58);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 243, y: 111 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[59].get();
    assert_eq!(cell.get_id(), 59);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 244, y: 675 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[60].get();
    assert_eq!(cell.get_id(), 60);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 257, y: 710 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[61].get();
    assert_eq!(cell.get_id(), 61);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 257, y: 710 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[62].get();
    assert_eq!(cell.get_id(), 62);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 258, y: 240 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[63].get();
    assert_eq!(cell.get_id(), 63);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 260, y: 343 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[64].get();
    assert_eq!(cell.get_id(), 64);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 266, y: 484 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[65].get();
    assert_eq!(cell.get_id(), 65);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 336, y: 541 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[66].get();
    assert_eq!(cell.get_id(), 66);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 298, y: 536 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[67].get();
    assert_eq!(cell.get_id(), 67);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 301, y: 773 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[68].get();
    assert_eq!(cell.get_id(), 68);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 303, y: 108 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[69].get();
    assert_eq!(cell.get_id(), 69);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 308, y: 17 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[70].get();
    assert_eq!(cell.get_id(), 70);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 308, y: 304 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[71].get();
    assert_eq!(cell.get_id(), 71);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 313, y: 618 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[72].get();
    assert_eq!(cell.get_id(), 72);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 318, y: 45 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[73].get();
    assert_eq!(cell.get_id(), 73);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 336, y: 541 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[74].get();
    assert_eq!(cell.get_id(), 74);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 433, y: 497 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[75].get();
    assert_eq!(cell.get_id(), 75);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 344, y: 238 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[76].get();
    assert_eq!(cell.get_id(), 76);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 346, y: 189 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[77].get();
    assert_eq!(cell.get_id(), 77);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 361, y: 494 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[78].get();
    assert_eq!(cell.get_id(), 78);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 361, y: 494 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[79].get();
    assert_eq!(cell.get_id(), 79);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 367, y: 107 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[80].get();
    assert_eq!(cell.get_id(), 80);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 367, y: 107 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[81].get();
    assert_eq!(cell.get_id(), 81);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 379, y: 35 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[82].get();
    assert_eq!(cell.get_id(), 82);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 614, y: 55 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[83].get();
    assert_eq!(cell.get_id(), 83);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 400, y: 200 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[84].get();
    assert_eq!(cell.get_id(), 84);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 400, y: 200 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[85].get();
    assert_eq!(cell.get_id(), 85);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 400, y: 400 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[86].get();
    assert_eq!(cell.get_id(), 86);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 401, y: 642 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[87].get();
    assert_eq!(cell.get_id(), 87);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 402, y: 20 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[88].get();
    assert_eq!(cell.get_id(), 88);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 408, y: 543 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[89].get();
    assert_eq!(cell.get_id(), 89);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 421, y: 82 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[90].get();
    assert_eq!(cell.get_id(), 90);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 421, y: 82 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[91].get();
    assert_eq!(cell.get_id(), 91);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 433, y: 497 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[92].get();
    assert_eq!(cell.get_id(), 92);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 433, y: 760 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[93].get();
    assert_eq!(cell.get_id(), 93);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 525, y: 467 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[94].get();
    assert_eq!(cell.get_id(), 94);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 451, y: 141 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[95].get();
    assert_eq!(cell.get_id(), 95);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 451, y: 141 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[96].get();
    assert_eq!(cell.get_id(), 96);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 453, y: 233 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[97].get();
    assert_eq!(cell.get_id(), 97);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 494, y: 371 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[98].get();
    assert_eq!(cell.get_id(), 98);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 462, y: 458 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[99].get();
    assert_eq!(cell.get_id(), 99);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 463, y: 56 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[100].get();
    assert_eq!(cell.get_id(), 100);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 670, y: 175 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[101].get();
    assert_eq!(cell.get_id(), 101);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 464, y: 554 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[102].get();
    assert_eq!(cell.get_id(), 102);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 464, y: 554 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[103].get();
    assert_eq!(cell.get_id(), 103);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 485, y: 742 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[104].get();
    assert_eq!(cell.get_id(), 104);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 486, y: 125 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[105].get();
    assert_eq!(cell.get_id(), 105);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 486, y: 125 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[106].get();
    assert_eq!(cell.get_id(), 106);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 494, y: 371 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[107].get();
    assert_eq!(cell.get_id(), 107);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 560, y: 262 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[108].get();
    assert_eq!(cell.get_id(), 108);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 499, y: 455 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[109].get();
    assert_eq!(cell.get_id(), 109);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 618, y: 281 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[110].get();
    assert_eq!(cell.get_id(), 110);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 512, y: 643 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[111].get();
    assert_eq!(cell.get_id(), 111);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 512, y: 691 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[112].get();
    assert_eq!(cell.get_id(), 112);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 512, y: 643 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[113].get();
    assert_eq!(cell.get_id(), 113);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 629, y: 758 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[114].get();
    assert_eq!(cell.get_id(), 114);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 514, y: 503 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[115].get();
    assert_eq!(cell.get_id(), 115);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 525, y: 467 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[116].get();
    assert_eq!(cell.get_id(), 116);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 594, y: 427 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[117].get();
    assert_eq!(cell.get_id(), 117);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 529, y: 242 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[118].get();
    assert_eq!(cell.get_id(), 118);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 560, y: 262 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[119].get();
    assert_eq!(cell.get_id(), 119);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 563, y: 200 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[120].get();
    assert_eq!(cell.get_id(), 120);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 563, y: 200 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[121].get();
    assert_eq!(cell.get_id(), 121);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 580, y: 380 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[122].get();
    assert_eq!(cell.get_id(), 122);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 581, y: 26 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[123].get();
    assert_eq!(cell.get_id(), 123);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 583, y: 672 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[124].get();
    assert_eq!(cell.get_id(), 124);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 587, y: 543 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[125].get();
    assert_eq!(cell.get_id(), 125);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 593, y: 759 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[126].get();
    assert_eq!(cell.get_id(), 126);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 594, y: 427 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[127].get();
    assert_eq!(cell.get_id(), 127);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 617, y: 342 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[128].get();
    assert_eq!(cell.get_id(), 128);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 611, y: 687 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[129].get();
    assert_eq!(cell.get_id(), 129);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 612, y: 105 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[130].get();
    assert_eq!(cell.get_id(), 130);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 612, y: 209 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[131].get();
    assert_eq!(cell.get_id(), 131);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 612, y: 209 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[132].get();
    assert_eq!(cell.get_id(), 132);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 614, y: 55 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[133].get();
    assert_eq!(cell.get_id(), 133);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 617, y: 342 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[134].get();
    assert_eq!(cell.get_id(), 134);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 675, y: 292 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[135].get();
    assert_eq!(cell.get_id(), 135);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 618, y: 281 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[136].get();
    assert_eq!(cell.get_id(), 136);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 629, y: 758 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[137].get();
    assert_eq!(cell.get_id(), 137);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 643, y: 601 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[138].get();
    assert_eq!(cell.get_id(), 138);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 643, y: 601 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[139].get();
    assert_eq!(cell.get_id(), 139);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 667, y: 431 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[140].get();
    assert_eq!(cell.get_id(), 140);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 670, y: 175 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[141].get();
    assert_eq!(cell.get_id(), 141);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 732, y: 346 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[142].get();
    assert_eq!(cell.get_id(), 142);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 675, y: 292 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[143].get();
    assert_eq!(cell.get_id(), 143);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 678, y: 686 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[144].get();
    assert_eq!(cell.get_id(), 144);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 678, y: 686 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[145].get();
    assert_eq!(cell.get_id(), 145);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 679, y: 37 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[146].get();
    assert_eq!(cell.get_id(), 146);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 679, y: 327 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[147].get();
    assert_eq!(cell.get_id(), 147);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 698, y: 404 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[148].get();
    assert_eq!(cell.get_id(), 148);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 710, y: 770 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[149].get();
    assert_eq!(cell.get_id(), 149);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 727, y: 134 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[150].get();
    assert_eq!(cell.get_id(), 150);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 732, y: 346 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[151].get();
    assert_eq!(cell.get_id(), 151);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 735, y: 479 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[152].get();
    assert_eq!(cell.get_id(), 152);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 733, y: 276 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[153].get();
    assert_eq!(cell.get_id(), 153);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 734, y: 594 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[154].get();
    assert_eq!(cell.get_id(), 154);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 735, y: 479 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[155].get();
    assert_eq!(cell.get_id(), 155);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 743, y: 37 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[156].get();
    assert_eq!(cell.get_id(), 156);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 744, y: 544 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[157].get();
    assert_eq!(cell.get_id(), 157);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 759, y: 140 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[158].get();
    assert_eq!(cell.get_id(), 158);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::Segment);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 784, y: 390 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[159].get();
    assert_eq!(cell.get_id(), 159);
    let (source_index, cat) = cell.source_index_2();
    assert_eq!(cat, VD::SourceCategory::SinglePoint);
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 764, y: 673 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[160].get();
    assert_eq!(cell.get_id(), 160);
    let (source_index, cat) = cell.source_index_2();
    let p = retrieve_point(&_v, &_s, (source_index, cat));
    assert_eq!(p, Point { x: 784, y: 390 });
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    assert_eq!(output.vertices().len(), 304);
    assert_eq!(output.edges().len(), 928);
    let v = output.vertices()[0].get();
    assert!(almost_equal(v.x(), -1174.3944223, v.y(), 374.6254980));
    assert_eq!(v.get_incident_edge().unwrap().0, 7);
    let v = output.vertices()[1].get();
    assert!(almost_equal(v.x(), 36.0000000, v.y(), 470.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 15);
    let v = output.vertices()[2].get();
    assert!(almost_equal(v.x(), -57.3429138, v.y(), 474.3622398));
    assert_eq!(v.get_incident_edge().unwrap().0, 17);
    let v = output.vertices()[3].get();
    assert!(almost_equal(v.x(), -30.1728061, v.y(), 465.9093174));
    assert_eq!(v.get_incident_edge().unwrap().0, 19);
    let v = output.vertices()[4].get();
    assert!(almost_equal(v.x(), 30.3683167, v.y(), 425.3523502));
    assert_eq!(v.get_incident_edge().unwrap().0, 23);
    let v = output.vertices()[5].get();
    assert!(almost_equal(v.x(), -11.2724640, v.y(), 490.8650875));
    assert_eq!(v.get_incident_edge().unwrap().0, 27);
    let v = output.vertices()[6].get();
    assert!(almost_equal(v.x(), -242.1180849, v.y(), 660.6617418));
    assert_eq!(v.get_incident_edge().unwrap().0, 29);
    let v = output.vertices()[7].get();
    assert!(almost_equal(v.x(), -376.3385590, v.y(), 695.7848566));
    assert_eq!(v.get_incident_edge().unwrap().0, 37);
    let v = output.vertices()[8].get();
    assert!(almost_equal(v.x(), 53.0000000, v.y(), 195.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 43);
    let v = output.vertices()[9].get();
    assert!(almost_equal(v.x(), 42.3708015, v.y(), 194.3429223));
    assert_eq!(v.get_incident_edge().unwrap().0, 45);
    let v = output.vertices()[10].get();
    assert!(almost_equal(v.x(), 42.5755898, v.y(), 192.3008224));
    assert_eq!(v.get_incident_edge().unwrap().0, 47);
    let v = output.vertices()[11].get();
    assert!(almost_equal(v.x(), 43.3231939, v.y(), 35.7057985));
    assert_eq!(v.get_incident_edge().unwrap().0, 49);
    let v = output.vertices()[12].get();
    assert!(almost_equal(v.x(), -22930.4262948, v.y(), -922.0916335));
    assert_eq!(v.get_incident_edge().unwrap().0, 51);
    let v = output.vertices()[13].get();
    assert!(almost_equal(v.x(), 34.0424487, v.y(), 526.2734608));
    assert_eq!(v.get_incident_edge().unwrap().0, 53);
    let v = output.vertices()[14].get();
    assert!(almost_equal(v.x(), 43.5817605, v.y(), 676.4039187));
    assert_eq!(v.get_incident_edge().unwrap().0, 55);
    let v = output.vertices()[15].get();
    assert!(almost_equal(v.x(), 80.0000000, v.y(), 236.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 63);
    let v = output.vertices()[16].get();
    assert!(almost_equal(v.x(), 65.2353585, v.y(), 236.5180576));
    assert_eq!(v.get_incident_edge().unwrap().0, 65);
    let v = output.vertices()[17].get();
    assert!(almost_equal(v.x(), 59.3649404, v.y(), 45.8995764));
    assert_eq!(v.get_incident_edge().unwrap().0, 67);
    let v = output.vertices()[18].get();
    assert!(almost_equal(v.x(), 82.0000000, v.y(), 83.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 73);
    let v = output.vertices()[19].get();
    assert!(almost_equal(v.x(), 59.6994840, v.y(), 77.2257592));
    assert_eq!(v.get_incident_edge().unwrap().0, 75);
    let v = output.vertices()[20].get();
    assert!(almost_equal(v.x(), 66.9861495, v.y(), 226.8247673));
    assert_eq!(v.get_incident_edge().unwrap().0, 77);
    let v = output.vertices()[21].get();
    assert!(almost_equal(v.x(), 88.0000000, v.y(), 464.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 83);
    let v = output.vertices()[22].get();
    assert!(almost_equal(v.x(), 62.1900542, v.y(), 464.9056121));
    assert_eq!(v.get_incident_edge().unwrap().0, 85);
    let v = output.vertices()[23].get();
    assert!(almost_equal(v.x(), 62.1862043, v.y(), 465.3170750));
    assert_eq!(v.get_incident_edge().unwrap().0, 87);
    let v = output.vertices()[24].get();
    assert!(almost_equal(v.x(), 70.7460129, v.y(), 483.7585982));
    assert_eq!(v.get_incident_edge().unwrap().0, 89);
    let v = output.vertices()[25].get();
    assert!(almost_equal(v.x(), 68.5231008, v.y(), 55.1673809));
    assert_eq!(v.get_incident_edge().unwrap().0, 93);
    let v = output.vertices()[26].get();
    assert!(almost_equal(v.x(), 100.0000000, v.y(), 615.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 99);
    let v = output.vertices()[27].get();
    assert!(almost_equal(v.x(), 76.8384688, v.y(), 198.7871843));
    assert_eq!(v.get_incident_edge().unwrap().0, 101);
    let v = output.vertices()[28].get();
    assert!(almost_equal(v.x(), 70.3891562, v.y(), 54.7396443));
    assert_eq!(v.get_incident_edge().unwrap().0, 103);
    let v = output.vertices()[29].get();
    assert!(almost_equal(v.x(), 79.4098241, v.y(), 624.0880777));
    assert_eq!(v.get_incident_edge().unwrap().0, 105);
    let v = output.vertices()[30].get();
    assert!(almost_equal(v.x(), 66.7100457, v.y(), 741.6598174));
    assert_eq!(v.get_incident_edge().unwrap().0, 109);
    let v = output.vertices()[31].get();
    assert!(almost_equal(v.x(), 87.3387120, v.y(), 635.9244443));
    assert_eq!(v.get_incident_edge().unwrap().0, 111);
    let v = output.vertices()[32].get();
    assert!(almost_equal(v.x(), 90.2866683, v.y(), 169.7504629));
    assert_eq!(v.get_incident_edge().unwrap().0, 115);
    let v = output.vertices()[33].get();
    assert!(almost_equal(v.x(), 103.6666667, v.y(), 749.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 121);
    let v = output.vertices()[34].get();
    assert!(almost_equal(v.x(), 103.9513759, v.y(), 102.1430840));
    assert_eq!(v.get_incident_edge().unwrap().0, 125);
    let v = output.vertices()[35].get();
    assert!(almost_equal(v.x(), 56.5625000, v.y(), 782.2500000));
    assert_eq!(v.get_incident_edge().unwrap().0, 127);
    let v = output.vertices()[36].get();
    assert!(almost_equal(v.x(), 103.6928324, v.y(), 113.0428369));
    assert_eq!(v.get_incident_edge().unwrap().0, 129);
    let v = output.vertices()[37].get();
    assert!(almost_equal(v.x(), 144.0000000, v.y(), 319.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 137);
    let v = output.vertices()[38].get();
    assert!(almost_equal(v.x(), 113.4619164, v.y(), 318.7686509));
    assert_eq!(v.get_incident_edge().unwrap().0, 139);
    let v = output.vertices()[39].get();
    assert!(almost_equal(v.x(), 112.4568064, v.y(), 245.6682878));
    assert_eq!(v.get_incident_edge().unwrap().0, 141);
    let v = output.vertices()[40].get();
    assert!(almost_equal(v.x(), 145.0000000, v.y(), 187.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 147);
    let v = output.vertices()[41].get();
    assert!(almost_equal(v.x(), 131.3372641, v.y(), 186.8964944));
    assert_eq!(v.get_incident_edge().unwrap().0, 149);
    let v = output.vertices()[42].get();
    assert!(almost_equal(v.x(), 114.1343309, v.y(), 327.0127405));
    assert_eq!(v.get_incident_edge().unwrap().0, 151);
    let v = output.vertices()[43].get();
    assert!(almost_equal(v.x(), 122.3376920, v.y(), 128.3411832));
    assert_eq!(v.get_incident_edge().unwrap().0, 157);
    let v = output.vertices()[44].get();
    assert!(almost_equal(v.x(), 137.2028817, v.y(), 176.9458211));
    assert_eq!(v.get_incident_edge().unwrap().0, 159);
    let v = output.vertices()[45].get();
    assert!(almost_equal(v.x(), 121.7452809, v.y(), 447.9473901));
    assert_eq!(v.get_incident_edge().unwrap().0, 161);
    let v = output.vertices()[46].get();
    assert!(almost_equal(v.x(), 120.5132924, v.y(), 369.6243596));
    assert_eq!(v.get_incident_edge().unwrap().0, 163);
    let v = output.vertices()[47].get();
    assert!(almost_equal(v.x(), 128.6569921, v.y(), 711.5145119));
    assert_eq!(v.get_incident_edge().unwrap().0, 165);
    let v = output.vertices()[48].get();
    assert!(almost_equal(v.x(), 141.0848907, v.y(), 486.5468686));
    assert_eq!(v.get_incident_edge().unwrap().0, 169);
    let v = output.vertices()[49].get();
    assert!(almost_equal(v.x(), 120.8761604, v.y(), 672.6712790));
    assert_eq!(v.get_incident_edge().unwrap().0, 171);
    let v = output.vertices()[50].get();
    assert!(almost_equal(v.x(), 124.5018028, v.y(), 441.0529309));
    assert_eq!(v.get_incident_edge().unwrap().0, 173);
    let v = output.vertices()[51].get();
    assert!(almost_equal(v.x(), 132.5058123, v.y(), 706.0645646));
    assert_eq!(v.get_incident_edge().unwrap().0, 175);
    let v = output.vertices()[52].get();
    assert!(almost_equal(v.x(), 136.8261871, v.y(), 434.7959358));
    assert_eq!(v.get_incident_edge().unwrap().0, 177);
    let v = output.vertices()[53].get();
    assert!(almost_equal(v.x(), 177.0000000, v.y(), 442.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 185);
    let v = output.vertices()[54].get();
    assert!(almost_equal(v.x(), 154.7471610, v.y(), 447.9702739));
    assert_eq!(v.get_incident_edge().unwrap().0, 187);
    let v = output.vertices()[55].get();
    assert!(almost_equal(v.x(), 178.0000000, v.y(), 97.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 193);
    let v = output.vertices()[56].get();
    assert!(almost_equal(v.x(), 156.4635417, v.y(), 449.7479539));
    assert_eq!(v.get_incident_edge().unwrap().0, 195);
    let v = output.vertices()[57].get();
    assert!(almost_equal(v.x(), 135.0400623, v.y(), 584.1961079));
    assert_eq!(v.get_incident_edge().unwrap().0, 197);
    let v = output.vertices()[58].get();
    assert!(almost_equal(v.x(), 152.2164348, v.y(), 84.6357643));
    assert_eq!(v.get_incident_edge().unwrap().0, 201);
    let v = output.vertices()[59].get();
    assert!(almost_equal(v.x(), 156.0857605, v.y(), 81.5496728));
    assert_eq!(v.get_incident_edge().unwrap().0, 203);
    let v = output.vertices()[60].get();
    assert!(almost_equal(v.x(), 135.3473554, v.y(), 571.5629890));
    assert_eq!(v.get_incident_edge().unwrap().0, 205);
    let v = output.vertices()[61].get();
    assert!(almost_equal(v.x(), 169.0970276, v.y(), 458.7467748));
    assert_eq!(v.get_incident_edge().unwrap().0, 207);
    let v = output.vertices()[62].get();
    assert!(almost_equal(v.x(), 174.4263632, v.y(), 72.1588663));
    assert_eq!(v.get_incident_edge().unwrap().0, 211);
    let v = output.vertices()[63].get();
    assert!(almost_equal(v.x(), 172.7321989, v.y(), 200.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 217);
    let v = output.vertices()[64].get();
    assert!(almost_equal(v.x(), 172.4489632, v.y(), 200.5780786));
    assert_eq!(v.get_incident_edge().unwrap().0, 219);
    let v = output.vertices()[65].get();
    assert!(almost_equal(v.x(), 172.3359319, v.y(), 230.4187524));
    assert_eq!(v.get_incident_edge().unwrap().0, 221);
    let v = output.vertices()[66].get();
    assert!(almost_equal(v.x(), 172.1339878, v.y(), 283.7327605));
    assert_eq!(v.get_incident_edge().unwrap().0, 223);
    let v = output.vertices()[67].get();
    assert!(almost_equal(v.x(), 172.0139802, v.y(), 315.4152249));
    assert_eq!(v.get_incident_edge().unwrap().0, 225);
    let v = output.vertices()[68].get();
    assert!(almost_equal(v.x(), 183.1635651, v.y(), 400.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 229);
    let v = output.vertices()[69].get();
    assert!(almost_equal(v.x(), 200.0000000, v.y(), 200.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 233);
    let v = output.vertices()[70].get();
    assert!(almost_equal(v.x(), 200.0000000, v.y(), 400.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 237);
    let v = output.vertices()[71].get();
    assert!(almost_equal(v.x(), 164.5723684, v.y(), 749.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 243);
    let v = output.vertices()[72].get();
    assert!(almost_equal(v.x(), 166.1917762, v.y(), 724.3850011));
    assert_eq!(v.get_incident_edge().unwrap().0, 245);
    let v = output.vertices()[73].get();
    assert!(almost_equal(v.x(), 209.0238909, v.y(), 25.1902251));
    assert_eq!(v.get_incident_edge().unwrap().0, 249);
    let v = output.vertices()[74].get();
    assert!(almost_equal(v.x(), 191.9065740, v.y(), 119.6929455));
    assert_eq!(v.get_incident_edge().unwrap().0, 253);
    let v = output.vertices()[75].get();
    assert!(almost_equal(v.x(), 197.8607829, v.y(), 424.5715117));
    assert_eq!(v.get_incident_edge().unwrap().0, 255);
    let v = output.vertices()[76].get();
    assert!(almost_equal(v.x(), 184.3835006, v.y(), 711.1813302));
    assert_eq!(v.get_incident_edge().unwrap().0, 257);
    let v = output.vertices()[77].get();
    assert!(almost_equal(v.x(), 200.0000000, v.y(), 175.4342171));
    assert_eq!(v.get_incident_edge().unwrap().0, 259);
    let v = output.vertices()[78].get();
    assert!(almost_equal(v.x(), 200.0000000, v.y(), 425.0997269));
    assert_eq!(v.get_incident_edge().unwrap().0, 261);
    let v = output.vertices()[79].get();
    assert!(almost_equal(v.x(), 201.6597254, v.y(), 174.8660641));
    assert_eq!(v.get_incident_edge().unwrap().0, 265);
    let v = output.vertices()[80].get();
    assert!(almost_equal(v.x(), 199.9344795, v.y(), 523.1841408));
    assert_eq!(v.get_incident_edge().unwrap().0, 267);
    let v = output.vertices()[81].get();
    assert!(almost_equal(v.x(), 230.0000000, v.y(), 588.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 273);
    let v = output.vertices()[82].get();
    assert!(almost_equal(v.x(), 205.2100089, v.y(), 521.4260500));
    assert_eq!(v.get_incident_edge().unwrap().0, 275);
    let v = output.vertices()[83].get();
    assert!(almost_equal(v.x(), 202.6670680, v.y(), 713.7043925));
    assert_eq!(v.get_incident_edge().unwrap().0, 277);
    let v = output.vertices()[84].get();
    assert!(almost_equal(v.x(), 208.7641689, v.y(), 62.7902787));
    assert_eq!(v.get_incident_edge().unwrap().0, 281);
    let v = output.vertices()[85].get();
    assert!(almost_equal(v.x(), 221.6406921, v.y(), 482.1097726));
    assert_eq!(v.get_incident_edge().unwrap().0, 283);
    let v = output.vertices()[86].get();
    assert!(almost_equal(v.x(), 219.0442769, v.y(), 49.5794664));
    assert_eq!(v.get_incident_edge().unwrap().0, 285);
    let v = output.vertices()[87].get();
    assert!(almost_equal(v.x(), 207.6665847, v.y(), 613.5753627));
    assert_eq!(v.get_incident_edge().unwrap().0, 287);
    let v = output.vertices()[88].get();
    assert!(almost_equal(v.x(), 243.0000000, v.y(), 111.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 293);
    let v = output.vertices()[89].get();
    assert!(almost_equal(v.x(), 235.4232780, v.y(), 101.2300164));
    assert_eq!(v.get_incident_edge().unwrap().0, 297);
    let v = output.vertices()[90].get();
    assert!(almost_equal(v.x(), 229.1313403, v.y(), 40.1880625));
    assert_eq!(v.get_incident_edge().unwrap().0, 299);
    let v = output.vertices()[91].get();
    assert!(almost_equal(v.x(), 241.1415449, v.y(), 99.5929305));
    assert_eq!(v.get_incident_edge().unwrap().0, 301);
    let v = output.vertices()[92].get();
    assert!(almost_equal(v.x(), 210.8909700, v.y(), 629.1110373));
    assert_eq!(v.get_incident_edge().unwrap().0, 303);
    let v = output.vertices()[93].get();
    assert!(almost_equal(v.x(), 257.0000000, v.y(), 710.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 309);
    let v = output.vertices()[94].get();
    assert!(almost_equal(v.x(), 229.8824545, v.y(), 229.8824545));
    assert_eq!(v.get_incident_edge().unwrap().0, 313);
    let v = output.vertices()[95].get();
    assert!(almost_equal(v.x(), 228.5720220, v.y(), 165.6535147));
    assert_eq!(v.get_incident_edge().unwrap().0, 317);
    let v = output.vertices()[96].get();
    assert!(almost_equal(v.x(), 238.3312309, v.y(), 559.7489573));
    assert_eq!(v.get_incident_edge().unwrap().0, 319);
    let v = output.vertices()[97].get();
    assert!(almost_equal(v.x(), 247.3670632, v.y(), 725.9196956));
    assert_eq!(v.get_incident_edge().unwrap().0, 321);
    let v = output.vertices()[98].get();
    assert!(almost_equal(v.x(), 266.0000000, v.y(), 484.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 327);
    let v = output.vertices()[99].get();
    assert!(almost_equal(v.x(), 221.6243311, v.y(), 633.9742456));
    assert_eq!(v.get_incident_edge().unwrap().0, 329);
    let v = output.vertices()[100].get();
    assert!(almost_equal(v.x(), 234.2957075, v.y(), 365.7042925));
    assert_eq!(v.get_incident_edge().unwrap().0, 331);
    let v = output.vertices()[101].get();
    assert!(almost_equal(v.x(), 245.0763323, v.y(), 509.8857985));
    assert_eq!(v.get_incident_edge().unwrap().0, 333);
    let v = output.vertices()[102].get();
    assert!(almost_equal(v.x(), 261.4408687, v.y(), 726.9018136));
    assert_eq!(v.get_incident_edge().unwrap().0, 335);
    let v = output.vertices()[103].get();
    assert!(almost_equal(v.x(), 236.3360372, v.y(), 631.6068446));
    assert_eq!(v.get_incident_edge().unwrap().0, 337);
    let v = output.vertices()[104].get();
    assert!(almost_equal(v.x(), 260.2101571, v.y(), 687.8005985));
    assert_eq!(v.get_incident_edge().unwrap().0, 339);
    let v = output.vertices()[105].get();
    assert!(almost_equal(v.x(), 248.3738450, v.y(), 511.9798227));
    assert_eq!(v.get_incident_edge().unwrap().0, 341);
    let v = output.vertices()[106].get();
    assert!(almost_equal(v.x(), 262.0000000, v.y(), 528.2358142));
    assert_eq!(v.get_incident_edge().unwrap().0, 345);
    let v = output.vertices()[107].get();
    assert!(almost_equal(v.x(), 262.0000000, v.y(), 545.6802052));
    assert_eq!(v.get_incident_edge().unwrap().0, 347);
    let v = output.vertices()[108].get();
    assert!(almost_equal(v.x(), 269.6162952, v.y(), 50.8287818));
    assert_eq!(v.get_incident_edge().unwrap().0, 351);
    let v = output.vertices()[109].get();
    assert!(almost_equal(v.x(), 251.9853267, v.y(), 291.6362072));
    assert_eq!(v.get_incident_edge().unwrap().0, 355);
    let v = output.vertices()[110].get();
    assert!(almost_equal(v.x(), 261.8343867, v.y(), 154.2672261));
    assert_eq!(v.get_incident_edge().unwrap().0, 357);
    let v = output.vertices()[111].get();
    assert!(almost_equal(v.x(), 258.0155317, v.y(), 291.5191159));
    assert_eq!(v.get_incident_edge().unwrap().0, 363);
    let v = output.vertices()[112].get();
    assert!(almost_equal(v.x(), 266.3920861, v.y(), 631.8430516));
    assert_eq!(v.get_incident_edge().unwrap().0, 367);
    let v = output.vertices()[113].get();
    assert!(almost_equal(v.x(), 286.6059238, v.y(), 49.3549986));
    assert_eq!(v.get_incident_edge().unwrap().0, 371);
    let v = output.vertices()[114].get();
    assert!(almost_equal(v.x(), 308.4714667, v.y(), 32.6173333));
    assert_eq!(v.get_incident_edge().unwrap().0, 373);
    let v = output.vertices()[115].get();
    assert!(almost_equal(v.x(), 284.8247952, v.y(), 444.1093626));
    assert_eq!(v.get_incident_edge().unwrap().0, 375);
    let v = output.vertices()[116].get();
    assert!(almost_equal(v.x(), 285.4433922, v.y(), 654.9051590));
    assert_eq!(v.get_incident_edge().unwrap().0, 377);
    let v = output.vertices()[117].get();
    assert!(almost_equal(v.x(), 285.5606950, v.y(), 152.3471231));
    assert_eq!(v.get_incident_edge().unwrap().0, 379);
    let v = output.vertices()[118].get();
    assert!(almost_equal(v.x(), 336.0000000, v.y(), 541.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 385);
    let v = output.vertices()[119].get();
    assert!(almost_equal(v.x(), 313.8723560, v.y(), 552.6029228));
    assert_eq!(v.get_incident_edge().unwrap().0, 387);
    let v = output.vertices()[120].get();
    assert!(almost_equal(v.x(), 324.2232914, v.y(), 555.4626247));
    assert_eq!(v.get_incident_edge().unwrap().0, 389);
    let v = output.vertices()[121].get();
    assert!(almost_equal(v.x(), 295.8742489, v.y(), 447.3123259));
    assert_eq!(v.get_incident_edge().unwrap().0, 391);
    let v = output.vertices()[122].get();
    assert!(almost_equal(v.x(), 301.0980947, v.y(), 243.2180721));
    assert_eq!(v.get_incident_edge().unwrap().0, 395);
    let v = output.vertices()[123].get();
    assert!(almost_equal(v.x(), 301.4325482, v.y(), 257.5995717));
    assert_eq!(v.get_incident_edge().unwrap().0, 399);
    let v = output.vertices()[124].get();
    assert!(almost_equal(v.x(), 307.1532156, v.y(), 351.9962654));
    assert_eq!(v.get_incident_edge().unwrap().0, 401);
    let v = output.vertices()[125].get();
    assert!(almost_equal(v.x(), 342.5884507, v.y(), 555.5245390));
    assert_eq!(v.get_incident_edge().unwrap().0, 403);
    let v = output.vertices()[126].get();
    assert!(almost_equal(v.x(), 315.8228628, v.y(), 153.1063815));
    assert_eq!(v.get_incident_edge().unwrap().0, 409);
    let v = output.vertices()[127].get();
    assert!(almost_equal(v.x(), 339.9989303, v.y(), 510.1171002));
    assert_eq!(v.get_incident_edge().unwrap().0, 411);
    let v = output.vertices()[128].get();
    assert!(almost_equal(v.x(), 320.0618342, v.y(), 455.9145395));
    assert_eq!(v.get_incident_edge().unwrap().0, 417);
    let v = output.vertices()[129].get();
    assert!(almost_equal(v.x(), 348.6958227, v.y(), 41.1945186));
    assert_eq!(v.get_incident_edge().unwrap().0, 423);
    let v = output.vertices()[130].get();
    assert!(almost_equal(v.x(), 333.2976577, v.y(), 143.8296385));
    assert_eq!(v.get_incident_edge().unwrap().0, 425);
    let v = output.vertices()[131].get();
    assert!(almost_equal(v.x(), 345.1426991, v.y(), 19.5204646));
    assert_eq!(v.get_incident_edge().unwrap().0, 427);
    let v = output.vertices()[132].get();
    assert!(almost_equal(v.x(), 366.4611605, v.y(), 509.3215891));
    assert_eq!(v.get_incident_edge().unwrap().0, 429);
    let v = output.vertices()[133].get();
    assert!(almost_equal(v.x(), 373.1847157, v.y(), 99.5783412));
    assert_eq!(v.get_incident_edge().unwrap().0, 431);
    let v = output.vertices()[134].get();
    assert!(almost_equal(v.x(), 337.3019780, v.y(), 142.6376264));
    assert_eq!(v.get_incident_edge().unwrap().0, 433);
    let v = output.vertices()[135].get();
    assert!(almost_equal(v.x(), 204.9025641, v.y(), 923.8410256));
    assert_eq!(v.get_incident_edge().unwrap().0, 435);
    let v = output.vertices()[136].get();
    assert!(almost_equal(v.x(), 337.1881287, v.y(), 143.0544731));
    assert_eq!(v.get_incident_edge().unwrap().0, 437);
    let v = output.vertices()[137].get();
    assert!(almost_equal(v.x(), 344.7482761, v.y(), 448.4048858));
    assert_eq!(v.get_incident_edge().unwrap().0, 439);
    let v = output.vertices()[138].get();
    assert!(almost_equal(v.x(), 377.5933595, v.y(), 51.5280262));
    assert_eq!(v.get_incident_edge().unwrap().0, 441);
    let v = output.vertices()[139].get();
    assert!(almost_equal(v.x(), 400.0000000, v.y(), 200.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 447);
    let v = output.vertices()[140].get();
    assert!(almost_equal(v.x(), 371.2380257, v.y(), 228.7619743));
    assert_eq!(v.get_incident_edge().unwrap().0, 449);
    let v = output.vertices()[141].get();
    assert!(almost_equal(v.x(), 352.0848706, v.y(), 285.2281112));
    assert_eq!(v.get_incident_edge().unwrap().0, 451);
    let v = output.vertices()[142].get();
    assert!(almost_equal(v.x(), 344.9059818, v.y(), 344.9059818));
    assert_eq!(v.get_incident_edge().unwrap().0, 453);
    let v = output.vertices()[143].get();
    assert!(almost_equal(v.x(), 400.0000000, v.y(), 400.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 457);
    let v = output.vertices()[144].get();
    assert!(almost_equal(v.x(), 198.3827160, v.y(), 946.6604938));
    assert_eq!(v.get_incident_edge().unwrap().0, 459);
    let v = output.vertices()[145].get();
    assert!(almost_equal(v.x(), 355.2211116, v.y(), 636.5225907));
    assert_eq!(v.get_incident_edge().unwrap().0, 463);
    let v = output.vertices()[146].get();
    assert!(almost_equal(v.x(), 380.8920954, v.y(), 12.7678795));
    assert_eq!(v.get_incident_edge().unwrap().0, 467);
    let v = output.vertices()[147].get();
    assert!(almost_equal(v.x(), 385.7416252, v.y(), 542.8784343));
    assert_eq!(v.get_incident_edge().unwrap().0, 471);
    let v = output.vertices()[148].get();
    assert!(almost_equal(v.x(), 360.5265260, v.y(), 617.0694048));
    assert_eq!(v.get_incident_edge().unwrap().0, 473);
    let v = output.vertices()[149].get();
    assert!(almost_equal(v.x(), 372.8562699, v.y(), 161.7154894));
    assert_eq!(v.get_incident_edge().unwrap().0, 475);
    let v = output.vertices()[150].get();
    assert!(almost_equal(v.x(), 356.3823529, v.y(), -24.8137255));
    assert_eq!(v.get_incident_edge().unwrap().0, 477);
    let v = output.vertices()[151].get();
    assert!(almost_equal(v.x(), 421.0000000, v.y(), 82.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 483);
    let v = output.vertices()[152].get();
    assert!(almost_equal(v.x(), 400.0000000, v.y(), 171.5428751));
    assert_eq!(v.get_incident_edge().unwrap().0, 485);
    let v = output.vertices()[153].get();
    assert!(almost_equal(v.x(), 419.4115863, v.y(), 72.2504265));
    assert_eq!(v.get_incident_edge().unwrap().0, 487);
    let v = output.vertices()[154].get();
    assert!(almost_equal(v.x(), 407.3496843, v.y(), 108.9094919));
    assert_eq!(v.get_incident_edge().unwrap().0, 489);
    let v = output.vertices()[155].get();
    assert!(almost_equal(v.x(), 433.0000000, v.y(), 497.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 497);
    let v = output.vertices()[156].get();
    assert!(almost_equal(v.x(), 400.0000000, v.y(), 438.8523777));
    assert_eq!(v.get_incident_edge().unwrap().0, 499);
    let v = output.vertices()[157].get();
    assert!(almost_equal(v.x(), 427.4135860, v.y(), 484.6844964));
    assert_eq!(v.get_incident_edge().unwrap().0, 501);
    let v = output.vertices()[158].get();
    assert!(almost_equal(v.x(), 428.8140207, v.y(), 484.1629967));
    assert_eq!(v.get_incident_edge().unwrap().0, 503);
    let v = output.vertices()[159].get();
    assert!(almost_equal(v.x(), 434.3229054, v.y(), 75.2256413));
    assert_eq!(v.get_incident_edge().unwrap().0, 505);
    let v = output.vertices()[160].get();
    assert!(almost_equal(v.x(), 451.0000000, v.y(), 141.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 511);
    let v = output.vertices()[161].get();
    assert!(almost_equal(v.x(), 438.1526246, v.y(), 147.5325638));
    assert_eq!(v.get_incident_edge().unwrap().0, 513);
    let v = output.vertices()[162].get();
    assert!(almost_equal(v.x(), 427.0603160, v.y(), 240.7067177));
    assert_eq!(v.get_incident_edge().unwrap().0, 519);
    let v = output.vertices()[163].get();
    assert!(almost_equal(v.x(), 444.4836586, v.y(), 153.3700040));
    assert_eq!(v.get_incident_edge().unwrap().0, 521);
    let v = output.vertices()[164].get();
    assert!(almost_equal(v.x(), 430.6785590, v.y(), 200.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 523);
    let v = output.vertices()[165].get();
    assert!(almost_equal(v.x(), 433.6088985, v.y(), 528.8508555));
    assert_eq!(v.get_incident_edge().unwrap().0, 527);
    let v = output.vertices()[166].get();
    assert!(almost_equal(v.x(), 463.0000000, v.y(), 56.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 533);
    let v = output.vertices()[167].get();
    assert!(almost_equal(v.x(), 464.0000000, v.y(), 554.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 539);
    let v = output.vertices()[168].get();
    assert!(almost_equal(v.x(), 433.4354821, v.y(), 205.8861456));
    assert_eq!(v.get_incident_edge().unwrap().0, 541);
    let v = output.vertices()[169].get();
    assert!(almost_equal(v.x(), 462.0045434, v.y(), 49.0803625));
    assert_eq!(v.get_incident_edge().unwrap().0, 543);
    let v = output.vertices()[170].get();
    assert!(almost_equal(v.x(), 466.6086393, v.y(), 49.7227871));
    assert_eq!(v.get_incident_edge().unwrap().0, 545);
    let v = output.vertices()[171].get();
    assert!(almost_equal(v.x(), 444.1708802, v.y(), 525.7556504));
    assert_eq!(v.get_incident_edge().unwrap().0, 547);
    let v = output.vertices()[172].get();
    assert!(almost_equal(v.x(), 466.7160588, v.y(), 471.2311650));
    assert_eq!(v.get_incident_edge().unwrap().0, 549);
    let v = output.vertices()[173].get();
    assert!(almost_equal(v.x(), 375.2807514, v.y(), 850.5814757));
    assert_eq!(v.get_incident_edge().unwrap().0, 551);
    let v = output.vertices()[174].get();
    assert!(almost_equal(v.x(), 453.1144143, v.y(), 523.6639203));
    assert_eq!(v.get_incident_edge().unwrap().0, 553);
    let v = output.vertices()[175].get();
    assert!(almost_equal(v.x(), 465.9853449, v.y(), 126.2295195));
    assert_eq!(v.get_incident_edge().unwrap().0, 561);
    let v = output.vertices()[176].get();
    assert!(almost_equal(v.x(), 457.0519803, v.y(), 87.0227389));
    assert_eq!(v.get_incident_edge().unwrap().0, 563);
    let v = output.vertices()[177].get();
    assert!(almost_equal(v.x(), 440.0618632, v.y(), 609.0911561));
    assert_eq!(v.get_incident_edge().unwrap().0, 565);
    let v = output.vertices()[178].get();
    assert!(almost_equal(v.x(), 477.5270465, v.y(), 137.7094302));
    assert_eq!(v.get_incident_edge().unwrap().0, 567);
    let v = output.vertices()[179].get();
    assert!(almost_equal(v.x(), 445.9624227, v.y(), 413.0056860));
    assert_eq!(v.get_incident_edge().unwrap().0, 569);
    let v = output.vertices()[180].get();
    assert!(almost_equal(v.x(), 494.0000000, v.y(), 371.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 575);
    let v = output.vertices()[181].get();
    assert!(almost_equal(v.x(), 447.9937680, v.y(), 384.6685182));
    assert_eq!(v.get_incident_edge().unwrap().0, 577);
    let v = output.vertices()[182].get();
    assert!(almost_equal(v.x(), 474.8348325, v.y(), 226.5128396));
    assert_eq!(v.get_incident_edge().unwrap().0, 579);
    let v = output.vertices()[183].get();
    assert!(almost_equal(v.x(), 480.8838423, v.y(), 461.2340549));
    assert_eq!(v.get_incident_edge().unwrap().0, 585);
    let v = output.vertices()[184].get();
    assert!(almost_equal(v.x(), 466.7018481, v.y(), 97.1811906));
    assert_eq!(v.get_incident_edge().unwrap().0, 587);
    let v = output.vertices()[185].get();
    assert!(almost_equal(v.x(), 451.4734043, v.y(), 400.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 589);
    let v = output.vertices()[186].get();
    assert!(almost_equal(v.x(), 479.2851505, v.y(), 441.5168558));
    assert_eq!(v.get_incident_edge().unwrap().0, 591);
    let v = output.vertices()[187].get();
    assert!(almost_equal(v.x(), 453.1240000, v.y(), 405.3502069));
    assert_eq!(v.get_incident_edge().unwrap().0, 593);
    let v = output.vertices()[188].get();
    assert!(almost_equal(v.x(), 512.0000000, v.y(), 643.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 601);
    let v = output.vertices()[189].get();
    assert!(almost_equal(v.x(), 469.4682806, v.y(), 698.9030418));
    assert_eq!(v.get_incident_edge().unwrap().0, 605);
    let v = output.vertices()[190].get();
    assert!(almost_equal(v.x(), 487.1751523, v.y(), 508.0435798));
    assert_eq!(v.get_incident_edge().unwrap().0, 609);
    let v = output.vertices()[191].get();
    assert!(almost_equal(v.x(), 484.6507237, v.y(), 709.1680302));
    assert_eq!(v.get_incident_edge().unwrap().0, 611);
    let v = output.vertices()[192].get();
    assert!(almost_equal(v.x(), 471.5545509, v.y(), 412.1292601));
    assert_eq!(v.get_incident_edge().unwrap().0, 613);
    let v = output.vertices()[193].get();
    assert!(almost_equal(v.x(), 507.9648712, v.y(), 461.1311476));
    assert_eq!(v.get_incident_edge().unwrap().0, 615);
    let v = output.vertices()[194].get();
    assert!(almost_equal(v.x(), 471.0879873, v.y(), 602.7820301));
    assert_eq!(v.get_incident_edge().unwrap().0, 617);
    let v = output.vertices()[195].get();
    assert!(almost_equal(v.x(), 499.4264473, v.y(), 104.8603291));
    assert_eq!(v.get_incident_edge().unwrap().0, 619);
    let v = output.vertices()[196].get();
    assert!(almost_equal(v.x(), 525.0000000, v.y(), 467.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 625);
    let v = output.vertices()[197].get();
    assert!(almost_equal(v.x(), 497.6539352, v.y(), 716.0520833));
    assert_eq!(v.get_incident_edge().unwrap().0, 627);
    let v = output.vertices()[198].get();
    assert!(almost_equal(v.x(), 518.7613497, v.y(), 453.0408324));
    assert_eq!(v.get_incident_edge().unwrap().0, 631);
    let v = output.vertices()[199].get();
    assert!(almost_equal(v.x(), 489.9905970, v.y(), 596.9113752));
    assert_eq!(v.get_incident_edge().unwrap().0, 633);
    let v = output.vertices()[200].get();
    assert!(almost_equal(v.x(), 489.9976364, v.y(), 596.9066682));
    assert_eq!(v.get_incident_edge().unwrap().0, 635);
    let v = output.vertices()[201].get();
    assert!(almost_equal(v.x(), 516.0634823, v.y(), 384.3595397));
    assert_eq!(v.get_incident_edge().unwrap().0, 637);
    let v = output.vertices()[202].get();
    assert!(almost_equal(v.x(), 504.4438216, v.y(), 271.4674140));
    assert_eq!(v.get_incident_edge().unwrap().0, 639);
    let v = output.vertices()[203].get();
    assert!(almost_equal(v.x(), 518.3058824, v.y(), 667.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 641);
    let v = output.vertices()[204].get();
    assert!(almost_equal(v.x(), 506.4241349, v.y(), 274.3379841));
    assert_eq!(v.get_incident_edge().unwrap().0, 643);
    let v = output.vertices()[205].get();
    assert!(almost_equal(v.x(), 530.2061647, v.y(), 488.3094489));
    assert_eq!(v.get_incident_edge().unwrap().0, 645);
    let v = output.vertices()[206].get();
    assert!(almost_equal(v.x(), 525.7435897, v.y(), 667.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 647);
    let v = output.vertices()[207].get();
    assert!(almost_equal(v.x(), 531.7220424, v.y(), 487.6142635));
    assert_eq!(v.get_incident_edge().unwrap().0, 649);
    let v = output.vertices()[208].get();
    assert!(almost_equal(v.x(), 535.7328635, v.y(), 485.5141895));
    assert_eq!(v.get_incident_edge().unwrap().0, 651);
    let v = output.vertices()[209].get();
    assert!(almost_equal(v.x(), 529.5686025, v.y(), 666.8890144));
    assert_eq!(v.get_incident_edge().unwrap().0, 653);
    let v = output.vertices()[210].get();
    assert!(almost_equal(v.x(), 560.0000000, v.y(), 262.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 659);
    let v = output.vertices()[211].get();
    assert!(almost_equal(v.x(), 541.3528746, v.y(), 256.3051731));
    assert_eq!(v.get_incident_edge().unwrap().0, 661);
    let v = output.vertices()[212].get();
    assert!(almost_equal(v.x(), 541.7574333, v.y(), 226.6910800));
    assert_eq!(v.get_incident_edge().unwrap().0, 663);
    let v = output.vertices()[213].get();
    assert!(almost_equal(v.x(), 536.6438082, v.y(), 215.0486016));
    assert_eq!(v.get_incident_edge().unwrap().0, 665);
    let v = output.vertices()[214].get();
    assert!(almost_equal(v.x(), 563.0000000, v.y(), 200.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 669);
    let v = output.vertices()[215].get();
    assert!(almost_equal(v.x(), 485.9231707, v.y(), 828.7780488));
    assert_eq!(v.get_incident_edge().unwrap().0, 671);
    let v = output.vertices()[216].get();
    assert!(almost_equal(v.x(), 567.6011411, v.y(), 191.2656305));
    assert_eq!(v.get_incident_edge().unwrap().0, 673);
    let v = output.vertices()[217].get();
    assert!(almost_equal(v.x(), 545.2510903, v.y(), 665.9938109));
    assert_eq!(v.get_incident_edge().unwrap().0, 681);
    let v = output.vertices()[218].get();
    assert!(almost_equal(v.x(), 575.9926968, v.y(), 200.6286789));
    assert_eq!(v.get_incident_edge().unwrap().0, 685);
    let v = output.vertices()[219].get();
    assert!(almost_equal(v.x(), 543.3512304, v.y(), 561.2273785));
    assert_eq!(v.get_incident_edge().unwrap().0, 687);
    let v = output.vertices()[220].get();
    assert!(almost_equal(v.x(), 594.0000000, v.y(), 427.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 695);
    let v = output.vertices()[221].get();
    assert!(almost_equal(v.x(), 535.5372274, v.y(), 772.4987907));
    assert_eq!(v.get_incident_edge().unwrap().0, 697);
    let v = output.vertices()[222].get();
    assert!(almost_equal(v.x(), 567.8763297, v.y(), 407.4540484));
    assert_eq!(v.get_incident_edge().unwrap().0, 699);
    let v = output.vertices()[223].get();
    assert!(almost_equal(v.x(), 568.4973753, v.y(), 407.4171139));
    assert_eq!(v.get_incident_edge().unwrap().0, 701);
    let v = output.vertices()[224].get();
    assert!(almost_equal(v.x(), 589.6072108, v.y(), 360.7022464));
    assert_eq!(v.get_incident_edge().unwrap().0, 705);
    let v = output.vertices()[225].get();
    assert!(almost_equal(v.x(), 612.0000000, v.y(), 209.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 713);
    let v = output.vertices()[226].get();
    assert!(almost_equal(v.x(), 587.5053351, v.y(), 223.4496521));
    assert_eq!(v.get_incident_edge().unwrap().0, 715);
    let v = output.vertices()[227].get();
    assert!(almost_equal(v.x(), 585.8085961, v.y(), 700.3906207));
    assert_eq!(v.get_incident_edge().unwrap().0, 719);
    let v = output.vertices()[228].get();
    assert!(almost_equal(v.x(), 585.0828452, v.y(), 277.1877779));
    assert_eq!(v.get_incident_edge().unwrap().0, 721);
    let v = output.vertices()[229].get();
    assert!(almost_equal(v.x(), 588.1997608, v.y(), 263.3645046));
    assert_eq!(v.get_incident_edge().unwrap().0, 723);
    let v = output.vertices()[230].get();
    assert!(almost_equal(v.x(), 617.0000000, v.y(), 342.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 729);
    let v = output.vertices()[231].get();
    assert!(almost_equal(v.x(), 587.5644917, v.y(), 272.9572005));
    assert_eq!(v.get_incident_edge().unwrap().0, 731);
    let v = output.vertices()[232].get();
    assert!(almost_equal(v.x(), 583.7445960, v.y(), 86.3840693));
    assert_eq!(v.get_incident_edge().unwrap().0, 733);
    let v = output.vertices()[233].get();
    assert!(almost_equal(v.x(), 600.3129897, v.y(), 337.4846913));
    assert_eq!(v.get_incident_edge().unwrap().0, 735);
    let v = output.vertices()[234].get();
    assert!(almost_equal(v.x(), 618.0000000, v.y(), 281.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 739);
    let v = output.vertices()[235].get();
    assert!(almost_equal(v.x(), 605.8405676, v.y(), 329.0550585));
    assert_eq!(v.get_incident_edge().unwrap().0, 741);
    let v = output.vertices()[236].get();
    assert!(almost_equal(v.x(), 604.3450942, v.y(), 444.8452874));
    assert_eq!(v.get_incident_edge().unwrap().0, 743);
    let v = output.vertices()[237].get();
    assert!(almost_equal(v.x(), 629.0000000, v.y(), 758.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 749);
    let v = output.vertices()[238].get();
    assert!(almost_equal(v.x(), 605.8065278, v.y(), 713.5468718));
    assert_eq!(v.get_incident_edge().unwrap().0, 751);
    let v = output.vertices()[239].get();
    assert!(almost_equal(v.x(), 611.8761662, v.y(), 79.9550466));
    assert_eq!(v.get_incident_edge().unwrap().0, 753);
    let v = output.vertices()[240].get();
    assert!(almost_equal(v.x(), 608.5858114, v.y(), 657.8731520));
    assert_eq!(v.get_incident_edge().unwrap().0, 755);
    let v = output.vertices()[241].get();
    assert!(almost_equal(v.x(), 597.7646129, v.y(), 632.3943898));
    assert_eq!(v.get_incident_edge().unwrap().0, 757);
    let v = output.vertices()[242].get();
    assert!(almost_equal(v.x(), 617.1017027, v.y(), 433.2510490));
    assert_eq!(v.get_incident_edge().unwrap().0, 759);
    let v = output.vertices()[243].get();
    assert!(almost_equal(v.x(), 616.7589723, v.y(), 598.6600357));
    assert_eq!(v.get_incident_edge().unwrap().0, 763);
    let v = output.vertices()[244].get();
    assert!(almost_equal(v.x(), 611.8194939, v.y(), 788.0017794));
    assert_eq!(v.get_incident_edge().unwrap().0, 765);
    let v = output.vertices()[245].get();
    assert!(almost_equal(v.x(), 617.1356621, v.y(), 18.1559708));
    assert_eq!(v.get_incident_edge().unwrap().0, 767);
    let v = output.vertices()[246].get();
    assert!(almost_equal(v.x(), 628.2103477, v.y(), 184.6844784));
    assert_eq!(v.get_incident_edge().unwrap().0, 769);
    let v = output.vertices()[247].get();
    assert!(almost_equal(v.x(), 637.1186829, v.y(), 294.0754211));
    assert_eq!(v.get_incident_edge().unwrap().0, 771);
    let v = output.vertices()[248].get();
    assert!(almost_equal(v.x(), 617.1530210, v.y(), 511.8737125));
    assert_eq!(v.get_incident_edge().unwrap().0, 773);
    let v = output.vertices()[249].get();
    assert!(almost_equal(v.x(), 670.0000000, v.y(), 175.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 781);
    let v = output.vertices()[250].get();
    assert!(almost_equal(v.x(), 646.5880885, v.y(), 278.6176593));
    assert_eq!(v.get_incident_edge().unwrap().0, 785);
    let v = output.vertices()[251].get();
    assert!(almost_equal(v.x(), 643.0426971, v.y(), 197.6341305));
    assert_eq!(v.get_incident_edge().unwrap().0, 787);
    let v = output.vertices()[252].get();
    assert!(almost_equal(v.x(), 645.5275612, v.y(), 206.2060366));
    assert_eq!(v.get_incident_edge().unwrap().0, 797);
    let v = output.vertices()[253].get();
    assert!(almost_equal(v.x(), 658.9026767, v.y(), 674.3755423));
    assert_eq!(v.get_incident_edge().unwrap().0, 799);
    let v = output.vertices()[254].get();
    assert!(almost_equal(v.x(), 513.9388260, v.y(), 909.7121641));
    assert_eq!(v.get_incident_edge().unwrap().0, 801);
    let v = output.vertices()[255].get();
    assert!(almost_equal(v.x(), 645.5000812, v.y(), 395.5163942));
    assert_eq!(v.get_incident_edge().unwrap().0, 803);
    let v = output.vertices()[256].get();
    assert!(almost_equal(v.x(), 652.0166959, v.y(), 265.3393672));
    assert_eq!(v.get_incident_edge().unwrap().0, 805);
    let v = output.vertices()[257].get();
    assert!(almost_equal(v.x(), 633.6008471, v.y(), -0.5802743));
    assert_eq!(v.get_incident_edge().unwrap().0, 807);
    let v = output.vertices()[258].get();
    assert!(almost_equal(v.x(), 497.4615489, v.y(), -154.8528761));
    assert_eq!(v.get_incident_edge().unwrap().0, 809);
    let v = output.vertices()[259].get();
    assert!(almost_equal(v.x(), 656.4537249, v.y(), 256.1234352));
    assert_eq!(v.get_incident_edge().unwrap().0, 811);
    let v = output.vertices()[260].get();
    assert!(almost_equal(v.x(), 654.4268711, v.y(), 385.2678891));
    assert_eq!(v.get_incident_edge().unwrap().0, 815);
    let v = output.vertices()[261].get();
    assert!(almost_equal(v.x(), 657.5904061, v.y(), 367.5499538));
    assert_eq!(v.get_incident_edge().unwrap().0, 817);
    let v = output.vertices()[262].get();
    assert!(almost_equal(v.x(), 656.3960797, v.y(), 81.7358432));
    assert_eq!(v.get_incident_edge().unwrap().0, 819);
    let v = output.vertices()[263].get();
    assert!(almost_equal(v.x(), 659.9045381, v.y(), 372.5560231));
    assert_eq!(v.get_incident_edge().unwrap().0, 821);
    let v = output.vertices()[264].get();
    assert!(almost_equal(v.x(), 688.9125561, v.y(), 308.1385650));
    assert_eq!(v.get_incident_edge().unwrap().0, 825);
    let v = output.vertices()[265].get();
    assert!(almost_equal(v.x(), 669.8492433, v.y(), 761.6426077));
    assert_eq!(v.get_incident_edge().unwrap().0, 827);
    let v = output.vertices()[266].get();
    assert!(almost_equal(v.x(), 692.8391698, v.y(), 307.6898092));
    assert_eq!(v.get_incident_edge().unwrap().0, 829);
    let v = output.vertices()[267].get();
    assert!(almost_equal(v.x(), 681.9032429, v.y(), 604.4690790));
    assert_eq!(v.get_incident_edge().unwrap().0, 831);
    let v = output.vertices()[268].get();
    assert!(almost_equal(v.x(), 679.0601012, v.y(), 733.6913900));
    assert_eq!(v.get_incident_edge().unwrap().0, 833);
    let v = output.vertices()[269].get();
    assert!(almost_equal(v.x(), 687.4987316, v.y(), 464.8312399));
    assert_eq!(v.get_incident_edge().unwrap().0, 837);
    let v = output.vertices()[270].get();
    assert!(almost_equal(v.x(), 689.2098887, v.y(), 141.5844794));
    assert_eq!(v.get_incident_edge().unwrap().0, 839);
    let v = output.vertices()[271].get();
    assert!(almost_equal(v.x(), 732.0000000, v.y(), 346.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 845);
    let v = output.vertices()[272].get();
    assert!(almost_equal(v.x(), 699.6541453, v.y(), 352.0596290));
    assert_eq!(v.get_incident_edge().unwrap().0, 847);
    let v = output.vertices()[273].get();
    assert!(almost_equal(v.x(), 692.5352674, v.y(), 364.5042847));
    assert_eq!(v.get_incident_edge().unwrap().0, 849);
    let v = output.vertices()[274].get();
    assert!(almost_equal(v.x(), 672.7868069, v.y(), 106.4661106));
    assert_eq!(v.get_incident_edge().unwrap().0, 853);
    let v = output.vertices()[275].get();
    assert!(almost_equal(v.x(), 700.1857471, v.y(), 437.8058578));
    assert_eq!(v.get_incident_edge().unwrap().0, 857);
    let v = output.vertices()[276].get();
    assert!(almost_equal(v.x(), 694.8607078, v.y(), 459.2594162));
    assert_eq!(v.get_incident_edge().unwrap().0, 859);
    let v = output.vertices()[277].get();
    assert!(almost_equal(v.x(), 735.0000000, v.y(), 479.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 863);
    let v = output.vertices()[278].get();
    assert!(almost_equal(v.x(), 686.3533327, v.y(), 576.5557752));
    assert_eq!(v.get_incident_edge().unwrap().0, 865);
    let v = output.vertices()[279].get();
    assert!(almost_equal(v.x(), 674.5199734, v.y(), 99.5932091));
    assert_eq!(v.get_incident_edge().unwrap().0, 867);
    let v = output.vertices()[280].get();
    assert!(almost_equal(v.x(), 704.3001872, v.y(), 162.5636748));
    assert_eq!(v.get_incident_edge().unwrap().0, 869);
    let v = output.vertices()[281].get();
    assert!(almost_equal(v.x(), 693.1519629, v.y(), 569.1359774));
    assert_eq!(v.get_incident_edge().unwrap().0, 871);
    let v = output.vertices()[282].get();
    assert!(almost_equal(v.x(), 701.0929604, v.y(), 561.4185921));
    assert_eq!(v.get_incident_edge().unwrap().0, 877);
    let v = output.vertices()[283].get();
    assert!(almost_equal(v.x(), 742.1210191, v.y(), 141.6878981));
    assert_eq!(v.get_incident_edge().unwrap().0, 883);
    let v = output.vertices()[284].get();
    assert!(almost_equal(v.x(), 718.1320317, v.y(), 177.5020688));
    assert_eq!(v.get_incident_edge().unwrap().0, 885);
    let v = output.vertices()[285].get();
    assert!(almost_equal(v.x(), 711.0000000, v.y(), 81.5412371));
    assert_eq!(v.get_incident_edge().unwrap().0, 889);
    let v = output.vertices()[286].get();
    assert!(almost_equal(v.x(), 732.2352217, v.y(), 239.4776595));
    assert_eq!(v.get_incident_edge().unwrap().0, 891);
    let v = output.vertices()[287].get();
    assert!(almost_equal(v.x(), 746.3236663, v.y(), 301.3884525));
    assert_eq!(v.get_incident_edge().unwrap().0, 893);
    let v = output.vertices()[288].get();
    assert!(almost_equal(v.x(), 754.6099983, v.y(), 337.8022228));
    assert_eq!(v.get_incident_edge().unwrap().0, 895);
    let v = output.vertices()[289].get();
    assert!(almost_equal(v.x(), 755.7110972, v.y(), 345.4651632));
    assert_eq!(v.get_incident_edge().unwrap().0, 897);
    let v = output.vertices()[290].get();
    assert!(almost_equal(v.x(), 726.4395880, v.y(), 715.4849666));
    assert_eq!(v.get_incident_edge().unwrap().0, 899);
    let v = output.vertices()[291].get();
    assert!(almost_equal(v.x(), 726.4620419, v.y(), 715.6335079));
    assert_eq!(v.get_incident_edge().unwrap().0, 901);
    let v = output.vertices()[292].get();
    assert!(almost_equal(v.x(), 758.5915076, v.y(), 392.5408492));
    assert_eq!(v.get_incident_edge().unwrap().0, 905);
    let v = output.vertices()[293].get();
    assert!(almost_equal(v.x(), 757.1083213, v.y(), 509.0619247));
    assert_eq!(v.get_incident_edge().unwrap().0, 907);
    let v = output.vertices()[294].get();
    assert!(almost_equal(v.x(), 752.1265625, v.y(), 88.3250000));
    assert_eq!(v.get_incident_edge().unwrap().0, 909);
    let v = output.vertices()[295].get();
    assert!(almost_equal(v.x(), 779.7246659, v.y(), 621.8324053));
    assert_eq!(v.get_incident_edge().unwrap().0, 911);
    let v = output.vertices()[296].get();
    assert!(almost_equal(v.x(), 836.1811616, v.y(), 476.7177182));
    assert_eq!(v.get_incident_edge().unwrap().0, 913);
    let v = output.vertices()[297].get();
    assert!(almost_equal(v.x(), 620.3654971, v.y(), 1095.6578947));
    assert_eq!(v.get_incident_edge().unwrap().0, 915);
    let v = output.vertices()[298].get();
    assert!(almost_equal(v.x(), 856.8056769, v.y(), 592.5611354));
    assert_eq!(v.get_incident_edge().unwrap().0, 917);
    let v = output.vertices()[299].get();
    assert!(almost_equal(v.x(), 867.2333166, v.y(), 493.8138485));
    assert_eq!(v.get_incident_edge().unwrap().0, 919);
    let v = output.vertices()[300].get();
    assert!(almost_equal(v.x(), 711.0000000, v.y(), -690.1363636));
    assert_eq!(v.get_incident_edge().unwrap().0, 921);
    let v = output.vertices()[301].get();
    assert!(almost_equal(v.x(), 1101.4076456, v.y(), 554.6383495));
    assert_eq!(v.get_incident_edge().unwrap().0, 923);
    let v = output.vertices()[302].get();
    assert!(almost_equal(v.x(), 521.6624933, v.y(), 2145.4989345));
    assert_eq!(v.get_incident_edge().unwrap().0, 925);
    let v = output.vertices()[303].get();
    assert!(almost_equal(v.x(), 516.6056696, v.y(), 2285.5729527));
    assert_eq!(v.get_incident_edge().unwrap().0, 927);
    Ok(())
}
