use boostvoronoi::builder as VB;
use boostvoronoi::diagram as VD;
use boostvoronoi::file_reader as FR;
use boostvoronoi::sync_diagram as VS;
use boostvoronoi::BvError;
use std::io::{BufReader, Cursor};

#[allow(dead_code)]
fn almost_equal(x1: F1, x2: F1, y1: F1, y2: F1) -> bool {
    let delta = 0.000001;
    assert!(F1::abs(x1 - x2) < delta, "{} != {}", x1, x2);
    assert!(F1::abs(y1 - y2) < delta, "{} != {}", y1, y2);

    (F1::abs(x1 - x2) < delta) && (F1::abs(y1 - y2) < delta)
}

type I1 = i32;
type F1 = f64;

#[test]
fn sample_primary_070() -> Result<(), BvError> {
    let output: VS::SyncVoronoiDiagram<I1, F1> = {
        let input = r#"1
-49400 -49572
2
-49431 -49703 -49427 -49798
-49427 -49798 -49423 -49892
"#;
        let mut vb = VB::Builder::<I1, F1>::default();
        let br = BufReader::new(Cursor::new(input));
        let (points, segments) = FR::read_boost_input_buffer::<I1, _>(br)?;
        vb.with_vertices(points.iter())?;
        vb.with_segments(segments.iter())?;
        vb.construct()?.into()
    };
    assert_eq!(output.cells().len(), 6);
    let cell = output.cells()[0];
    assert_eq!(cell.get_id(), 0);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[1];
    assert_eq!(cell.get_id(), 1);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[2];
    assert_eq!(cell.get_id(), 2);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[3];
    assert_eq!(cell.get_id(), 3);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[4];
    assert_eq!(cell.get_id(), 4);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[5];
    assert_eq!(cell.get_id(), 5);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::SinglePoint);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    assert_eq!(output.vertices().len(), 4);
    assert_eq!(output.edges().len(), 18);
    let v = output.vertices()[0];
    assert!(almost_equal(v.x(), -49427.0000000, v.y(), -49798.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 7);
    let v = output.vertices()[1];
    assert!(almost_equal(v.x(), -49182.8607668, v.y(), -49692.5520323));
    assert_eq!(v.get_incident_edge().unwrap().0, 13);
    let v = output.vertices()[2];
    assert!(almost_equal(v.x(), -48718.6308662, v.y(), -49768.0152834));
    assert_eq!(v.get_incident_edge().unwrap().0, 15);
    let v = output.vertices()[3];
    assert!(almost_equal(v.x(), -48017.5197560, v.y(), -49832.1923300));
    assert_eq!(v.get_incident_edge().unwrap().0, 17);
    Ok(())
}

#[test]
fn sample_primary_071() -> Result<(), BvError> {
    let output: VS::SyncVoronoiDiagram<I1, F1> = {
        let input = r#"0
90
-50051 -49552 -49991 -49625
-50042 -49515 -49976 -49442
-50024 -49188 -49996 -49221
-50007 -49191 -49933 -49159
-50000 -49509 -49989 -49549
-49991 -49047 -49950 -49120
-49984 -49938 -49909 -49904
-49959 -49631 -49908 -49568
-49954 -49626 -49887 -49595
-49949 -49320 -49859 -49414
-49916 -49188 -49859 -49241
-49906 -49611 -49830 -49675
-49905 -49754 -49847 -49843
-49892 -49327 -49848 -49239
-49882 -49416 -49872 -49486
-49871 -49640 -49797 -49703
-49846 -49283 -49821 -49368
-49767 -49927 -49707 -49928
-49765 -49619 -49762 -49671
-49764 -49476 -49721 -49531
-49753 -49486 -49691 -49491
-49738 -49590 -49712 -49679
-49736 -49108 -49686 -49138
-49736 -49008 -49671 -48932
-49724 -49802 -49627 -49770
-49717 -49691 -49694 -49699
-49714 -49152 -49708 -49200
-49707 -49875 -49621 -49790
-49700 -49062 -49648 -49109
-49684 -49680 -49611 -49740
-49654 -49456 -49562 -49400
-49611 -49270 -49584 -49339
-49585 -49234 -49567 -49210
-49578 -49012 -49527 -49080
-49561 -49658 -49538 -49670
-49536 -49144 -49445 -49082
-49523 -49784 -49500 -49868
-49509 -49742 -49424 -49733
-49508 -49070 -49472 -49092
-49490 -49176 -49439 -49082
-49477 -49856 -49475 -49779
-49467 -49051 -49446 -49014
-49462 -49793 -49457 -49818
-49460 -49642 -49381 -49620
-49444 -49977 -49349 -50073
-49444 -49973 -49396 -49915
-49425 -49114 -49363 -49182
-49424 -49002 -49351 -49096
-49400 -49470 -49311 -49557
-49392 -49631 -49324 -49704
-49375 -49195 -49281 -49123
-49361 -49096 -49314 -49143
-49345 -49402 -49256 -49437
-49339 -49476 -49298 -49437
-49324 -49287 -49242 -49273
-49290 -49579 -49252 -49548
-49280 -49601 -49217 -49612
-49268 -49108 -49169 -49112
-49257 -49481 -49241 -49480
-49248 -49521 -49233 -49580
-49246 -49802 -49245 -49893
-49236 -49292 -49205 -49336
-49220 -49157 -49210 -49143
-49209 -49367 -49171 -49405
-49205 -49883 -49176 -49926
-49204 -49128 -49131 -49099
-49200 -49252 -49128 -49166
-49194 -49507 -49189 -49496
-49185 -49344 -49140 -49354
-49183 -49895 -49144 -49867
-49183 -49289 -49135 -49213
-49177 -49192 -49168 -49158
-49140 -49876 -49120 -49924
-49140 -49855 -49116 -49874
-49138 -49132 -49081 -49142
-49127 -49293 -49090 -49363
-49096 -49761 -49037 -49744
-49096 -49101 -49081 -49093
-49090 -49308 -49083 -49401
-49086 -49605 -49074 -49539
-49084 -49865 -49059 -49795
-49084 -49124 -48989 -49171
-49077 -49034 -49067 -49009
-49069 -49979 -49067 -49965
-49063 -49317 -49050 -49248
-49036 -49851 -49002 -49942
-49032 -49126 -49003 -49097
-49025 -49181 -48951 -49179
-49022 -49065 -48973 -49118
-49018 -49876 -48988 -49875
"#;
        let mut vb = VB::Builder::<I1, F1>::default();
        let br = BufReader::new(Cursor::new(input));
        let (points, segments) = FR::read_boost_input_buffer::<I1, _>(br)?;
        vb.with_vertices(points.iter())?;
        vb.with_segments(segments.iter())?;
        vb.construct()?.into()
    };
    assert_eq!(output.cells().len(), 270);
    let cell = output.cells()[0];
    assert_eq!(cell.get_id(), 0);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[1];
    assert_eq!(cell.get_id(), 1);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[2];
    assert_eq!(cell.get_id(), 2);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[3];
    assert_eq!(cell.get_id(), 3);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[4];
    assert_eq!(cell.get_id(), 4);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[5];
    assert_eq!(cell.get_id(), 5);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[6];
    assert_eq!(cell.get_id(), 6);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[7];
    assert_eq!(cell.get_id(), 7);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[8];
    assert_eq!(cell.get_id(), 8);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[9];
    assert_eq!(cell.get_id(), 9);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[10];
    assert_eq!(cell.get_id(), 10);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[11];
    assert_eq!(cell.get_id(), 11);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[12];
    assert_eq!(cell.get_id(), 12);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[13];
    assert_eq!(cell.get_id(), 13);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[14];
    assert_eq!(cell.get_id(), 14);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[15];
    assert_eq!(cell.get_id(), 15);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[16];
    assert_eq!(cell.get_id(), 16);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[17];
    assert_eq!(cell.get_id(), 17);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[18];
    assert_eq!(cell.get_id(), 18);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[19];
    assert_eq!(cell.get_id(), 19);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[20];
    assert_eq!(cell.get_id(), 20);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[21];
    assert_eq!(cell.get_id(), 21);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[22];
    assert_eq!(cell.get_id(), 22);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[23];
    assert_eq!(cell.get_id(), 23);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[24];
    assert_eq!(cell.get_id(), 24);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[25];
    assert_eq!(cell.get_id(), 25);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[26];
    assert_eq!(cell.get_id(), 26);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[27];
    assert_eq!(cell.get_id(), 27);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[28];
    assert_eq!(cell.get_id(), 28);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[29];
    assert_eq!(cell.get_id(), 29);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[30];
    assert_eq!(cell.get_id(), 30);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[31];
    assert_eq!(cell.get_id(), 31);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[32];
    assert_eq!(cell.get_id(), 32);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[33];
    assert_eq!(cell.get_id(), 33);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[34];
    assert_eq!(cell.get_id(), 34);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[35];
    assert_eq!(cell.get_id(), 35);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[36];
    assert_eq!(cell.get_id(), 36);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[37];
    assert_eq!(cell.get_id(), 37);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[38];
    assert_eq!(cell.get_id(), 38);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[39];
    assert_eq!(cell.get_id(), 39);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[40];
    assert_eq!(cell.get_id(), 40);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[41];
    assert_eq!(cell.get_id(), 41);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[42];
    assert_eq!(cell.get_id(), 42);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[43];
    assert_eq!(cell.get_id(), 43);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[44];
    assert_eq!(cell.get_id(), 44);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[45];
    assert_eq!(cell.get_id(), 45);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[46];
    assert_eq!(cell.get_id(), 46);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[47];
    assert_eq!(cell.get_id(), 47);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[48];
    assert_eq!(cell.get_id(), 48);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[49];
    assert_eq!(cell.get_id(), 49);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[50];
    assert_eq!(cell.get_id(), 50);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[51];
    assert_eq!(cell.get_id(), 51);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[52];
    assert_eq!(cell.get_id(), 52);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[53];
    assert_eq!(cell.get_id(), 53);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[54];
    assert_eq!(cell.get_id(), 54);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[55];
    assert_eq!(cell.get_id(), 55);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[56];
    assert_eq!(cell.get_id(), 56);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[57];
    assert_eq!(cell.get_id(), 57);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[58];
    assert_eq!(cell.get_id(), 58);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[59];
    assert_eq!(cell.get_id(), 59);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[60];
    assert_eq!(cell.get_id(), 60);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[61];
    assert_eq!(cell.get_id(), 61);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[62];
    assert_eq!(cell.get_id(), 62);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[63];
    assert_eq!(cell.get_id(), 63);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[64];
    assert_eq!(cell.get_id(), 64);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[65];
    assert_eq!(cell.get_id(), 65);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[66];
    assert_eq!(cell.get_id(), 66);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[67];
    assert_eq!(cell.get_id(), 67);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[68];
    assert_eq!(cell.get_id(), 68);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[69];
    assert_eq!(cell.get_id(), 69);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[70];
    assert_eq!(cell.get_id(), 70);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[71];
    assert_eq!(cell.get_id(), 71);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[72];
    assert_eq!(cell.get_id(), 72);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[73];
    assert_eq!(cell.get_id(), 73);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[74];
    assert_eq!(cell.get_id(), 74);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[75];
    assert_eq!(cell.get_id(), 75);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[76];
    assert_eq!(cell.get_id(), 76);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[77];
    assert_eq!(cell.get_id(), 77);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[78];
    assert_eq!(cell.get_id(), 78);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[79];
    assert_eq!(cell.get_id(), 79);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[80];
    assert_eq!(cell.get_id(), 80);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[81];
    assert_eq!(cell.get_id(), 81);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[82];
    assert_eq!(cell.get_id(), 82);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[83];
    assert_eq!(cell.get_id(), 83);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[84];
    assert_eq!(cell.get_id(), 84);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[85];
    assert_eq!(cell.get_id(), 85);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[86];
    assert_eq!(cell.get_id(), 86);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[87];
    assert_eq!(cell.get_id(), 87);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[88];
    assert_eq!(cell.get_id(), 88);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[89];
    assert_eq!(cell.get_id(), 89);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[90];
    assert_eq!(cell.get_id(), 90);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[91];
    assert_eq!(cell.get_id(), 91);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[92];
    assert_eq!(cell.get_id(), 92);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[93];
    assert_eq!(cell.get_id(), 93);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[94];
    assert_eq!(cell.get_id(), 94);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[95];
    assert_eq!(cell.get_id(), 95);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[96];
    assert_eq!(cell.get_id(), 96);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[97];
    assert_eq!(cell.get_id(), 97);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[98];
    assert_eq!(cell.get_id(), 98);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[99];
    assert_eq!(cell.get_id(), 99);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[100];
    assert_eq!(cell.get_id(), 100);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[101];
    assert_eq!(cell.get_id(), 101);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[102];
    assert_eq!(cell.get_id(), 102);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[103];
    assert_eq!(cell.get_id(), 103);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[104];
    assert_eq!(cell.get_id(), 104);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[105];
    assert_eq!(cell.get_id(), 105);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[106];
    assert_eq!(cell.get_id(), 106);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[107];
    assert_eq!(cell.get_id(), 107);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[108];
    assert_eq!(cell.get_id(), 108);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[109];
    assert_eq!(cell.get_id(), 109);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[110];
    assert_eq!(cell.get_id(), 110);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[111];
    assert_eq!(cell.get_id(), 111);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[112];
    assert_eq!(cell.get_id(), 112);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[113];
    assert_eq!(cell.get_id(), 113);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[114];
    assert_eq!(cell.get_id(), 114);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[115];
    assert_eq!(cell.get_id(), 115);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[116];
    assert_eq!(cell.get_id(), 116);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[117];
    assert_eq!(cell.get_id(), 117);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[118];
    assert_eq!(cell.get_id(), 118);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[119];
    assert_eq!(cell.get_id(), 119);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[120];
    assert_eq!(cell.get_id(), 120);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[121];
    assert_eq!(cell.get_id(), 121);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[122];
    assert_eq!(cell.get_id(), 122);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[123];
    assert_eq!(cell.get_id(), 123);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[124];
    assert_eq!(cell.get_id(), 124);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[125];
    assert_eq!(cell.get_id(), 125);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[126];
    assert_eq!(cell.get_id(), 126);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[127];
    assert_eq!(cell.get_id(), 127);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[128];
    assert_eq!(cell.get_id(), 128);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[129];
    assert_eq!(cell.get_id(), 129);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[130];
    assert_eq!(cell.get_id(), 130);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[131];
    assert_eq!(cell.get_id(), 131);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[132];
    assert_eq!(cell.get_id(), 132);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[133];
    assert_eq!(cell.get_id(), 133);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[134];
    assert_eq!(cell.get_id(), 134);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[135];
    assert_eq!(cell.get_id(), 135);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[136];
    assert_eq!(cell.get_id(), 136);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[137];
    assert_eq!(cell.get_id(), 137);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[138];
    assert_eq!(cell.get_id(), 138);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[139];
    assert_eq!(cell.get_id(), 139);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[140];
    assert_eq!(cell.get_id(), 140);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[141];
    assert_eq!(cell.get_id(), 141);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[142];
    assert_eq!(cell.get_id(), 142);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[143];
    assert_eq!(cell.get_id(), 143);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[144];
    assert_eq!(cell.get_id(), 144);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[145];
    assert_eq!(cell.get_id(), 145);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[146];
    assert_eq!(cell.get_id(), 146);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[147];
    assert_eq!(cell.get_id(), 147);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[148];
    assert_eq!(cell.get_id(), 148);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[149];
    assert_eq!(cell.get_id(), 149);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[150];
    assert_eq!(cell.get_id(), 150);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[151];
    assert_eq!(cell.get_id(), 151);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[152];
    assert_eq!(cell.get_id(), 152);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[153];
    assert_eq!(cell.get_id(), 153);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[154];
    assert_eq!(cell.get_id(), 154);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[155];
    assert_eq!(cell.get_id(), 155);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[156];
    assert_eq!(cell.get_id(), 156);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[157];
    assert_eq!(cell.get_id(), 157);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[158];
    assert_eq!(cell.get_id(), 158);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[159];
    assert_eq!(cell.get_id(), 159);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[160];
    assert_eq!(cell.get_id(), 160);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[161];
    assert_eq!(cell.get_id(), 161);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[162];
    assert_eq!(cell.get_id(), 162);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[163];
    assert_eq!(cell.get_id(), 163);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[164];
    assert_eq!(cell.get_id(), 164);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[165];
    assert_eq!(cell.get_id(), 165);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[166];
    assert_eq!(cell.get_id(), 166);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[167];
    assert_eq!(cell.get_id(), 167);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[168];
    assert_eq!(cell.get_id(), 168);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[169];
    assert_eq!(cell.get_id(), 169);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[170];
    assert_eq!(cell.get_id(), 170);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[171];
    assert_eq!(cell.get_id(), 171);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[172];
    assert_eq!(cell.get_id(), 172);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[173];
    assert_eq!(cell.get_id(), 173);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[174];
    assert_eq!(cell.get_id(), 174);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[175];
    assert_eq!(cell.get_id(), 175);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[176];
    assert_eq!(cell.get_id(), 176);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[177];
    assert_eq!(cell.get_id(), 177);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[178];
    assert_eq!(cell.get_id(), 178);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[179];
    assert_eq!(cell.get_id(), 179);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[180];
    assert_eq!(cell.get_id(), 180);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[181];
    assert_eq!(cell.get_id(), 181);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[182];
    assert_eq!(cell.get_id(), 182);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[183];
    assert_eq!(cell.get_id(), 183);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[184];
    assert_eq!(cell.get_id(), 184);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[185];
    assert_eq!(cell.get_id(), 185);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[186];
    assert_eq!(cell.get_id(), 186);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[187];
    assert_eq!(cell.get_id(), 187);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[188];
    assert_eq!(cell.get_id(), 188);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[189];
    assert_eq!(cell.get_id(), 189);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[190];
    assert_eq!(cell.get_id(), 190);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[191];
    assert_eq!(cell.get_id(), 191);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[192];
    assert_eq!(cell.get_id(), 192);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[193];
    assert_eq!(cell.get_id(), 193);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[194];
    assert_eq!(cell.get_id(), 194);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[195];
    assert_eq!(cell.get_id(), 195);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[196];
    assert_eq!(cell.get_id(), 196);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[197];
    assert_eq!(cell.get_id(), 197);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[198];
    assert_eq!(cell.get_id(), 198);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[199];
    assert_eq!(cell.get_id(), 199);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[200];
    assert_eq!(cell.get_id(), 200);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[201];
    assert_eq!(cell.get_id(), 201);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[202];
    assert_eq!(cell.get_id(), 202);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[203];
    assert_eq!(cell.get_id(), 203);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[204];
    assert_eq!(cell.get_id(), 204);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[205];
    assert_eq!(cell.get_id(), 205);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[206];
    assert_eq!(cell.get_id(), 206);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[207];
    assert_eq!(cell.get_id(), 207);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[208];
    assert_eq!(cell.get_id(), 208);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[209];
    assert_eq!(cell.get_id(), 209);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[210];
    assert_eq!(cell.get_id(), 210);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[211];
    assert_eq!(cell.get_id(), 211);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[212];
    assert_eq!(cell.get_id(), 212);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[213];
    assert_eq!(cell.get_id(), 213);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[214];
    assert_eq!(cell.get_id(), 214);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[215];
    assert_eq!(cell.get_id(), 215);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[216];
    assert_eq!(cell.get_id(), 216);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[217];
    assert_eq!(cell.get_id(), 217);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[218];
    assert_eq!(cell.get_id(), 218);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[219];
    assert_eq!(cell.get_id(), 219);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[220];
    assert_eq!(cell.get_id(), 220);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[221];
    assert_eq!(cell.get_id(), 221);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[222];
    assert_eq!(cell.get_id(), 222);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[223];
    assert_eq!(cell.get_id(), 223);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[224];
    assert_eq!(cell.get_id(), 224);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[225];
    assert_eq!(cell.get_id(), 225);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[226];
    assert_eq!(cell.get_id(), 226);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[227];
    assert_eq!(cell.get_id(), 227);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[228];
    assert_eq!(cell.get_id(), 228);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[229];
    assert_eq!(cell.get_id(), 229);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[230];
    assert_eq!(cell.get_id(), 230);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[231];
    assert_eq!(cell.get_id(), 231);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[232];
    assert_eq!(cell.get_id(), 232);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[233];
    assert_eq!(cell.get_id(), 233);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[234];
    assert_eq!(cell.get_id(), 234);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[235];
    assert_eq!(cell.get_id(), 235);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[236];
    assert_eq!(cell.get_id(), 236);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[237];
    assert_eq!(cell.get_id(), 237);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[238];
    assert_eq!(cell.get_id(), 238);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[239];
    assert_eq!(cell.get_id(), 239);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[240];
    assert_eq!(cell.get_id(), 240);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[241];
    assert_eq!(cell.get_id(), 241);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[242];
    assert_eq!(cell.get_id(), 242);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[243];
    assert_eq!(cell.get_id(), 243);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[244];
    assert_eq!(cell.get_id(), 244);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[245];
    assert_eq!(cell.get_id(), 245);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[246];
    assert_eq!(cell.get_id(), 246);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[247];
    assert_eq!(cell.get_id(), 247);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[248];
    assert_eq!(cell.get_id(), 248);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[249];
    assert_eq!(cell.get_id(), 249);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[250];
    assert_eq!(cell.get_id(), 250);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[251];
    assert_eq!(cell.get_id(), 251);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[252];
    assert_eq!(cell.get_id(), 252);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[253];
    assert_eq!(cell.get_id(), 253);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[254];
    assert_eq!(cell.get_id(), 254);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[255];
    assert_eq!(cell.get_id(), 255);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[256];
    assert_eq!(cell.get_id(), 256);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[257];
    assert_eq!(cell.get_id(), 257);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[258];
    assert_eq!(cell.get_id(), 258);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[259];
    assert_eq!(cell.get_id(), 259);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[260];
    assert_eq!(cell.get_id(), 260);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[261];
    assert_eq!(cell.get_id(), 261);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[262];
    assert_eq!(cell.get_id(), 262);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[263];
    assert_eq!(cell.get_id(), 263);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[264];
    assert_eq!(cell.get_id(), 264);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[265];
    assert_eq!(cell.get_id(), 265);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[266];
    assert_eq!(cell.get_id(), 266);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[267];
    assert_eq!(cell.get_id(), 267);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[268];
    assert_eq!(cell.get_id(), 268);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[269];
    assert_eq!(cell.get_id(), 269);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    assert_eq!(output.vertices().len(), 526);
    assert_eq!(output.edges().len(), 1590);
    let v = output.vertices()[0];
    assert!(almost_equal(v.x(), -51017.5184453, v.y(), -49297.3063241));
    assert_eq!(v.get_incident_edge().unwrap().0, 11);
    let v = output.vertices()[1];
    assert!(almost_equal(v.x(), -50032.6041015, v.y(), -49536.8800834));
    assert_eq!(v.get_incident_edge().unwrap().0, 13);
    let v = output.vertices()[2];
    assert!(almost_equal(v.x(), -50013.6918239, v.y(), -49179.2536688));
    assert_eq!(v.get_incident_edge().unwrap().0, 19);
    let v = output.vertices()[3];
    assert!(almost_equal(v.x(), -50017.0154946, v.y(), -49513.6792610));
    assert_eq!(v.get_incident_edge().unwrap().0, 25);
    let v = output.vertices()[4];
    assert!(almost_equal(v.x(), -50013.2245431, v.y(), -49176.6057441));
    assert_eq!(v.get_incident_edge().unwrap().0, 27);
    let v = output.vertices()[5];
    assert!(almost_equal(v.x(), -50003.6119133, v.y(), -49198.8349504));
    assert_eq!(v.get_incident_edge().unwrap().0, 31);
    let v = output.vertices()[6];
    assert!(almost_equal(v.x(), -50021.6932499, v.y(), -49533.3595275));
    assert_eq!(v.get_incident_edge().unwrap().0, 33);
    let v = output.vertices()[7];
    assert!(almost_equal(v.x(), -50023.1307817, v.y(), -49538.4746669));
    assert_eq!(v.get_incident_edge().unwrap().0, 35);
    let v = output.vertices()[8];
    assert!(almost_equal(v.x(), -50014.0984921, v.y(), -49555.9020853));
    assert_eq!(v.get_incident_edge().unwrap().0, 45);
    let v = output.vertices()[9];
    assert!(almost_equal(v.x(), -50084.4560440, v.y(), -49099.4890110));
    assert_eq!(v.get_incident_edge().unwrap().0, 51);
    let v = output.vertices()[10];
    assert!(almost_equal(v.x(), -50235.1485346, v.y(), -49340.3725577));
    assert_eq!(v.get_incident_edge().unwrap().0, 55);
    let v = output.vertices()[11];
    assert!(almost_equal(v.x(), -50204.8611090, v.y(), -49341.4579107));
    assert_eq!(v.get_incident_edge().unwrap().0, 57);
    let v = output.vertices()[12];
    assert!(almost_equal(v.x(), -50357.6507472, v.y(), -49804.0417100));
    assert_eq!(v.get_incident_edge().unwrap().0, 59);
    let v = output.vertices()[13];
    assert!(almost_equal(v.x(), -49973.3798450, v.y(), -49619.3591731));
    assert_eq!(v.get_incident_edge().unwrap().0, 65);
    let v = output.vertices()[14];
    assert!(almost_equal(v.x(), -50014.3376063, v.y(), -49128.5145281));
    assert_eq!(v.get_incident_edge().unwrap().0, 67);
    let v = output.vertices()[15];
    assert!(almost_equal(v.x(), -49954.1638210, v.y(), -49625.6459352));
    assert_eq!(v.get_incident_edge().unwrap().0, 73);
    let v = output.vertices()[16];
    assert!(almost_equal(v.x(), -49977.1604269, v.y(), -49205.0149076));
    assert_eq!(v.get_incident_edge().unwrap().0, 75);
    let v = output.vertices()[17];
    assert!(almost_equal(v.x(), -50139.1861730, v.y(), -49342.4912983));
    assert_eq!(v.get_incident_edge().unwrap().0, 77);
    let v = output.vertices()[18];
    assert!(almost_equal(v.x(), -49972.5718535, v.y(), -49609.8535782));
    assert_eq!(v.get_incident_edge().unwrap().0, 85);
    let v = output.vertices()[19];
    assert!(almost_equal(v.x(), -50124.2103079, v.y(), -49342.5240856));
    assert_eq!(v.get_incident_edge().unwrap().0, 87);
    let v = output.vertices()[20];
    assert!(almost_equal(v.x(), -49983.6175852, v.y(), -49138.8811095));
    assert_eq!(v.get_incident_edge().unwrap().0, 89);
    let v = output.vertices()[21];
    assert!(almost_equal(v.x(), -49950.9699600, v.y(), -49632.5487962));
    assert_eq!(v.get_incident_edge().unwrap().0, 91);
    let v = output.vertices()[22];
    assert!(almost_equal(v.x(), -50069.7139451, v.y(), -49357.2723236));
    assert_eq!(v.get_incident_edge().unwrap().0, 93);
    let v = output.vertices()[23];
    assert!(almost_equal(v.x(), -50186.8321200, v.y(), -49785.9579068));
    assert_eq!(v.get_incident_edge().unwrap().0, 95);
    let v = output.vertices()[24];
    assert!(almost_equal(v.x(), -49972.6642612, v.y(), -49585.0198830));
    assert_eq!(v.get_incident_edge().unwrap().0, 97);
    let v = output.vertices()[25];
    assert!(almost_equal(v.x(), -50003.2839917, v.y(), -49371.9740346));
    assert_eq!(v.get_incident_edge().unwrap().0, 101);
    let v = output.vertices()[26];
    assert!(almost_equal(v.x(), -49941.4431487, v.y(), -49139.4752187));
    assert_eq!(v.get_incident_edge().unwrap().0, 103);
    let v = output.vertices()[27];
    assert!(almost_equal(v.x(), -49940.5265301, v.y(), -49645.9547138));
    assert_eq!(v.get_incident_edge().unwrap().0, 105);
    let v = output.vertices()[28];
    assert!(almost_equal(v.x(), -49926.2788104, v.y(), -49174.5427509));
    assert_eq!(v.get_incident_edge().unwrap().0, 111);
    let v = output.vertices()[29];
    assert!(almost_equal(v.x(), -49951.2823074, v.y(), -49215.9368307));
    assert_eq!(v.get_incident_edge().unwrap().0, 117);
    let v = output.vertices()[30];
    assert!(almost_equal(v.x(), -50089.4190083, v.y(), -49783.7793388));
    assert_eq!(v.get_incident_edge().unwrap().0, 127);
    let v = output.vertices()[31];
    assert!(almost_equal(v.x(), -49913.3129673, v.y(), -49619.6841487));
    assert_eq!(v.get_incident_edge().unwrap().0, 129);
    let v = output.vertices()[32];
    assert!(almost_equal(v.x(), -49948.4225617, v.y(), -49222.8695475));
    assert_eq!(v.get_incident_edge().unwrap().0, 131);
    let v = output.vertices()[33];
    assert!(almost_equal(v.x(), -49903.0399344, v.y(), -49607.4849221));
    assert_eq!(v.get_incident_edge().unwrap().0, 133);
    let v = output.vertices()[34];
    assert!(almost_equal(v.x(), -49992.0357143, v.y(), -49718.8571429));
    assert_eq!(v.get_incident_edge().unwrap().0, 135);
    let v = output.vertices()[35];
    assert!(almost_equal(v.x(), -49943.9095850, v.y(), -49538.9303360));
    assert_eq!(v.get_incident_edge().unwrap().0, 137);
    let v = output.vertices()[36];
    assert!(almost_equal(v.x(), -50045.3324636, v.y(), -49802.7078010));
    assert_eq!(v.get_incident_edge().unwrap().0, 139);
    let v = output.vertices()[37];
    assert!(almost_equal(v.x(), -49943.3253052, v.y(), -49536.4394589));
    assert_eq!(v.get_incident_edge().unwrap().0, 141);
    let v = output.vertices()[38];
    assert!(almost_equal(v.x(), -49916.3085907, v.y(), -49314.8457047));
    assert_eq!(v.get_incident_edge().unwrap().0, 147);
    let v = output.vertices()[39];
    assert!(almost_equal(v.x(), -50003.2096901, v.y(), -49818.0018205));
    assert_eq!(v.get_incident_edge().unwrap().0, 151);
    let v = output.vertices()[40];
    assert!(almost_equal(v.x(), -49943.4511364, v.y(), -49493.4490625));
    assert_eq!(v.get_incident_edge().unwrap().0, 153);
    let v = output.vertices()[41];
    assert!(almost_equal(v.x(), -49933.9351733, v.y(), -49411.1574505));
    assert_eq!(v.get_incident_edge().unwrap().0, 159);
    let v = output.vertices()[42];
    assert!(almost_equal(v.x(), -49930.6725146, v.y(), -49422.9532164));
    assert_eq!(v.get_incident_edge().unwrap().0, 161);
    let v = output.vertices()[43];
    assert!(almost_equal(v.x(), -49919.7607963, v.y(), -49292.0050177));
    assert_eq!(v.get_incident_edge().unwrap().0, 163);
    let v = output.vertices()[44];
    assert!(almost_equal(v.x(), -49894.3719466, v.y(), -49579.0670832));
    assert_eq!(v.get_incident_edge().unwrap().0, 165);
    let v = output.vertices()[45];
    assert!(almost_equal(v.x(), -49894.3500000, v.y(), -49579.0500000));
    assert_eq!(v.get_incident_edge().unwrap().0, 167);
    let v = output.vertices()[46];
    assert!(almost_equal(v.x(), -49940.0545025, v.y(), -49255.0965820));
    assert_eq!(v.get_incident_edge().unwrap().0, 169);
    let v = output.vertices()[47];
    assert!(almost_equal(v.x(), -49932.2338127, v.y(), -49481.5694296));
    assert_eq!(v.get_incident_edge().unwrap().0, 171);
    let v = output.vertices()[48];
    assert!(almost_equal(v.x(), -49936.3780781, v.y(), -49495.1302407));
    assert_eq!(v.get_incident_edge().unwrap().0, 175);
    let v = output.vertices()[49];
    assert!(almost_equal(v.x(), -49936.3646158, v.y(), -49495.1949451));
    assert_eq!(v.get_incident_edge().unwrap().0, 177);
    let v = output.vertices()[50];
    assert!(almost_equal(v.x(), -49871.1174391, v.y(), -49640.1379443));
    assert_eq!(v.get_incident_edge().unwrap().0, 183);
    let v = output.vertices()[51];
    assert!(almost_equal(v.x(), -49932.9677272, v.y(), -49508.1361197));
    assert_eq!(v.get_incident_edge().unwrap().0, 185);
    let v = output.vertices()[52];
    assert!(almost_equal(v.x(), -49928.6653790, v.y(), -49261.1913363));
    assert_eq!(v.get_incident_edge().unwrap().0, 187);
    let v = output.vertices()[53];
    assert!(almost_equal(v.x(), -49939.5709738, v.y(), -49836.5640285));
    assert_eq!(v.get_incident_edge().unwrap().0, 189);
    let v = output.vertices()[54];
    assert!(almost_equal(v.x(), -49920.6387863, v.y(), -49674.3282820));
    assert_eq!(v.get_incident_edge().unwrap().0, 191);
    let v = output.vertices()[55];
    assert!(almost_equal(v.x(), -49879.7806268, v.y(), -49610.6031613));
    assert_eq!(v.get_incident_edge().unwrap().0, 193);
    let v = output.vertices()[56];
    assert!(almost_equal(v.x(), -49872.1727435, v.y(), -49414.5961062));
    assert_eq!(v.get_incident_edge().unwrap().0, 195);
    let v = output.vertices()[57];
    assert!(almost_equal(v.x(), -49868.1846544, v.y(), -49422.7938181));
    assert_eq!(v.get_incident_edge().unwrap().0, 201);
    let v = output.vertices()[58];
    assert!(almost_equal(v.x(), -49867.4772720, v.y(), -49250.1170661));
    assert_eq!(v.get_incident_edge().unwrap().0, 203);
    let v = output.vertices()[59];
    assert!(almost_equal(v.x(), -49920.9346972, v.y(), -49687.6420622));
    assert_eq!(v.get_incident_edge().unwrap().0, 205);
    let v = output.vertices()[60];
    assert!(almost_equal(v.x(), -49854.2474940, v.y(), -49235.8888143));
    assert_eq!(v.get_incident_edge().unwrap().0, 209);
    let v = output.vertices()[61];
    assert!(almost_equal(v.x(), -49854.2499937, v.y(), -49235.8750032));
    assert_eq!(v.get_incident_edge().unwrap().0, 211);
    let v = output.vertices()[62];
    assert!(almost_equal(v.x(), -49857.8910179, v.y(), -49286.4973582));
    assert_eq!(v.get_incident_edge().unwrap().0, 219);
    let v = output.vertices()[63];
    assert!(almost_equal(v.x(), -49884.1734209, v.y(), -49867.2253754));
    assert_eq!(v.get_incident_edge().unwrap().0, 221);
    let v = output.vertices()[64];
    assert!(almost_equal(v.x(), -49862.7970591, v.y(), -49341.6014705));
    assert_eq!(v.get_incident_edge().unwrap().0, 223);
    let v = output.vertices()[65];
    assert!(almost_equal(v.x(), -49829.9768022, v.y(), -49674.9724526));
    assert_eq!(v.get_incident_edge().unwrap().0, 227);
    let v = output.vertices()[66];
    assert!(almost_equal(v.x(), -49864.7698408, v.y(), -49353.2664770));
    assert_eq!(v.get_incident_edge().unwrap().0, 229);
    let v = output.vertices()[67];
    assert!(almost_equal(v.x(), -49851.1107603, v.y(), -49376.8561060));
    assert_eq!(v.get_incident_edge().unwrap().0, 233);
    let v = output.vertices()[68];
    assert!(almost_equal(v.x(), -49876.1795213, v.y(), -49145.1742021));
    assert_eq!(v.get_incident_edge().unwrap().0, 235);
    let v = output.vertices()[69];
    assert!(almost_equal(v.x(), -49850.3862369, v.y(), -49607.3262176));
    assert_eq!(v.get_incident_edge().unwrap().0, 237);
    let v = output.vertices()[70];
    assert!(almost_equal(v.x(), -49865.1455017, v.y(), -49716.7352832));
    assert_eq!(v.get_incident_edge().unwrap().0, 239);
    let v = output.vertices()[71];
    assert!(almost_equal(v.x(), -49863.3195029, v.y(), -49718.4688264));
    assert_eq!(v.get_incident_edge().unwrap().0, 241);
    let v = output.vertices()[72];
    assert!(almost_equal(v.x(), -49837.3039419, v.y(), -49393.2271784));
    assert_eq!(v.get_incident_edge().unwrap().0, 243);
    let v = output.vertices()[73];
    assert!(almost_equal(v.x(), -49858.4471725, v.y(), -49723.6622023));
    assert_eq!(v.get_incident_edge().unwrap().0, 245);
    let v = output.vertices()[74];
    assert!(almost_equal(v.x(), -49840.0562016, v.y(), -49603.6533161));
    assert_eq!(v.get_incident_edge().unwrap().0, 249);
    let v = output.vertices()[75];
    assert!(almost_equal(v.x(), -49850.0044543, v.y(), -49544.5590200));
    assert_eq!(v.get_incident_edge().unwrap().0, 251);
    let v = output.vertices()[76];
    assert!(almost_equal(v.x(), -49836.1587051, v.y(), -49748.9959393));
    assert_eq!(v.get_incident_edge().unwrap().0, 253);
    let v = output.vertices()[77];
    assert!(almost_equal(v.x(), -49837.8130452, v.y(), -49914.3457573));
    assert_eq!(v.get_incident_edge().unwrap().0, 259);
    let v = output.vertices()[78];
    assert!(almost_equal(v.x(), -49815.0072222, v.y(), -49621.8850320));
    assert_eq!(v.get_incident_edge().unwrap().0, 265);
    let v = output.vertices()[79];
    assert!(almost_equal(v.x(), -49818.2480418, v.y(), -49478.3211488));
    assert_eq!(v.get_incident_edge().unwrap().0, 271);
    let v = output.vertices()[80];
    assert!(almost_equal(v.x(), -49807.5833333, v.y(), -49259.2083333));
    assert_eq!(v.get_incident_edge().unwrap().0, 273);
    let v = output.vertices()[81];
    assert!(almost_equal(v.x(), -49789.7797652, v.y(), -49672.6026788));
    assert_eq!(v.get_incident_edge().unwrap().0, 277);
    let v = output.vertices()[82];
    assert!(almost_equal(v.x(), -49824.0839517, v.y(), -49597.2600880));
    assert_eq!(v.get_incident_edge().unwrap().0, 279);
    let v = output.vertices()[83];
    assert!(almost_equal(v.x(), -49817.9321540, v.y(), -49454.8557198));
    assert_eq!(v.get_incident_edge().unwrap().0, 281);
    let v = output.vertices()[84];
    assert!(almost_equal(v.x(), -49781.5083096, v.y(), -49684.8034113));
    assert_eq!(v.get_incident_edge().unwrap().0, 283);
    let v = output.vertices()[85];
    assert!(almost_equal(v.x(), -49753.1201342, v.y(), -49487.4896637));
    assert_eq!(v.get_incident_edge().unwrap().0, 289);
    let v = output.vertices()[86];
    assert!(almost_equal(v.x(), -49814.7864992, v.y(), -49515.7058085));
    assert_eq!(v.get_incident_edge().unwrap().0, 291);
    let v = output.vertices()[87];
    assert!(almost_equal(v.x(), -49752.3756792, v.y(), -49478.2584217));
    assert_eq!(v.get_incident_edge().unwrap().0, 293);
    let v = output.vertices()[88];
    assert!(almost_equal(v.x(), -49798.5856378, v.y(), -49425.2118644));
    assert_eq!(v.get_incident_edge().unwrap().0, 299);
    let v = output.vertices()[89];
    assert!(almost_equal(v.x(), -49760.1301869, v.y(), -49596.4649984));
    assert_eq!(v.get_incident_edge().unwrap().0, 301);
    let v = output.vertices()[90];
    assert!(almost_equal(v.x(), -49843.9543860, v.y(), -49096.9801170));
    assert_eq!(v.get_incident_edge().unwrap().0, 311);
    let v = output.vertices()[91];
    assert!(almost_equal(v.x(), -49837.8603701, v.y(), -49120.5197467));
    assert_eq!(v.get_incident_edge().unwrap().0, 313);
    let v = output.vertices()[92];
    assert!(almost_equal(v.x(), -49870.8392698, v.y(), -48979.5124666));
    assert_eq!(v.get_incident_edge().unwrap().0, 315);
    let v = output.vertices()[93];
    assert!(almost_equal(v.x(), -49747.7838362, v.y(), -49618.0067598));
    assert_eq!(v.get_incident_edge().unwrap().0, 317);
    let v = output.vertices()[94];
    assert!(almost_equal(v.x(), -49814.9090877, v.y(), -49549.3886577));
    assert_eq!(v.get_incident_edge().unwrap().0, 319);
    let v = output.vertices()[95];
    assert!(almost_equal(v.x(), -49814.7196240, v.y(), -49549.6580888));
    assert_eq!(v.get_incident_edge().unwrap().0, 321);
    let v = output.vertices()[96];
    assert!(almost_equal(v.x(), -49809.5414336, v.y(), -49550.4614239));
    assert_eq!(v.get_incident_edge().unwrap().0, 323);
    let v = output.vertices()[97];
    assert!(almost_equal(v.x(), -49845.9407968, v.y(), -49061.5557900));
    assert_eq!(v.get_incident_edge().unwrap().0, 325);
    let v = output.vertices()[98];
    assert!(almost_equal(v.x(), -49819.6511533, v.y(), -49149.8593194));
    assert_eq!(v.get_incident_edge().unwrap().0, 327);
    let v = output.vertices()[99];
    assert!(almost_equal(v.x(), -49855.3175645, v.y(), -50022.4171372));
    assert_eq!(v.get_incident_edge().unwrap().0, 329);
    let v = output.vertices()[100];
    assert!(almost_equal(v.x(), -49846.1164661, v.y(), -49058.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 331);
    let v = output.vertices()[101];
    assert!(almost_equal(v.x(), -49790.8615385, v.y(), -49806.4153846));
    assert_eq!(v.get_incident_edge().unwrap().0, 337);
    let v = output.vertices()[102];
    assert!(almost_equal(v.x(), -49797.0972477, v.y(), -49779.4858493));
    assert_eq!(v.get_incident_edge().unwrap().0, 341);
    let v = output.vertices()[103];
    assert!(almost_equal(v.x(), -49738.8669729, v.y(), -49669.6654023));
    assert_eq!(v.get_incident_edge().unwrap().0, 347);
    let v = output.vertices()[104];
    assert!(almost_equal(v.x(), -49759.5714286, v.y(), -49147.2857143));
    assert_eq!(v.get_incident_edge().unwrap().0, 353);
    let v = output.vertices()[105];
    assert!(almost_equal(v.x(), -49741.8276917, v.y(), -49458.6652863));
    assert_eq!(v.get_incident_edge().unwrap().0, 355);
    let v = output.vertices()[106];
    assert!(almost_equal(v.x(), -49787.3333333, v.y(), -49161.1666667));
    assert_eq!(v.get_incident_edge().unwrap().0, 357);
    let v = output.vertices()[107];
    assert!(almost_equal(v.x(), -49738.1685566, v.y(), -49678.0042523));
    assert_eq!(v.get_incident_edge().unwrap().0, 359);
    let v = output.vertices()[108];
    assert!(almost_equal(v.x(), -49750.8603111, v.y(), -49554.3453341));
    assert_eq!(v.get_incident_edge().unwrap().0, 361);
    let v = output.vertices()[109];
    assert!(almost_equal(v.x(), -49799.5375106, v.y(), -49167.0557162));
    assert_eq!(v.get_incident_edge().unwrap().0, 363);
    let v = output.vertices()[110];
    assert!(almost_equal(v.x(), -49721.9346103, v.y(), -49681.9022457));
    assert_eq!(v.get_incident_edge().unwrap().0, 367);
    let v = output.vertices()[111];
    assert!(almost_equal(v.x(), -49754.3785047, v.y(), -49714.4766355));
    assert_eq!(v.get_incident_edge().unwrap().0, 369);
    let v = output.vertices()[112];
    assert!(almost_equal(v.x(), -49714.8607595, v.y(), -49684.8496835));
    assert_eq!(v.get_incident_edge().unwrap().0, 371);
    let v = output.vertices()[113];
    assert!(almost_equal(v.x(), -49780.8938740, v.y(), -49209.1117343));
    assert_eq!(v.get_incident_edge().unwrap().0, 375);
    let v = output.vertices()[114];
    assert!(almost_equal(v.x(), -49766.0088050, v.y(), -49867.5283019));
    assert_eq!(v.get_incident_edge().unwrap().0, 383);
    let v = output.vertices()[115];
    assert!(almost_equal(v.x(), -49776.2486957, v.y(), -49855.7130435));
    assert_eq!(v.get_incident_edge().unwrap().0, 385);
    let v = output.vertices()[116];
    assert!(almost_equal(v.x(), -49775.5084746, v.y(), -49852.4745763));
    assert_eq!(v.get_incident_edge().unwrap().0, 387);
    let v = output.vertices()[117];
    assert!(almost_equal(v.x(), -49752.5000000, v.y(), -49058.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 393);
    let v = output.vertices()[118];
    assert!(almost_equal(v.x(), -49737.9160948, v.y(), -49843.7201865));
    assert_eq!(v.get_incident_edge().unwrap().0, 397);
    let v = output.vertices()[119];
    assert!(almost_equal(v.x(), -49720.8579882, v.y(), -49082.7633136));
    assert_eq!(v.get_incident_edge().unwrap().0, 399);
    let v = output.vertices()[120];
    assert!(almost_equal(v.x(), -49719.6194252, v.y(), -49083.7065980));
    assert_eq!(v.get_incident_edge().unwrap().0, 403);
    let v = output.vertices()[121];
    assert!(almost_equal(v.x(), -49886.1186109, v.y(), -48879.6090828));
    assert_eq!(v.get_incident_edge().unwrap().0, 405);
    let v = output.vertices()[122];
    assert!(almost_equal(v.x(), -49749.8526603, v.y(), -49744.6489313));
    assert_eq!(v.get_incident_edge().unwrap().0, 407);
    let v = output.vertices()[123];
    assert!(almost_equal(v.x(), -49767.4466175, v.y(), -49257.3839372));
    assert_eq!(v.get_incident_edge().unwrap().0, 411);
    let v = output.vertices()[124];
    assert!(almost_equal(v.x(), -49698.1043172, v.y(), -49682.4208822));
    assert_eq!(v.get_incident_edge().unwrap().0, 417);
    let v = output.vertices()[125];
    assert!(almost_equal(v.x(), -49697.8343471, v.y(), -49674.8617194));
    assert_eq!(v.get_incident_edge().unwrap().0, 419);
    let v = output.vertices()[126];
    assert!(almost_equal(v.x(), -49766.1628033, v.y(), -49259.5184716));
    assert_eq!(v.get_incident_edge().unwrap().0, 421);
    let v = output.vertices()[127];
    assert!(almost_equal(v.x(), -49742.7726454, v.y(), -49745.0954188));
    assert_eq!(v.get_incident_edge().unwrap().0, 423);
    let v = output.vertices()[128];
    assert!(almost_equal(v.x(), -49697.7917422, v.y(), -49149.9739678));
    assert_eq!(v.get_incident_edge().unwrap().0, 425);
    let v = output.vertices()[129];
    assert!(almost_equal(v.x(), -49706.5583333, v.y(), -49901.5000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 427);
    let v = output.vertices()[130];
    assert!(almost_equal(v.x(), -49690.9550566, v.y(), -49688.4619856));
    assert_eq!(v.get_incident_edge().unwrap().0, 429);
    let v = output.vertices()[131];
    assert!(almost_equal(v.x(), -49713.4443815, v.y(), -49833.9967185));
    assert_eq!(v.get_incident_edge().unwrap().0, 431);
    let v = output.vertices()[132];
    assert!(almost_equal(v.x(), -49690.4359258, v.y(), -49688.7532868));
    assert_eq!(v.get_incident_edge().unwrap().0, 433);
    let v = output.vertices()[133];
    assert!(almost_equal(v.x(), -49735.9013717, v.y(), -49745.3414435));
    assert_eq!(v.get_incident_edge().unwrap().0, 435);
    let v = output.vertices()[134];
    assert!(almost_equal(v.x(), -49695.3934221, v.y(), -49153.6557035));
    assert_eq!(v.get_incident_edge().unwrap().0, 437);
    let v = output.vertices()[135];
    assert!(almost_equal(v.x(), -49710.3746398, v.y(), -49029.9164265));
    assert_eq!(v.get_incident_edge().unwrap().0, 439);
    let v = output.vertices()[136];
    assert!(almost_equal(v.x(), -49701.1401053, v.y(), -49515.4731732));
    assert_eq!(v.get_incident_edge().unwrap().0, 441);
    let v = output.vertices()[137];
    assert!(almost_equal(v.x(), -49693.3764259, v.y(), -49520.4676806));
    assert_eq!(v.get_incident_edge().unwrap().0, 445);
    let v = output.vertices()[138];
    assert!(almost_equal(v.x(), -49710.1906751, v.y(), -49745.5481909));
    assert_eq!(v.get_incident_edge().unwrap().0, 447);
    let v = output.vertices()[139];
    assert!(almost_equal(v.x(), -49688.2462845, v.y(), -49456.8539278));
    assert_eq!(v.get_incident_edge().unwrap().0, 453);
    let v = output.vertices()[140];
    assert!(almost_equal(v.x(), -49673.7955962, v.y(), -49117.6593269));
    assert_eq!(v.get_incident_edge().unwrap().0, 455);
    let v = output.vertices()[141];
    assert!(almost_equal(v.x(), -49680.8081395, v.y(), -49901.5000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 459);
    let v = output.vertices()[142];
    assert!(almost_equal(v.x(), -49664.3016090, v.y(), -49127.0358227));
    assert_eq!(v.get_incident_edge().unwrap().0, 461);
    let v = output.vertices()[143];
    assert!(almost_equal(v.x(), -49720.1691831, v.y(), -49404.5694931));
    assert_eq!(v.get_incident_edge().unwrap().0, 463);
    let v = output.vertices()[144];
    assert!(almost_equal(v.x(), -49679.6377557, v.y(), -49740.8117497));
    assert_eq!(v.get_incident_edge().unwrap().0, 465);
    let v = output.vertices()[145];
    assert!(almost_equal(v.x(), -49682.9409255, v.y(), -49573.9153266));
    assert_eq!(v.get_incident_edge().unwrap().0, 469);
    let v = output.vertices()[146];
    assert!(almost_equal(v.x(), -49723.6105991, v.y(), -49385.6417051));
    assert_eq!(v.get_incident_edge().unwrap().0, 471);
    let v = output.vertices()[147];
    assert!(almost_equal(v.x(), -49628.7402288, v.y(), -49782.1687097));
    assert_eq!(v.get_incident_edge().unwrap().0, 475);
    let v = output.vertices()[148];
    assert!(almost_equal(v.x(), -49623.7279550, v.y(), -49779.9183865));
    assert_eq!(v.get_incident_edge().unwrap().0, 477);
    let v = output.vertices()[149];
    assert!(almost_equal(v.x(), -49633.8049489, v.y(), -49749.3724985));
    assert_eq!(v.get_incident_edge().unwrap().0, 479);
    let v = output.vertices()[150];
    assert!(almost_equal(v.x(), -49622.0095238, v.y(), -49753.3949206));
    assert_eq!(v.get_incident_edge().unwrap().0, 487);
    let v = output.vertices()[151];
    assert!(almost_equal(v.x(), -49713.7816989, v.y(), -49310.2189257));
    assert_eq!(v.get_incident_edge().unwrap().0, 489);
    let v = output.vertices()[152];
    assert!(almost_equal(v.x(), -49716.9516469, v.y(), -49314.5523096));
    assert_eq!(v.get_incident_edge().unwrap().0, 491);
    let v = output.vertices()[153];
    assert!(almost_equal(v.x(), -49711.1827144, v.y(), -49362.0569693));
    assert_eq!(v.get_incident_edge().unwrap().0, 493);
    let v = output.vertices()[154];
    assert!(almost_equal(v.x(), -49709.8528754, v.y(), -49335.3096692));
    assert_eq!(v.get_incident_edge().unwrap().0, 495);
    let v = output.vertices()[155];
    assert!(almost_equal(v.x(), -49656.9407552, v.y(), -49014.3599845));
    assert_eq!(v.get_incident_edge().unwrap().0, 497);
    let v = output.vertices()[156];
    assert!(almost_equal(v.x(), -49931.8340754, v.y(), -48708.9182250));
    assert_eq!(v.get_incident_edge().unwrap().0, 499);
    let v = output.vertices()[157];
    assert!(almost_equal(v.x(), -49705.3100052, v.y(), -49350.2019273));
    assert_eq!(v.get_incident_edge().unwrap().0, 501);
    let v = output.vertices()[158];
    assert!(almost_equal(v.x(), -49646.4935994, v.y(), -49216.9768449));
    assert_eq!(v.get_incident_edge().unwrap().0, 509);
    let v = output.vertices()[159];
    assert!(almost_equal(v.x(), -49655.5484639, v.y(), -49366.9972250));
    assert_eq!(v.get_incident_edge().unwrap().0, 511);
    let v = output.vertices()[160];
    assert!(almost_equal(v.x(), -49648.3572595, v.y(), -49013.8085162));
    assert_eq!(v.get_incident_edge().unwrap().0, 517);
    let v = output.vertices()[161];
    assert!(almost_equal(v.x(), -49639.8316498, v.y(), -49192.8762626));
    assert_eq!(v.get_incident_edge().unwrap().0, 519);
    let v = output.vertices()[162];
    assert!(almost_equal(v.x(), -49639.4397397, v.y(), -49191.4299675));
    assert_eq!(v.get_incident_edge().unwrap().0, 521);
    let v = output.vertices()[163];
    assert!(almost_equal(v.x(), -49638.9468445, v.y(), -49189.5739391));
    assert_eq!(v.get_incident_edge().unwrap().0, 523);
    let v = output.vertices()[164];
    assert!(almost_equal(v.x(), -49626.0024898, v.y(), -49048.0018673));
    assert_eq!(v.get_incident_edge().unwrap().0, 527);
    let v = output.vertices()[165];
    assert!(almost_equal(v.x(), -49861.5263238, v.y(), -50208.1625210));
    assert_eq!(v.get_incident_edge().unwrap().0, 529);
    let v = output.vertices()[166];
    assert!(almost_equal(v.x(), -49624.3858820, v.y(), -48971.8673378));
    assert_eq!(v.get_incident_edge().unwrap().0, 531);
    let v = output.vertices()[167];
    assert!(almost_equal(v.x(), -49586.4034707, v.y(), -49260.3752711));
    assert_eq!(v.get_incident_edge().unwrap().0, 539);
    let v = output.vertices()[168];
    assert!(almost_equal(v.x(), -49627.5359391, v.y(), -49175.2115281));
    assert_eq!(v.get_incident_edge().unwrap().0, 541);
    let v = output.vertices()[169];
    assert!(almost_equal(v.x(), -49590.8000000, v.y(), -49770.0400000));
    assert_eq!(v.get_incident_edge().unwrap().0, 543);
    let v = output.vertices()[170];
    assert!(almost_equal(v.x(), -49620.4673046, v.y(), -49169.8995215));
    assert_eq!(v.get_incident_edge().unwrap().0, 545);
    let v = output.vertices()[171];
    assert!(almost_equal(v.x(), -49609.3558070, v.y(), -49066.2447226));
    assert_eq!(v.get_incident_edge().unwrap().0, 547);
    let v = output.vertices()[172];
    assert!(almost_equal(v.x(), -49631.8755842, v.y(), -49616.5819608));
    assert_eq!(v.get_incident_edge().unwrap().0, 549);
    let v = output.vertices()[173];
    assert!(almost_equal(v.x(), -49579.2033898, v.y(), -49371.7372881));
    assert_eq!(v.get_incident_edge().unwrap().0, 551);
    let v = output.vertices()[174];
    assert!(almost_equal(v.x(), -49635.4718692, v.y(), -49596.4754588));
    assert_eq!(v.get_incident_edge().unwrap().0, 553);
    let v = output.vertices()[175];
    assert!(almost_equal(v.x(), -49600.5776218, v.y(), -49153.9483898));
    assert_eq!(v.get_incident_edge().unwrap().0, 561);
    let v = output.vertices()[176];
    assert!(almost_equal(v.x(), -49583.2317040, v.y(), -49700.6107661));
    assert_eq!(v.get_incident_edge().unwrap().0, 563);
    let v = output.vertices()[177];
    assert!(almost_equal(v.x(), -49580.2477096, v.y(), -49702.5847134));
    assert_eq!(v.get_incident_edge().unwrap().0, 565);
    let v = output.vertices()[178];
    assert!(almost_equal(v.x(), -49632.4625756, v.y(), -49583.7441898));
    assert_eq!(v.get_incident_edge().unwrap().0, 567);
    let v = output.vertices()[179];
    assert!(almost_equal(v.x(), -49585.5946065, v.y(), -49106.0027406));
    assert_eq!(v.get_incident_edge().unwrap().0, 571);
    let v = output.vertices()[180];
    assert!(almost_equal(v.x(), -49572.8181818, v.y(), -49773.6363636));
    assert_eq!(v.get_incident_edge().unwrap().0, 577);
    let v = output.vertices()[181];
    assert!(almost_equal(v.x(), -49571.3727599, v.y(), -49797.2449223));
    assert_eq!(v.get_incident_edge().unwrap().0, 579);
    let v = output.vertices()[182];
    assert!(almost_equal(v.x(), -49623.7100881, v.y(), -49572.7174339));
    assert_eq!(v.get_incident_edge().unwrap().0, 581);
    let v = output.vertices()[183];
    assert!(almost_equal(v.x(), -49563.6403509, v.y(), -49107.4802632));
    assert_eq!(v.get_incident_edge().unwrap().0, 583);
    let v = output.vertices()[184];
    assert!(almost_equal(v.x(), -49560.5892518, v.y(), -49107.9093240));
    assert_eq!(v.get_incident_edge().unwrap().0, 585);
    let v = output.vertices()[185];
    assert!(almost_equal(v.x(), -49575.6642702, v.y(), -49835.8690913));
    assert_eq!(v.get_incident_edge().unwrap().0, 587);
    let v = output.vertices()[186];
    assert!(almost_equal(v.x(), -49562.6878520, v.y(), -49717.3183829));
    assert_eq!(v.get_incident_edge().unwrap().0, 589);
    let v = output.vertices()[187];
    assert!(almost_equal(v.x(), -49560.1428571, v.y(), -49748.2857143));
    assert_eq!(v.get_incident_edge().unwrap().0, 595);
    let v = output.vertices()[188];
    assert!(almost_equal(v.x(), -49518.3018868, v.y(), -49073.4764151));
    assert_eq!(v.get_incident_edge().unwrap().0, 601);
    let v = output.vertices()[189];
    assert!(almost_equal(v.x(), -49559.5988197, v.y(), -49720.5398024));
    assert_eq!(v.get_incident_edge().unwrap().0, 603);
    let v = output.vertices()[190];
    assert!(almost_equal(v.x(), -49514.5179949, v.y(), -49080.6658098));
    assert_eq!(v.get_incident_edge().unwrap().0, 605);
    let v = output.vertices()[191];
    assert!(almost_equal(v.x(), -49590.9589778, v.y(), -49892.9054344));
    assert_eq!(v.get_incident_edge().unwrap().0, 609);
    let v = output.vertices()[192];
    assert!(almost_equal(v.x(), -49519.1124219, v.y(), -49168.7866065));
    assert_eq!(v.get_incident_edge().unwrap().0, 615);
    let v = output.vertices()[193];
    assert!(almost_equal(v.x(), -49531.3952334, v.y(), -49186.4431480));
    assert_eq!(v.get_incident_edge().unwrap().0, 617);
    let v = output.vertices()[194];
    assert!(almost_equal(v.x(), -49514.3567799, v.y(), -49162.7851513));
    assert_eq!(v.get_incident_edge().unwrap().0, 619);
    let v = output.vertices()[195];
    assert!(almost_equal(v.x(), -49596.7365647, v.y(), -49921.3338517));
    assert_eq!(v.get_incident_edge().unwrap().0, 621);
    let v = output.vertices()[196];
    assert!(almost_equal(v.x(), -49597.7567807, v.y(), -49552.5142604));
    assert_eq!(v.get_incident_edge().unwrap().0, 623);
    let v = output.vertices()[197];
    assert!(almost_equal(v.x(), -49509.9591131, v.y(), -49097.0708892));
    assert_eq!(v.get_incident_edge().unwrap().0, 625);
    let v = output.vertices()[198];
    assert!(almost_equal(v.x(), -49595.8004371, v.y(), -49551.6135676));
    assert_eq!(v.get_incident_edge().unwrap().0, 627);
    let v = output.vertices()[199];
    assert!(almost_equal(v.x(), -49506.4390244, v.y(), -49766.1869919));
    assert_eq!(v.get_incident_edge().unwrap().0, 629);
    let v = output.vertices()[200];
    assert!(almost_equal(v.x(), -49489.9483334, v.y(), -49855.6636797));
    assert_eq!(v.get_incident_edge().unwrap().0, 635);
    let v = output.vertices()[201];
    assert!(almost_equal(v.x(), -49499.3191751, v.y(), -49778.3683331));
    assert_eq!(v.get_incident_edge().unwrap().0, 639);
    let v = output.vertices()[202];
    assert!(almost_equal(v.x(), -49499.4123463, v.y(), -49777.5414758));
    assert_eq!(v.get_incident_edge().unwrap().0, 641);
    let v = output.vertices()[203];
    assert!(almost_equal(v.x(), -49487.1983696, v.y(), -49864.4947917));
    assert_eq!(v.get_incident_edge().unwrap().0, 643);
    let v = output.vertices()[204];
    assert!(almost_equal(v.x(), -49500.3415667, v.y(), -49768.6209601));
    assert_eq!(v.get_incident_edge().unwrap().0, 645);
    let v = output.vertices()[205];
    assert!(almost_equal(v.x(), -49513.2489422, v.y(), -49701.8711017));
    assert_eq!(v.get_incident_edge().unwrap().0, 647);
    let v = output.vertices()[206];
    assert!(almost_equal(v.x(), -49474.5630950, v.y(), -49096.1941554));
    assert_eq!(v.get_incident_edge().unwrap().0, 651);
    let v = output.vertices()[207];
    assert!(almost_equal(v.x(), -49493.8373266, v.y(), -49046.8247163));
    assert_eq!(v.get_incident_edge().unwrap().0, 657);
    let v = output.vertices()[208];
    assert!(almost_equal(v.x(), -49500.7898032, v.y(), -49031.8220036));
    assert_eq!(v.get_incident_edge().unwrap().0, 663);
    let v = output.vertices()[209];
    assert!(almost_equal(v.x(), -49468.6322321, v.y(), -49794.3264464));
    assert_eq!(v.get_incident_edge().unwrap().0, 665);
    let v = output.vertices()[210];
    assert!(almost_equal(v.x(), -49466.4359454, v.y(), -49819.8871891));
    assert_eq!(v.get_incident_edge().unwrap().0, 673);
    let v = output.vertices()[211];
    assert!(almost_equal(v.x(), -49508.2975154, v.y(), -49017.5971421));
    assert_eq!(v.get_incident_edge().unwrap().0, 675);
    let v = output.vertices()[212];
    assert!(almost_equal(v.x(), -49511.9189873, v.y(), -49620.0113924));
    assert_eq!(v.get_incident_edge().unwrap().0, 677);
    let v = output.vertices()[213];
    assert!(almost_equal(v.x(), -49465.8515496, v.y(), -49081.9388993));
    assert_eq!(v.get_incident_edge().unwrap().0, 679);
    let v = output.vertices()[214];
    assert!(almost_equal(v.x(), -49461.3435374, v.y(), -49779.3547133));
    assert_eq!(v.get_incident_edge().unwrap().0, 681);
    let v = output.vertices()[215];
    assert!(almost_equal(v.x(), -49466.1095180, v.y(), -49071.5240442));
    assert_eq!(v.get_incident_edge().unwrap().0, 685);
    let v = output.vertices()[216];
    assert!(almost_equal(v.x(), -49580.2746789, v.y(), -49978.1273579));
    assert_eq!(v.get_incident_edge().unwrap().0, 697);
    let v = output.vertices()[217];
    assert!(almost_equal(v.x(), -49574.1875000, v.y(), -49975.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 699);
    let v = output.vertices()[218];
    assert!(almost_equal(v.x(), -49521.9474464, v.y(), -49281.2894152));
    assert_eq!(v.get_incident_edge().unwrap().0, 701);
    let v = output.vertices()[219];
    assert!(almost_equal(v.x(), -49443.4280496, v.y(), -49084.3072175));
    assert_eq!(v.get_incident_edge().unwrap().0, 703);
    let v = output.vertices()[220];
    assert!(almost_equal(v.x(), -49441.9789474, v.y(), -49975.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 705);
    let v = output.vertices()[221];
    assert!(almost_equal(v.x(), -49442.0000000, v.y(), -49080.3723404));
    assert_eq!(v.get_incident_edge().unwrap().0, 709);
    let v = output.vertices()[222];
    assert!(almost_equal(v.x(), -49441.5939092, v.y(), -49974.9912475));
    assert_eq!(v.get_incident_edge().unwrap().0, 711);
    let v = output.vertices()[223];
    assert!(almost_equal(v.x(), -49455.7037037, v.y(), -49066.2897252));
    assert_eq!(v.get_incident_edge().unwrap().0, 713);
    let v = output.vertices()[224];
    assert!(almost_equal(v.x(), -49510.8729318, v.y(), -48977.1802279));
    assert_eq!(v.get_incident_edge().unwrap().0, 715);
    let v = output.vertices()[225];
    assert!(almost_equal(v.x(), -49521.2898734, v.y(), -49581.8889241));
    assert_eq!(v.get_incident_edge().unwrap().0, 717);
    let v = output.vertices()[226];
    assert!(almost_equal(v.x(), -49487.9700045, v.y(), -49686.7264161));
    assert_eq!(v.get_incident_edge().unwrap().0, 719);
    let v = output.vertices()[227];
    assert!(almost_equal(v.x(), -49493.5498311, v.y(), -49931.9932432));
    assert_eq!(v.get_incident_edge().unwrap().0, 721);
    let v = output.vertices()[228];
    assert!(almost_equal(v.x(), -49448.7525597, v.y(), -49061.3566553));
    assert_eq!(v.get_incident_edge().unwrap().0, 723);
    let v = output.vertices()[229];
    assert!(almost_equal(v.x(), -49511.2260536, v.y(), -48961.9195402));
    assert_eq!(v.get_incident_edge().unwrap().0, 725);
    let v = output.vertices()[230];
    assert!(almost_equal(v.x(), -49434.0258152, v.y(), -49009.7860054));
    assert_eq!(v.get_incident_edge().unwrap().0, 737);
    let v = output.vertices()[231];
    assert!(almost_equal(v.x(), -49440.3363537, v.y(), -49127.9831460));
    assert_eq!(v.get_incident_edge().unwrap().0, 739);
    let v = output.vertices()[232];
    assert!(almost_equal(v.x(), -49442.0000000, v.y(), -49057.3199532));
    assert_eq!(v.get_incident_edge().unwrap().0, 741);
    let v = output.vertices()[233];
    assert!(almost_equal(v.x(), -49447.2011028, v.y(), -49766.2224526));
    assert_eq!(v.get_incident_edge().unwrap().0, 743);
    let v = output.vertices()[234];
    assert!(almost_equal(v.x(), -49430.4576462, v.y(), -49022.8213359));
    assert_eq!(v.get_incident_edge().unwrap().0, 745);
    let v = output.vertices()[235];
    assert!(almost_equal(v.x(), -49498.2135922, v.y(), -49261.5898058));
    assert_eq!(v.get_incident_edge().unwrap().0, 747);
    let v = output.vertices()[236];
    assert!(almost_equal(v.x(), -49463.5826742, v.y(), -49909.7582078));
    assert_eq!(v.get_incident_edge().unwrap().0, 749);
    let v = output.vertices()[237];
    assert!(almost_equal(v.x(), -49447.2653461, v.y(), -49687.7289843));
    assert_eq!(v.get_incident_edge().unwrap().0, 755);
    let v = output.vertices()[238];
    assert!(almost_equal(v.x(), -49529.1811468, v.y(), -49532.0752610));
    assert_eq!(v.get_incident_edge().unwrap().0, 757);
    let v = output.vertices()[239];
    assert!(almost_equal(v.x(), -49505.7833935, v.y(), -49492.3558535));
    assert_eq!(v.get_incident_edge().unwrap().0, 759);
    let v = output.vertices()[240];
    assert!(almost_equal(v.x(), -49419.5495251, v.y(), -49092.5529172));
    assert_eq!(v.get_incident_edge().unwrap().0, 761);
    let v = output.vertices()[241];
    assert!(almost_equal(v.x(), -49428.4629172, v.y(), -49050.4634574));
    assert_eq!(v.get_incident_edge().unwrap().0, 765);
    let v = output.vertices()[242];
    assert!(almost_equal(v.x(), -49402.3664164, v.y(), -49640.6563879));
    assert_eq!(v.get_incident_edge().unwrap().0, 771);
    let v = output.vertices()[243];
    assert!(almost_equal(v.x(), -49517.7424465, v.y(), -49525.3921698));
    assert_eq!(v.get_incident_edge().unwrap().0, 773);
    let v = output.vertices()[244];
    assert!(almost_equal(v.x(), -49434.6743692, v.y(), -49882.9936255));
    assert_eq!(v.get_incident_edge().unwrap().0, 775);
    let v = output.vertices()[245];
    assert!(almost_equal(v.x(), -49388.1689302, v.y(), -49627.4313322));
    assert_eq!(v.get_incident_edge().unwrap().0, 777);
    let v = output.vertices()[246];
    assert!(almost_equal(v.x(), -49429.1592091, v.y(), -49684.2741365));
    assert_eq!(v.get_incident_edge().unwrap().0, 781);
    let v = output.vertices()[247];
    assert!(almost_equal(v.x(), -49428.5068306, v.y(), -49857.2595628));
    assert_eq!(v.get_incident_edge().unwrap().0, 783);
    let v = output.vertices()[248];
    assert!(almost_equal(v.x(), -49489.7541572, v.y(), -49535.1555266));
    assert_eq!(v.get_incident_edge().unwrap().0, 785);
    let v = output.vertices()[249];
    assert!(almost_equal(v.x(), -49486.3584471, v.y(), -49286.1326362));
    assert_eq!(v.get_incident_edge().unwrap().0, 787);
    let v = output.vertices()[250];
    assert!(almost_equal(v.x(), -49610.8217355, v.y(), -50142.0840091));
    assert_eq!(v.get_incident_edge().unwrap().0, 789);
    let v = output.vertices()[251];
    assert!(almost_equal(v.x(), -49432.6272622, v.y(), -49191.3386952));
    assert_eq!(v.get_incident_edge().unwrap().0, 795);
    let v = output.vertices()[252];
    assert!(almost_equal(v.x(), -49435.7897954, v.y(), -49205.4119195));
    assert_eq!(v.get_incident_edge().unwrap().0, 797);
    let v = output.vertices()[253];
    assert!(almost_equal(v.x(), -49419.2295082, v.y(), -49778.0546448));
    assert_eq!(v.get_incident_edge().unwrap().0, 799);
    let v = output.vertices()[254];
    assert!(almost_equal(v.x(), -49469.6527662, v.y(), -49541.2539792));
    assert_eq!(v.get_incident_edge().unwrap().0, 801);
    let v = output.vertices()[255];
    assert!(almost_equal(v.x(), -49384.8735762, v.y(), -49182.1094977));
    assert_eq!(v.get_incident_edge().unwrap().0, 803);
    let v = output.vertices()[256];
    assert!(almost_equal(v.x(), -49403.8144203, v.y(), -49085.6688089));
    assert_eq!(v.get_incident_edge().unwrap().0, 805);
    let v = output.vertices()[257];
    assert!(almost_equal(v.x(), -49378.1439609, v.y(), -49630.2557766));
    assert_eq!(v.get_incident_edge().unwrap().0, 807);
    let v = output.vertices()[258];
    assert!(almost_equal(v.x(), -49399.2863934, v.y(), -49090.5552410));
    assert_eq!(v.get_incident_edge().unwrap().0, 809);
    let v = output.vertices()[259];
    assert!(almost_equal(v.x(), -49414.1754542, v.y(), -49682.5709118));
    assert_eq!(v.get_incident_edge().unwrap().0, 813);
    let v = output.vertices()[260];
    assert!(almost_equal(v.x(), -49365.0581787, v.y(), -49183.8765747));
    assert_eq!(v.get_incident_edge().unwrap().0, 815);
    let v = output.vertices()[261];
    assert!(almost_equal(v.x(), -49419.4152184, v.y(), -49862.0446219));
    assert_eq!(v.get_incident_edge().unwrap().0, 817);
    let v = output.vertices()[262];
    assert!(almost_equal(v.x(), -49395.5536384, v.y(), -49094.9887983));
    assert_eq!(v.get_incident_edge().unwrap().0, 823);
    let v = output.vertices()[263];
    assert!(almost_equal(v.x(), -49411.5600000, v.y(), -49782.9120000));
    assert_eq!(v.get_incident_edge().unwrap().0, 825);
    let v = output.vertices()[264];
    assert!(almost_equal(v.x(), -49360.5567881, v.y(), -49179.7723656));
    assert_eq!(v.get_incident_edge().unwrap().0, 827);
    let v = output.vertices()[265];
    assert!(almost_equal(v.x(), -49358.1966673, v.y(), -49093.1966673));
    assert_eq!(v.get_incident_edge().unwrap().0, 829);
    let v = output.vertices()[266];
    assert!(almost_equal(v.x(), -49509.8452088, v.y(), -48870.7837838));
    assert_eq!(v.get_incident_edge().unwrap().0, 831);
    let v = output.vertices()[267];
    assert!(almost_equal(v.x(), -49382.0195813, v.y(), -49117.0195813));
    assert_eq!(v.get_incident_edge().unwrap().0, 833);
    let v = output.vertices()[268];
    assert!(almost_equal(v.x(), -49471.8044916, v.y(), -49295.0974098));
    assert_eq!(v.get_incident_edge().unwrap().0, 835);
    let v = output.vertices()[269];
    assert!(almost_equal(v.x(), -49353.8033327, v.y(), -49098.1770562));
    assert_eq!(v.get_incident_edge().unwrap().0, 839);
    let v = output.vertices()[270];
    assert!(almost_equal(v.x(), -49453.2208395, v.y(), -49370.7110857));
    assert_eq!(v.get_incident_edge().unwrap().0, 847);
    let v = output.vertices()[271];
    assert!(almost_equal(v.x(), -49397.7935503, v.y(), -49559.6958875));
    assert_eq!(v.get_incident_edge().unwrap().0, 853);
    let v = output.vertices()[272];
    assert!(almost_equal(v.x(), -49511.7815094, v.y(), -48840.9647547));
    assert_eq!(v.get_incident_edge().unwrap().0, 855);
    let v = output.vertices()[273];
    assert!(almost_equal(v.x(), -49357.2718229, v.y(), -49142.9358885));
    assert_eq!(v.get_incident_edge().unwrap().0, 857);
    let v = output.vertices()[274];
    assert!(almost_equal(v.x(), -49366.4283897, v.y(), -49447.1650262));
    assert_eq!(v.get_incident_edge().unwrap().0, 859);
    let v = output.vertices()[275];
    assert!(almost_equal(v.x(), -49369.2967150, v.y(), -49438.5908923));
    assert_eq!(v.get_incident_edge().unwrap().0, 861);
    let v = output.vertices()[276];
    assert!(almost_equal(v.x(), -49450.9622828, v.y(), -49297.2453959));
    assert_eq!(v.get_incident_edge().unwrap().0, 869);
    let v = output.vertices()[277];
    assert!(almost_equal(v.x(), -49455.7260749, v.y(), -49304.3696255));
    assert_eq!(v.get_incident_edge().unwrap().0, 871);
    let v = output.vertices()[278];
    assert!(almost_equal(v.x(), -49366.3853282, v.y(), -49440.8472383));
    assert_eq!(v.get_incident_edge().unwrap().0, 873);
    let v = output.vertices()[279];
    assert!(almost_equal(v.x(), -49452.8101645, v.y(), -49326.1528462));
    assert_eq!(v.get_incident_edge().unwrap().0, 875);
    let v = output.vertices()[280];
    assert!(almost_equal(v.x(), -49359.8630248, v.y(), -49439.7945488));
    assert_eq!(v.get_incident_edge().unwrap().0, 877);
    let v = output.vertices()[281];
    assert!(almost_equal(v.x(), -49451.9909747, v.y(), -49323.0451264));
    assert_eq!(v.get_incident_edge().unwrap().0, 879);
    let v = output.vertices()[282];
    assert!(almost_equal(v.x(), -49316.6180845, v.y(), -49145.6180845));
    assert_eq!(v.get_incident_edge().unwrap().0, 883);
    let v = output.vertices()[283];
    assert!(almost_equal(v.x(), -49366.6764991, v.y(), -49743.7534513));
    assert_eq!(v.get_incident_edge().unwrap().0, 887);
    let v = output.vertices()[284];
    assert!(almost_equal(v.x(), -49310.5899283, v.y(), -49139.5899283));
    assert_eq!(v.get_incident_edge().unwrap().0, 889);
    let v = output.vertices()[285];
    assert!(almost_equal(v.x(), -49343.9829443, v.y(), -49590.7411730));
    assert_eq!(v.get_incident_edge().unwrap().0, 893);
    let v = output.vertices()[286];
    assert!(almost_equal(v.x(), -49305.4497046, v.y(), -49429.1682593));
    assert_eq!(v.get_incident_edge().unwrap().0, 895);
    let v = output.vertices()[287];
    assert!(almost_equal(v.x(), -49342.6672078, v.y(), -49237.2122565));
    assert_eq!(v.get_incident_edge().unwrap().0, 901);
    let v = output.vertices()[288];
    assert!(almost_equal(v.x(), -49373.8600000, v.y(), -49801.3720000));
    assert_eq!(v.get_incident_edge().unwrap().0, 903);
    let v = output.vertices()[289];
    assert!(almost_equal(v.x(), -49334.8985330, v.y(), -49600.8349633));
    assert_eq!(v.get_incident_edge().unwrap().0, 905);
    let v = output.vertices()[290];
    assert!(almost_equal(v.x(), -49299.6418964, v.y(), -49567.1809011));
    assert_eq!(v.get_incident_edge().unwrap().0, 907);
    let v = output.vertices()[291];
    assert!(almost_equal(v.x(), -49330.4684909, v.y(), -49610.6674959));
    assert_eq!(v.get_incident_edge().unwrap().0, 915);
    let v = output.vertices()[292];
    assert!(almost_equal(v.x(), -49333.2902043, v.y(), -49232.5859461));
    assert_eq!(v.get_incident_edge().unwrap().0, 917);
    let v = output.vertices()[293];
    assert!(almost_equal(v.x(), -49313.4957095, v.y(), -49502.8122028));
    assert_eq!(v.get_incident_edge().unwrap().0, 919);
    let v = output.vertices()[294];
    assert!(almost_equal(v.x(), -49328.3324903, v.y(), -49622.0220254));
    assert_eq!(v.get_incident_edge().unwrap().0, 921);
    let v = output.vertices()[295];
    assert!(almost_equal(v.x(), -49282.1012216, v.y(), -49588.6823735));
    assert_eq!(v.get_incident_edge().unwrap().0, 923);
    let v = output.vertices()[296];
    assert!(almost_equal(v.x(), -49307.2355519, v.y(), -49088.7480294));
    assert_eq!(v.get_incident_edge().unwrap().0, 929);
    let v = output.vertices()[297];
    assert!(almost_equal(v.x(), -49323.1991796, v.y(), -49346.5636281));
    assert_eq!(v.get_incident_edge().unwrap().0, 931);
    let v = output.vertices()[298];
    assert!(almost_equal(v.x(), -49277.5707760, v.y(), -49587.0871718));
    assert_eq!(v.get_incident_edge().unwrap().0, 933);
    let v = output.vertices()[299];
    assert!(almost_equal(v.x(), -49307.1261826, v.y(), -49087.2239751));
    assert_eq!(v.get_incident_edge().unwrap().0, 935);
    let v = output.vertices()[300];
    assert!(almost_equal(v.x(), -49307.2454949, v.y(), -49086.4063396));
    assert_eq!(v.get_incident_edge().unwrap().0, 937);
    let v = output.vertices()[301];
    assert!(almost_equal(v.x(), -49268.5126871, v.y(), -49120.6890046));
    assert_eq!(v.get_incident_edge().unwrap().0, 945);
    let v = output.vertices()[302];
    assert!(almost_equal(v.x(), -49279.6696928, v.y(), -49456.2703230));
    assert_eq!(v.get_incident_edge().unwrap().0, 947);
    let v = output.vertices()[303];
    assert!(almost_equal(v.x(), -49287.1248395, v.y(), -49532.5759853));
    assert_eq!(v.get_incident_edge().unwrap().0, 949);
    let v = output.vertices()[304];
    assert!(almost_equal(v.x(), -49313.5973328, v.y(), -49347.9299081));
    assert_eq!(v.get_incident_edge().unwrap().0, 953);
    let v = output.vertices()[305];
    assert!(almost_equal(v.x(), -49298.8067952, v.y(), -49502.9123214));
    assert_eq!(v.get_incident_edge().unwrap().0, 955);
    let v = output.vertices()[306];
    assert!(almost_equal(v.x(), -49278.8753301, v.y(), -49460.3379620));
    assert_eq!(v.get_incident_edge().unwrap().0, 957);
    let v = output.vertices()[307];
    assert!(almost_equal(v.x(), -49286.8253464, v.y(), -49525.9057980));
    assert_eq!(v.get_incident_edge().unwrap().0, 963);
    let v = output.vertices()[308];
    assert!(almost_equal(v.x(), -49278.8250488, v.y(), -49528.8368768));
    assert_eq!(v.get_incident_edge().unwrap().0, 965);
    let v = output.vertices()[309];
    assert!(almost_equal(v.x(), -49292.8949698, v.y(), -49510.0888682));
    assert_eq!(v.get_incident_edge().unwrap().0, 967);
    let v = output.vertices()[310];
    assert!(almost_equal(v.x(), -49347.9720000, v.y(), -49803.1205714));
    assert_eq!(v.get_incident_edge().unwrap().0, 973);
    let v = output.vertices()[311];
    assert!(almost_equal(v.x(), -49349.1820556, v.y(), -49804.0791188));
    assert_eq!(v.get_incident_edge().unwrap().0, 975);
    let v = output.vertices()[312];
    assert!(almost_equal(v.x(), -49351.6103976, v.y(), -49810.0100810));
    assert_eq!(v.get_incident_edge().unwrap().0, 977);
    let v = output.vertices()[313];
    assert!(almost_equal(v.x(), -49351.4121217, v.y(), -49819.2797879));
    assert_eq!(v.get_incident_edge().unwrap().0, 979);
    let v = output.vertices()[314];
    assert!(almost_equal(v.x(), -49263.5097250, v.y(), -49533.8913048));
    assert_eq!(v.get_incident_edge().unwrap().0, 981);
    let v = output.vertices()[315];
    assert!(almost_equal(v.x(), -49321.9794013, v.y(), -49893.8459275));
    assert_eq!(v.get_incident_edge().unwrap().0, 985);
    let v = output.vertices()[316];
    assert!(almost_equal(v.x(), -49329.5285771, v.y(), -49970.0108328));
    assert_eq!(v.get_incident_edge().unwrap().0, 987);
    let v = output.vertices()[317];
    assert!(almost_equal(v.x(), -49303.2738955, v.y(), -49058.9361103));
    assert_eq!(v.get_incident_edge().unwrap().0, 989);
    let v = output.vertices()[318];
    assert!(almost_equal(v.x(), -49264.5794736, v.y(), -49458.8163756));
    assert_eq!(v.get_incident_edge().unwrap().0, 993);
    let v = output.vertices()[319];
    assert!(almost_equal(v.x(), -49247.3183603, v.y(), -49553.7387842));
    assert_eq!(v.get_incident_edge().unwrap().0, 997);
    let v = output.vertices()[320];
    assert!(almost_equal(v.x(), -49258.3776671, v.y(), -49458.9573257));
    assert_eq!(v.get_incident_edge().unwrap().0, 999);
    let v = output.vertices()[321];
    assert!(almost_equal(v.x(), -49255.7049307, v.y(), -49501.7211094));
    assert_eq!(v.get_incident_edge().unwrap().0, 1005);
    let v = output.vertices()[322];
    assert!(almost_equal(v.x(), -49256.5066895, v.y(), -49581.5801615));
    assert_eq!(v.get_incident_edge().unwrap().0, 1007);
    let v = output.vertices()[323];
    assert!(almost_equal(v.x(), -49289.7997220, v.y(), -49657.1256804));
    assert_eq!(v.get_incident_edge().unwrap().0, 1011);
    let v = output.vertices()[324];
    assert!(almost_equal(v.x(), -49252.7839780, v.y(), -49585.0298249));
    assert_eq!(v.get_incident_edge().unwrap().0, 1013);
    let v = output.vertices()[325];
    assert!(almost_equal(v.x(), -49240.3075518, v.y(), -49282.9129111));
    assert_eq!(v.get_incident_edge().unwrap().0, 1015);
    let v = output.vertices()[326];
    assert!(almost_equal(v.x(), -49264.3168423, v.y(), -49311.9505025));
    assert_eq!(v.get_incident_edge().unwrap().0, 1017);
    let v = output.vertices()[327];
    assert!(almost_equal(v.x(), -49257.8618872, v.y(), -49153.2080917));
    assert_eq!(v.get_incident_edge().unwrap().0, 1023);
    let v = output.vertices()[328];
    assert!(almost_equal(v.x(), -49279.6102488, v.y(), -49662.6506427));
    assert_eq!(v.get_incident_edge().unwrap().0, 1025);
    let v = output.vertices()[329];
    assert!(almost_equal(v.x(), -49242.4751067, v.y(), -49456.3982930));
    assert_eq!(v.get_incident_edge().unwrap().0, 1027);
    let v = output.vertices()[330];
    assert!(almost_equal(v.x(), -49239.6671803, v.y(), -49501.3251156));
    assert_eq!(v.get_incident_edge().unwrap().0, 1029);
    let v = output.vertices()[331];
    assert!(almost_equal(v.x(), -49252.6420548, v.y(), -49143.8430984));
    assert_eq!(v.get_incident_edge().unwrap().0, 1031);
    let v = output.vertices()[332];
    assert!(almost_equal(v.x(), -49244.7194578, v.y(), -49139.3432444));
    assert_eq!(v.get_incident_edge().unwrap().0, 1035);
    let v = output.vertices()[333];
    assert!(almost_equal(v.x(), -49311.0917792, v.y(), -49968.5746067));
    assert_eq!(v.get_incident_edge().unwrap().0, 1037);
    let v = output.vertices()[334];
    assert!(almost_equal(v.x(), -49280.6845945, v.y(), -49351.3249737));
    assert_eq!(v.get_incident_edge().unwrap().0, 1041);
    let v = output.vertices()[335];
    assert!(almost_equal(v.x(), -49277.2116580, v.y(), -49354.0530607));
    assert_eq!(v.get_incident_edge().unwrap().0, 1047);
    let v = output.vertices()[336];
    assert!(almost_equal(v.x(), -49696.7430195, v.y(), -50417.1206964));
    assert_eq!(v.get_incident_edge().unwrap().0, 1049);
    let v = output.vertices()[337];
    assert!(almost_equal(v.x(), -49227.1534679, v.y(), -49130.7475229));
    assert_eq!(v.get_incident_edge().unwrap().0, 1051);
    let v = output.vertices()[338];
    assert!(almost_equal(v.x(), -49223.1754318, v.y(), -49129.0298273));
    assert_eq!(v.get_incident_edge().unwrap().0, 1063);
    let v = output.vertices()[339];
    assert!(almost_equal(v.x(), -49223.8082192, v.y(), -49892.7671233));
    assert_eq!(v.get_incident_edge().unwrap().0, 1065);
    let v = output.vertices()[340];
    assert!(almost_equal(v.x(), -49223.1840796, v.y(), -49895.2636816));
    assert_eq!(v.get_incident_edge().unwrap().0, 1067);
    let v = output.vertices()[341];
    assert!(almost_equal(v.x(), -49223.9041337, v.y(), -49349.3188215));
    assert_eq!(v.get_incident_edge().unwrap().0, 1069);
    let v = output.vertices()[342];
    assert!(almost_equal(v.x(), -49271.9214484, v.y(), -49205.8132579));
    assert_eq!(v.get_incident_edge().unwrap().0, 1071);
    let v = output.vertices()[343];
    assert!(almost_equal(v.x(), -49207.3568488, v.y(), -49119.5500014));
    assert_eq!(v.get_incident_edge().unwrap().0, 1077);
    let v = output.vertices()[344];
    assert!(almost_equal(v.x(), -49240.2022222, v.y(), -49396.8285079));
    assert_eq!(v.get_incident_edge().unwrap().0, 1079);
    let v = output.vertices()[345];
    assert!(almost_equal(v.x(), -49239.3803419, v.y(), -49397.3803419));
    assert_eq!(v.get_incident_edge().unwrap().0, 1081);
    let v = output.vertices()[346];
    assert!(almost_equal(v.x(), -49223.5737240, v.y(), -49504.0727788));
    assert_eq!(v.get_incident_edge().unwrap().0, 1087);
    let v = output.vertices()[347];
    assert!(almost_equal(v.x(), -49220.9670200, v.y(), -49514.1272085));
    assert_eq!(v.get_incident_edge().unwrap().0, 1089);
    let v = output.vertices()[348];
    assert!(almost_equal(v.x(), -49218.7837423, v.y(), -49495.7346626));
    assert_eq!(v.get_incident_edge().unwrap().0, 1091);
    let v = output.vertices()[349];
    assert!(almost_equal(v.x(), -49213.1739130, v.y(), -49590.0869565));
    assert_eq!(v.get_incident_edge().unwrap().0, 1093);
    let v = output.vertices()[350];
    assert!(almost_equal(v.x(), -49252.6019122, v.y(), -49210.9030856));
    assert_eq!(v.get_incident_edge().unwrap().0, 1095);
    let v = output.vertices()[351];
    assert!(almost_equal(v.x(), -49199.8908795, v.y(), -49138.3436482));
    assert_eq!(v.get_incident_edge().unwrap().0, 1099);
    let v = output.vertices()[352];
    assert!(almost_equal(v.x(), -49214.8181818, v.y(), -49274.8636364));
    assert_eq!(v.get_incident_edge().unwrap().0, 1101);
    let v = output.vertices()[353];
    assert!(almost_equal(v.x(), -49213.9693252, v.y(), -49484.6503067));
    assert_eq!(v.get_incident_edge().unwrap().0, 1103);
    let v = output.vertices()[354];
    assert!(almost_equal(v.x(), -49245.1500000, v.y(), -49214.2000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 1105);
    let v = output.vertices()[355];
    assert!(almost_equal(v.x(), -49213.4390935, v.y(), -49276.1048159));
    assert_eq!(v.get_incident_edge().unwrap().0, 1107);
    let v = output.vertices()[356];
    assert!(almost_equal(v.x(), -49246.2016592, v.y(), -49212.1169267));
    assert_eq!(v.get_incident_edge().unwrap().0, 1109);
    let v = output.vertices()[357];
    assert!(almost_equal(v.x(), -49199.9631902, v.y(), -49352.4079755));
    assert_eq!(v.get_incident_edge().unwrap().0, 1123);
    let v = output.vertices()[358];
    assert!(almost_equal(v.x(), -49209.9728919, v.y(), -49278.9875449));
    assert_eq!(v.get_incident_edge().unwrap().0, 1125);
    let v = output.vertices()[359];
    assert!(almost_equal(v.x(), -49197.2446809, v.y(), -49355.2446809));
    assert_eq!(v.get_incident_edge().unwrap().0, 1127);
    let v = output.vertices()[360];
    assert!(almost_equal(v.x(), -49715.9420403, v.y(), -50464.5224167));
    assert_eq!(v.get_incident_edge().unwrap().0, 1129);
    let v = output.vertices()[361];
    assert!(almost_equal(v.x(), -49203.5358540, v.y(), -49276.0299870));
    assert_eq!(v.get_incident_edge().unwrap().0, 1131);
    let v = output.vertices()[362];
    assert!(almost_equal(v.x(), -49189.5327900, v.y(), -49885.9007567));
    assert_eq!(v.get_incident_edge().unwrap().0, 1133);
    let v = output.vertices()[363];
    assert!(almost_equal(v.x(), -49206.4079910, v.y(), -49184.2155318));
    assert_eq!(v.get_incident_edge().unwrap().0, 1141);
    let v = output.vertices()[364];
    assert!(almost_equal(v.x(), -49223.2807777, v.y(), -49204.9449555));
    assert_eq!(v.get_incident_edge().unwrap().0, 1143);
    let v = output.vertices()[365];
    assert!(almost_equal(v.x(), -49198.7568142, v.y(), -49172.1737041));
    assert_eq!(v.get_incident_edge().unwrap().0, 1145);
    let v = output.vertices()[366];
    assert!(almost_equal(v.x(), -49188.9104092, v.y(), -49361.5968415));
    assert_eq!(v.get_incident_edge().unwrap().0, 1149);
    let v = output.vertices()[367];
    assert!(almost_equal(v.x(), -49186.8562214, v.y(), -49263.0040937));
    assert_eq!(v.get_incident_edge().unwrap().0, 1151);
    let v = output.vertices()[368];
    assert!(almost_equal(v.x(), -49188.8481013, v.y(), -49324.6202532));
    assert_eq!(v.get_incident_edge().unwrap().0, 1153);
    let v = output.vertices()[369];
    assert!(almost_equal(v.x(), -49191.3150820, v.y(), -49156.3463700));
    assert_eq!(v.get_incident_edge().unwrap().0, 1157);
    let v = output.vertices()[370];
    assert!(almost_equal(v.x(), -49169.0410911, v.y(), -49113.0170051));
    assert_eq!(v.get_incident_edge().unwrap().0, 1159);
    let v = output.vertices()[371];
    assert!(almost_equal(v.x(), -49189.6333973, v.y(), -49152.2735125));
    assert_eq!(v.get_incident_edge().unwrap().0, 1163);
    let v = output.vertices()[372];
    assert!(almost_equal(v.x(), -49186.8139985, v.y(), -49870.7350222));
    assert_eq!(v.get_incident_edge().unwrap().0, 1165);
    let v = output.vertices()[373];
    assert!(almost_equal(v.x(), -49272.4098361, v.y(), -49997.2076503));
    assert_eq!(v.get_incident_edge().unwrap().0, 1167);
    let v = output.vertices()[374];
    assert!(almost_equal(v.x(), -49187.3738748, v.y(), -49145.9468493));
    assert_eq!(v.get_incident_edge().unwrap().0, 1169);
    let v = output.vertices()[375];
    assert!(almost_equal(v.x(), -49176.0726052, v.y(), -49904.6488713));
    assert_eq!(v.get_incident_edge().unwrap().0, 1171);
    let v = output.vertices()[376];
    assert!(almost_equal(v.x(), -49238.5746269, v.y(), -49968.2014925));
    assert_eq!(v.get_incident_edge().unwrap().0, 1173);
    let v = output.vertices()[377];
    assert!(almost_equal(v.x(), -49185.0782942, v.y(), -49316.4607893));
    assert_eq!(v.get_incident_edge().unwrap().0, 1175);
    let v = output.vertices()[378];
    assert!(almost_equal(v.x(), -49206.2521368, v.y(), -49440.2521368));
    assert_eq!(v.get_incident_edge().unwrap().0, 1177);
    let v = output.vertices()[379];
    assert!(almost_equal(v.x(), -49166.4066647, v.y(), -49194.8041182));
    assert_eq!(v.get_incident_edge().unwrap().0, 1179);
    let v = output.vertices()[380];
    assert!(almost_equal(v.x(), -49205.0840387, v.y(), -49443.3548972));
    assert_eq!(v.get_incident_edge().unwrap().0, 1181);
    let v = output.vertices()[381];
    assert!(almost_equal(v.x(), -49178.9298597, v.y(), -49316.6843687));
    assert_eq!(v.get_incident_edge().unwrap().0, 1183);
    let v = output.vertices()[382];
    assert!(almost_equal(v.x(), -49202.1155378, v.y(), -49446.1254980));
    assert_eq!(v.get_incident_edge().unwrap().0, 1185);
    let v = output.vertices()[383];
    assert!(almost_equal(v.x(), -49168.0889200, v.y(), -49089.4507695));
    assert_eq!(v.get_incident_edge().unwrap().0, 1187);
    let v = output.vertices()[384];
    assert!(almost_equal(v.x(), -49256.7691143, v.y(), -50006.2445117));
    assert_eq!(v.get_incident_edge().unwrap().0, 1191);
    let v = output.vertices()[385];
    assert!(almost_equal(v.x(), -49232.7065108, v.y(), -49701.9554711));
    assert_eq!(v.get_incident_edge().unwrap().0, 1193);
    let v = output.vertices()[386];
    assert!(almost_equal(v.x(), -49150.2921348, v.y(), -49858.2359551));
    assert_eq!(v.get_incident_edge().unwrap().0, 1205);
    let v = output.vertices()[387];
    assert!(almost_equal(v.x(), -49147.2614598, v.y(), -49879.0256083));
    assert_eq!(v.get_incident_edge().unwrap().0, 1207);
    let v = output.vertices()[388];
    assert!(almost_equal(v.x(), -49160.9056963, v.y(), -49135.8780427));
    assert_eq!(v.get_incident_edge().unwrap().0, 1213);
    let v = output.vertices()[389];
    assert!(almost_equal(v.x(), -49144.1758242, v.y(), -49860.2747253));
    assert_eq!(v.get_incident_edge().unwrap().0, 1215);
    let v = output.vertices()[390];
    assert!(almost_equal(v.x(), -49141.0669546, v.y(), -49871.0853132));
    assert_eq!(v.get_incident_edge().unwrap().0, 1217);
    let v = output.vertices()[391];
    assert!(almost_equal(v.x(), -49187.2328200, v.y(), -49834.7988107));
    assert_eq!(v.get_incident_edge().unwrap().0, 1219);
    let v = output.vertices()[392];
    assert!(almost_equal(v.x(), -49189.6802981, v.y(), -49829.1171651));
    assert_eq!(v.get_incident_edge().unwrap().0, 1223);
    let v = output.vertices()[393];
    assert!(almost_equal(v.x(), -49145.5326377, v.y(), -49206.3478078));
    assert_eq!(v.get_incident_edge().unwrap().0, 1225);
    let v = output.vertices()[394];
    assert!(almost_equal(v.x(), -49227.8962179, v.y(), -49707.5500510));
    assert_eq!(v.get_incident_edge().unwrap().0, 1227);
    let v = output.vertices()[395];
    assert!(almost_equal(v.x(), -49149.8348273, v.y(), -49162.8084281));
    assert_eq!(v.get_incident_edge().unwrap().0, 1229);
    let v = output.vertices()[396];
    assert!(almost_equal(v.x(), -49155.5803132, v.y(), -49908.3225655));
    assert_eq!(v.get_incident_edge().unwrap().0, 1233);
    let v = output.vertices()[397];
    assert!(almost_equal(v.x(), -49135.9156527, v.y(), -49868.7958457));
    assert_eq!(v.get_incident_edge().unwrap().0, 1237);
    let v = output.vertices()[398];
    assert!(almost_equal(v.x(), -49154.0370344, v.y(), -49911.1877674));
    assert_eq!(v.get_incident_edge().unwrap().0, 1239);
    let v = output.vertices()[399];
    assert!(almost_equal(v.x(), -49156.0719424, v.y(), -49306.0071942));
    assert_eq!(v.get_incident_edge().unwrap().0, 1245);
    let v = output.vertices()[400];
    assert!(almost_equal(v.x(), -49146.2009095, v.y(), -49152.8451044));
    assert_eq!(v.get_incident_edge().unwrap().0, 1247);
    let v = output.vertices()[401];
    assert!(almost_equal(v.x(), -49156.2470838, v.y(), -49308.4591729));
    assert_eq!(v.get_incident_edge().unwrap().0, 1249);
    let v = output.vertices()[402];
    assert!(almost_equal(v.x(), -49131.8920650, v.y(), -49872.6216937));
    assert_eq!(v.get_incident_edge().unwrap().0, 1251);
    let v = output.vertices()[403];
    assert!(almost_equal(v.x(), -49144.3264813, v.y(), -49152.3313180));
    assert_eq!(v.get_incident_edge().unwrap().0, 1253);
    let v = output.vertices()[404];
    assert!(almost_equal(v.x(), -49141.4167573, v.y(), -49151.4755169));
    assert_eq!(v.get_incident_edge().unwrap().0, 1255);
    let v = output.vertices()[405];
    assert!(almost_equal(v.x(), -49156.7659677, v.y(), -49313.5019295));
    assert_eq!(v.get_incident_edge().unwrap().0, 1257);
    let v = output.vertices()[406];
    assert!(almost_equal(v.x(), -49135.3790183, v.y(), -49117.0604042));
    assert_eq!(v.get_incident_edge().unwrap().0, 1259);
    let v = output.vertices()[407];
    assert!(almost_equal(v.x(), -49149.3263489, v.y(), -49383.3263489));
    assert_eq!(v.get_incident_edge().unwrap().0, 1263);
    let v = output.vertices()[408];
    assert!(almost_equal(v.x(), -49147.6246334, v.y(), -49935.5102639));
    assert_eq!(v.get_incident_edge().unwrap().0, 1265);
    let v = output.vertices()[409];
    assert!(almost_equal(v.x(), -49146.8368522, v.y(), -49384.7658349));
    assert_eq!(v.get_incident_edge().unwrap().0, 1269);
    let v = output.vertices()[410];
    assert!(almost_equal(v.x(), -49179.3839779, v.y(), -49801.2679558));
    assert_eq!(v.get_incident_edge().unwrap().0, 1271);
    let v = output.vertices()[411];
    assert!(almost_equal(v.x(), -49173.4961720, v.y(), -49564.8719081));
    assert_eq!(v.get_incident_edge().unwrap().0, 1273);
    let v = output.vertices()[412];
    assert!(almost_equal(v.x(), -49123.5904643, v.y(), -49883.5879549));
    assert_eq!(v.get_incident_edge().unwrap().0, 1275);
    let v = output.vertices()[413];
    assert!(almost_equal(v.x(), -49134.7347247, v.y(), -49330.3062610));
    assert_eq!(v.get_incident_edge().unwrap().0, 1277);
    let v = output.vertices()[414];
    assert!(almost_equal(v.x(), -49124.2777641, v.y(), -49115.9214905));
    assert_eq!(v.get_incident_edge().unwrap().0, 1279);
    let v = output.vertices()[415];
    assert!(almost_equal(v.x(), -49168.2980132, v.y(), -49567.6490066));
    assert_eq!(v.get_incident_edge().unwrap().0, 1281);
    let v = output.vertices()[416];
    assert!(almost_equal(v.x(), -49261.3869263, v.y(), -48944.3264250));
    assert_eq!(v.get_incident_edge().unwrap().0, 1283);
    let v = output.vertices()[417];
    assert!(almost_equal(v.x(), -49170.4698276, v.y(), -49783.4396552));
    assert_eq!(v.get_incident_edge().unwrap().0, 1293);
    let v = output.vertices()[418];
    assert!(almost_equal(v.x(), -49114.2779627, v.y(), -49113.6143467));
    assert_eq!(v.get_incident_edge().unwrap().0, 1295);
    let v = output.vertices()[419];
    assert!(almost_equal(v.x(), -49103.3718011, v.y(), -49309.0064797));
    assert_eq!(v.get_incident_edge().unwrap().0, 1303);
    let v = output.vertices()[420];
    assert!(almost_equal(v.x(), -49118.5238396, v.y(), -49378.0768866));
    assert_eq!(v.get_incident_edge().unwrap().0, 1305);
    let v = output.vertices()[421];
    assert!(almost_equal(v.x(), -49088.0958493, v.y(), -49361.9935203));
    assert_eq!(v.get_incident_edge().unwrap().0, 1311);
    let v = output.vertices()[422];
    assert!(almost_equal(v.x(), -49152.3314646, v.y(), -49592.9397337));
    assert_eq!(v.get_incident_edge().unwrap().0, 1313);
    let v = output.vertices()[423];
    assert!(almost_equal(v.x(), -49189.6159606, v.y(), -49713.3928271));
    assert_eq!(v.get_incident_edge().unwrap().0, 1315);
    let v = output.vertices()[424];
    assert!(almost_equal(v.x(), -49119.1640069, v.y(), -49381.6333716));
    assert_eq!(v.get_incident_edge().unwrap().0, 1317);
    let v = output.vertices()[425];
    assert!(almost_equal(v.x(), -49127.6546134, v.y(), -49396.4256271));
    assert_eq!(v.get_incident_edge().unwrap().0, 1327);
    let v = output.vertices()[426];
    assert!(almost_equal(v.x(), -49102.3410537, v.y(), -49118.9388107));
    assert_eq!(v.get_incident_edge().unwrap().0, 1329);
    let v = output.vertices()[427];
    assert!(almost_equal(v.x(), -49126.9405748, v.y(), -49404.3073551));
    assert_eq!(v.get_incident_edge().unwrap().0, 1333);
    let v = output.vertices()[428];
    assert!(almost_equal(v.x(), -49103.1935484, v.y(), -49858.1451613));
    assert_eq!(v.get_incident_edge().unwrap().0, 1335);
    let v = output.vertices()[429];
    assert!(almost_equal(v.x(), -49103.2607786, v.y(), -49857.9083519));
    assert_eq!(v.get_incident_edge().unwrap().0, 1337);
    let v = output.vertices()[430];
    assert!(almost_equal(v.x(), -49155.3034091, v.y(), -49570.4954437));
    assert_eq!(v.get_incident_edge().unwrap().0, 1343);
    let v = output.vertices()[431];
    assert!(almost_equal(v.x(), -49087.8949201, v.y(), -49131.8727108));
    assert_eq!(v.get_incident_edge().unwrap().0, 1345);
    let v = output.vertices()[432];
    assert!(almost_equal(v.x(), -49111.8580645, v.y(), -49071.2661290));
    assert_eq!(v.get_incident_edge().unwrap().0, 1347);
    let v = output.vertices()[433];
    assert!(almost_equal(v.x(), -49089.8956916, v.y(), -49112.4455782));
    assert_eq!(v.get_incident_edge().unwrap().0, 1353);
    let v = output.vertices()[434];
    assert!(almost_equal(v.x(), -49163.5714061, v.y(), -49017.0099088));
    assert_eq!(v.get_incident_edge().unwrap().0, 1355);
    let v = output.vertices()[435];
    assert!(almost_equal(v.x(), -49136.7868259, v.y(), -49533.0059882));
    assert_eq!(v.get_incident_edge().unwrap().0, 1359);
    let v = output.vertices()[436];
    assert!(almost_equal(v.x(), -49778.7501798, v.y(), -50632.0107888));
    assert_eq!(v.get_incident_edge().unwrap().0, 1361);
    let v = output.vertices()[437];
    assert!(almost_equal(v.x(), -49135.1817179, v.y(), -49527.8760513));
    assert_eq!(v.get_incident_edge().unwrap().0, 1363);
    let v = output.vertices()[438];
    assert!(almost_equal(v.x(), -49079.6945378, v.y(), -49134.5588655));
    assert_eq!(v.get_incident_edge().unwrap().0, 1365);
    let v = output.vertices()[439];
    assert!(almost_equal(v.x(), -49183.4429924, v.y(), -48999.3997587));
    assert_eq!(v.get_incident_edge().unwrap().0, 1367);
    let v = output.vertices()[440];
    assert!(almost_equal(v.x(), -49132.9817568, v.y(), -49521.4628378));
    assert_eq!(v.get_incident_edge().unwrap().0, 1369);
    let v = output.vertices()[441];
    assert!(almost_equal(v.x(), -49113.6364499, v.y(), -49821.6986736));
    assert_eq!(v.get_incident_edge().unwrap().0, 1371);
    let v = output.vertices()[442];
    assert!(almost_equal(v.x(), -49099.5847666, v.y(), -49278.5090909));
    assert_eq!(v.get_incident_edge().unwrap().0, 1377);
    let v = output.vertices()[443];
    assert!(almost_equal(v.x(), -49116.7936893, v.y(), -49972.1723301));
    assert_eq!(v.get_incident_edge().unwrap().0, 1379);
    let v = output.vertices()[444];
    assert!(almost_equal(v.x(), -49111.6426884, v.y(), -49060.1506897));
    assert_eq!(v.get_incident_edge().unwrap().0, 1381);
    let v = output.vertices()[445];
    assert!(almost_equal(v.x(), -49190.7621862, v.y(), -48988.4951255));
    assert_eq!(v.get_incident_edge().unwrap().0, 1383);
    let v = output.vertices()[446];
    assert!(almost_equal(v.x(), -49118.6446691, v.y(), -49807.6982400));
    assert_eq!(v.get_incident_edge().unwrap().0, 1389);
    let v = output.vertices()[447];
    assert!(almost_equal(v.x(), -49105.1432039, v.y(), -49959.5509709));
    assert_eq!(v.get_incident_edge().unwrap().0, 1391);
    let v = output.vertices()[448];
    assert!(almost_equal(v.x(), -49145.3689503, v.y(), -49998.6693903));
    assert_eq!(v.get_incident_edge().unwrap().0, 1393);
    let v = output.vertices()[449];
    assert!(almost_equal(v.x(), -49076.1200214, v.y(), -49314.5281119));
    assert_eq!(v.get_incident_edge().unwrap().0, 1399);
    let v = output.vertices()[450];
    assert!(almost_equal(v.x(), -49097.3985932, v.y(), -49062.2526377));
    assert_eq!(v.get_incident_edge().unwrap().0, 1401);
    let v = output.vertices()[451];
    assert!(almost_equal(v.x(), -49075.6971634, v.y(), -49306.9234424));
    assert_eq!(v.get_incident_edge().unwrap().0, 1403);
    let v = output.vertices()[452];
    assert!(almost_equal(v.x(), -49076.7295103, v.y(), -49109.3043294));
    assert_eq!(v.get_incident_edge().unwrap().0, 1405);
    let v = output.vertices()[453];
    assert!(almost_equal(v.x(), -49087.9102573, v.y(), -49789.0761657));
    assert_eq!(v.get_incident_edge().unwrap().0, 1409);
    let v = output.vertices()[454];
    assert!(almost_equal(v.x(), -49084.5156128, v.y(), -49785.8872811));
    assert_eq!(v.get_incident_edge().unwrap().0, 1411);
    let v = output.vertices()[455];
    assert!(almost_equal(v.x(), -49091.4670331, v.y(), -49899.8394379));
    assert_eq!(v.get_incident_edge().unwrap().0, 1413);
    let v = output.vertices()[456];
    assert!(almost_equal(v.x(), -49216.1283047, v.y(), -48949.3486781));
    assert_eq!(v.get_incident_edge().unwrap().0, 1415);
    let v = output.vertices()[457];
    assert!(almost_equal(v.x(), -49095.1537437, v.y(), -49267.5792345));
    assert_eq!(v.get_incident_edge().unwrap().0, 1417);
    let v = output.vertices()[458];
    assert!(almost_equal(v.x(), -49072.3400951, v.y(), -49109.2373217));
    assert_eq!(v.get_incident_edge().unwrap().0, 1419);
    let v = output.vertices()[459];
    assert!(almost_equal(v.x(), -49100.7966648, v.y(), -49251.2123375));
    assert_eq!(v.get_incident_edge().unwrap().0, 1421);
    let v = output.vertices()[460];
    assert!(almost_equal(v.x(), -49147.7133458, v.y(), -49679.3645291));
    assert_eq!(v.get_incident_edge().unwrap().0, 1423);
    let v = output.vertices()[461];
    assert!(almost_equal(v.x(), -49124.3404738, v.y(), -49461.5095766));
    assert_eq!(v.get_incident_edge().unwrap().0, 1425);
    let v = output.vertices()[462];
    assert!(almost_equal(v.x(), -49096.1153832, v.y(), -49239.3115945));
    assert_eq!(v.get_incident_edge().unwrap().0, 1429);
    let v = output.vertices()[463];
    assert!(almost_equal(v.x(), -49095.5528256, v.y(), -49237.9140049));
    assert_eq!(v.get_incident_edge().unwrap().0, 1431);
    let v = output.vertices()[464];
    assert!(almost_equal(v.x(), -49093.0992092, v.y(), -49195.2192667));
    assert_eq!(v.get_incident_edge().unwrap().0, 1433);
    let v = output.vertices()[465];
    assert!(almost_equal(v.x(), -49088.5761697, v.y(), -49185.1841676));
    assert_eq!(v.get_incident_edge().unwrap().0, 1435);
    let v = output.vertices()[466];
    assert!(almost_equal(v.x(), -49241.7277784, v.y(), -48906.1402459));
    assert_eq!(v.get_incident_edge().unwrap().0, 1437);
    let v = output.vertices()[467];
    assert!(almost_equal(v.x(), -49080.5749656, v.y(), -49907.5729023));
    assert_eq!(v.get_incident_edge().unwrap().0, 1439);
    let v = output.vertices()[468];
    assert!(almost_equal(v.x(), -49114.6199380, v.y(), -49472.3556481));
    assert_eq!(v.get_incident_edge().unwrap().0, 1443);
    let v = output.vertices()[469];
    assert!(almost_equal(v.x(), -49118.9915873, v.y(), -49681.2056675));
    assert_eq!(v.get_incident_edge().unwrap().0, 1449);
    let v = output.vertices()[470];
    assert!(almost_equal(v.x(), -49057.4987117, v.y(), -49859.0324857));
    assert_eq!(v.get_incident_edge().unwrap().0, 1451);
    let v = output.vertices()[471];
    assert!(almost_equal(v.x(), -49057.3862103, v.y(), -49874.5049249));
    assert_eq!(v.get_incident_edge().unwrap().0, 1457);
    let v = output.vertices()[472];
    assert!(almost_equal(v.x(), -49188.7638365, v.y(), -50086.2779340));
    assert_eq!(v.get_incident_edge().unwrap().0, 1459);
    let v = output.vertices()[473];
    assert!(almost_equal(v.x(), -49082.6577658, v.y(), -49196.7743753));
    assert_eq!(v.get_incident_edge().unwrap().0, 1461);
    let v = output.vertices()[474];
    assert!(almost_equal(v.x(), -49064.3202077, v.y(), -49177.7546572));
    assert_eq!(v.get_incident_edge().unwrap().0, 1467);
    let v = output.vertices()[475];
    assert!(almost_equal(v.x(), -49053.8340306, v.y(), -49105.5414394));
    assert_eq!(v.get_incident_edge().unwrap().0, 1469);
    let v = output.vertices()[476];
    assert!(almost_equal(v.x(), -49053.2804878, v.y(), -49104.7195122));
    assert_eq!(v.get_incident_edge().unwrap().0, 1471);
    let v = output.vertices()[477];
    assert!(almost_equal(v.x(), -49079.5776025, v.y(), -49199.1170347));
    assert_eq!(v.get_incident_edge().unwrap().0, 1473);
    let v = output.vertices()[478];
    assert!(almost_equal(v.x(), -49079.1003808, v.y(), -49198.9774699));
    assert_eq!(v.get_incident_edge().unwrap().0, 1475);
    let v = output.vertices()[479];
    assert!(almost_equal(v.x(), -49058.1862570, v.y(), -49064.9111012));
    assert_eq!(v.get_incident_edge().unwrap().0, 1481);
    let v = output.vertices()[480];
    assert!(almost_equal(v.x(), -49069.9492722, v.y(), -49914.0563763));
    assert_eq!(v.get_incident_edge().unwrap().0, 1483);
    let v = output.vertices()[481];
    assert!(almost_equal(v.x(), -49067.4404268, v.y(), -49913.6298726));
    assert_eq!(v.get_incident_edge().unwrap().0, 1489);
    let v = output.vertices()[482];
    assert!(almost_equal(v.x(), -49047.4302671, v.y(), -49045.8278932));
    assert_eq!(v.get_incident_edge().unwrap().0, 1491);
    let v = output.vertices()[483];
    assert!(almost_equal(v.x(), -49242.6701970, v.y(), -48861.1800466));
    assert_eq!(v.get_incident_edge().unwrap().0, 1493);
    let v = output.vertices()[484];
    assert!(almost_equal(v.x(), -49047.1216937, v.y(), -49088.2257168));
    assert_eq!(v.get_incident_edge().unwrap().0, 1495);
    let v = output.vertices()[485];
    assert!(almost_equal(v.x(), -49025.3574578, v.y(), -49167.7740627));
    assert_eq!(v.get_incident_edge().unwrap().0, 1497);
    let v = output.vertices()[486];
    assert!(almost_equal(v.x(), -49017.7950556, v.y(), -49882.1483319));
    assert_eq!(v.get_incident_edge().unwrap().0, 1499);
    let v = output.vertices()[487];
    assert!(almost_equal(v.x(), -49046.0691979, v.y(), -49090.5480286));
    assert_eq!(v.get_incident_edge().unwrap().0, 1501);
    let v = output.vertices()[488];
    assert!(almost_equal(v.x(), -49018.3965887, v.y(), -49864.1023398));
    assert_eq!(v.get_incident_edge().unwrap().0, 1503);
    let v = output.vertices()[489];
    assert!(almost_equal(v.x(), -49020.6100234, v.y(), -49137.3899766));
    assert_eq!(v.get_incident_edge().unwrap().0, 1505);
    let v = output.vertices()[490];
    assert!(almost_equal(v.x(), -49008.7199138, v.y(), -49091.2800862));
    assert_eq!(v.get_incident_edge().unwrap().0, 1511);
    let v = output.vertices()[491];
    assert!(almost_equal(v.x(), -49034.2990891, v.y(), -49954.0677915));
    assert_eq!(v.get_incident_edge().unwrap().0, 1513);
    let v = output.vertices()[492];
    assert!(almost_equal(v.x(), -49027.0483871, v.y(), -49778.5379507));
    assert_eq!(v.get_incident_edge().unwrap().0, 1515);
    let v = output.vertices()[493];
    assert!(almost_equal(v.x(), -48997.7114588, v.y(), -49102.2885412));
    assert_eq!(v.get_incident_edge().unwrap().0, 1517);
    let v = output.vertices()[494];
    assert!(almost_equal(v.x(), -49028.4837963, v.y(), -49970.5023148));
    assert_eq!(v.get_incident_edge().unwrap().0, 1519);
    let v = output.vertices()[495];
    assert!(almost_equal(v.x(), -49028.7136499, v.y(), -49024.3145401));
    assert_eq!(v.get_incident_edge().unwrap().0, 1525);
    let v = output.vertices()[496];
    assert!(almost_equal(v.x(), -48991.1239107, v.y(), -49175.2930109));
    assert_eq!(v.get_incident_edge().unwrap().0, 1527);
    let v = output.vertices()[497];
    assert!(almost_equal(v.x(), -49023.9580277, v.y(), -49219.5529747));
    assert_eq!(v.get_incident_edge().unwrap().0, 1529);
    let v = output.vertices()[498];
    assert!(almost_equal(v.x(), -49057.4380120, v.y(), -49673.0680762));
    assert_eq!(v.get_incident_edge().unwrap().0, 1531);
    let v = output.vertices()[499];
    assert!(almost_equal(v.x(), -49021.5393519, v.y(), -49985.7800926));
    assert_eq!(v.get_incident_edge().unwrap().0, 1533);
    let v = output.vertices()[500];
    assert!(almost_equal(v.x(), -49832.2731533, v.y(), -50785.2477949));
    assert_eq!(v.get_incident_edge().unwrap().0, 1537);
    let v = output.vertices()[501];
    assert!(almost_equal(v.x(), -49016.3837209, v.y(), -49810.2200997));
    assert_eq!(v.get_incident_edge().unwrap().0, 1539);
    let v = output.vertices()[502];
    assert!(almost_equal(v.x(), -48998.3276384, v.y(), -49133.7257798));
    assert_eq!(v.get_incident_edge().unwrap().0, 1541);
    let v = output.vertices()[503];
    assert!(almost_equal(v.x(), -48993.7287108, v.y(), -49137.1642798));
    assert_eq!(v.get_incident_edge().unwrap().0, 1543);
    let v = output.vertices()[504];
    assert!(almost_equal(v.x(), -49000.8264097, v.y(), -49837.8582190));
    assert_eq!(v.get_incident_edge().unwrap().0, 1545);
    let v = output.vertices()[505];
    assert!(almost_equal(v.x(), -48987.0761897, v.y(), -49902.7143084));
    assert_eq!(v.get_incident_edge().unwrap().0, 1547);
    let v = output.vertices()[506];
    assert!(almost_equal(v.x(), -48976.5535683, v.y(), -49145.8423190));
    assert_eq!(v.get_incident_edge().unwrap().0, 1551);
    let v = output.vertices()[507];
    assert!(almost_equal(v.x(), -48967.7589744, v.y(), -49148.4972907));
    assert_eq!(v.get_incident_edge().unwrap().0, 1553);
    let v = output.vertices()[508];
    assert!(almost_equal(v.x(), -48989.8750000, v.y(), -49818.7500000));
    assert_eq!(v.get_incident_edge().unwrap().0, 1555);
    let v = output.vertices()[509];
    assert!(almost_equal(v.x(), -48951.9225538, v.y(), -49144.8655112));
    assert_eq!(v.get_incident_edge().unwrap().0, 1557);
    let v = output.vertices()[510];
    assert!(almost_equal(v.x(), -48984.2234927, v.y(), -49797.0114345));
    assert_eq!(v.get_incident_edge().unwrap().0, 1559);
    let v = output.vertices()[511];
    assert!(almost_equal(v.x(), -48978.9812207, v.y(), -49796.9624413));
    assert_eq!(v.get_incident_edge().unwrap().0, 1561);
    let v = output.vertices()[512];
    assert!(almost_equal(v.x(), -48967.3562003, v.y(), -49263.5705710));
    assert_eq!(v.get_incident_edge().unwrap().0, 1563);
    let v = output.vertices()[513];
    assert!(almost_equal(v.x(), -48941.9866273, v.y(), -49919.5774212));
    assert_eq!(v.get_incident_edge().unwrap().0, 1565);
    let v = output.vertices()[514];
    assert!(almost_equal(v.x(), -48948.1037625, v.y(), -49286.1607867));
    assert_eq!(v.get_incident_edge().unwrap().0, 1567);
    let v = output.vertices()[515];
    assert!(almost_equal(v.x(), -48941.3725490, v.y(), -49390.3398693));
    assert_eq!(v.get_incident_edge().unwrap().0, 1569);
    let v = output.vertices()[516];
    assert!(almost_equal(v.x(), -48939.7686659, v.y(), -49631.5875153));
    assert_eq!(v.get_incident_edge().unwrap().0, 1571);
    let v = output.vertices()[517];
    assert!(almost_equal(v.x(), -48879.3488372, v.y(), -49351.6009437));
    assert_eq!(v.get_incident_edge().unwrap().0, 1573);
    let v = output.vertices()[518];
    assert!(almost_equal(v.x(), -48786.0505415, v.y(), -49427.3212996));
    assert_eq!(v.get_incident_edge().unwrap().0, 1575);
    let v = output.vertices()[519];
    assert!(almost_equal(v.x(), -48781.8739290, v.y(), -49592.1138311));
    assert_eq!(v.get_incident_edge().unwrap().0, 1577);
    let v = output.vertices()[520];
    assert!(almost_equal(v.x(), -48750.2738201, v.y(), -49448.5939448));
    assert_eq!(v.get_incident_edge().unwrap().0, 1579);
    let v = output.vertices()[521];
    assert!(almost_equal(v.x(), -48641.0348189, v.y(), -48712.7869081));
    assert_eq!(v.get_incident_edge().unwrap().0, 1581);
    let v = output.vertices()[522];
    assert!(almost_equal(v.x(), -49222.2411977, v.y(), -47819.3010834));
    assert_eq!(v.get_incident_edge().unwrap().0, 1583);
    let v = output.vertices()[523];
    assert!(almost_equal(v.x(), -48486.3357337, v.y(), -49538.7727910));
    assert_eq!(v.get_incident_edge().unwrap().0, 1585);
    let v = output.vertices()[524];
    assert!(almost_equal(v.x(), -48345.8767811, v.y(), -49560.1523838));
    assert_eq!(v.get_incident_edge().unwrap().0, 1587);
    let v = output.vertices()[525];
    assert!(almost_equal(v.x(), -47443.7506964, v.y(), -47704.1657382));
    assert_eq!(v.get_incident_edge().unwrap().0, 1589);
    Ok(())
}

#[test]
fn sample_primary_072() -> Result<(), BvError> {
    let output: VS::SyncVoronoiDiagram<I1, F1> = {
        let input = r#"0
3
1403829871 74 1403829871 275
1403829871 275 1403829741 275
1403829741 275 1403829744 73
"#;
        let mut vb = VB::Builder::<I1, F1>::default();
        let br = BufReader::new(Cursor::new(input));
        let (points, segments) = FR::read_boost_input_buffer::<I1, _>(br)?;
        vb.with_vertices(points.iter())?;
        vb.with_segments(segments.iter())?;
        vb.construct()?.into()
    };
    assert_eq!(output.cells().len(), 7);
    let cell = output.cells()[0];
    assert_eq!(cell.get_id(), 0);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[1];
    assert_eq!(cell.get_id(), 1);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[2];
    assert_eq!(cell.get_id(), 2);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[3];
    assert_eq!(cell.get_id(), 3);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[4];
    assert_eq!(cell.get_id(), 4);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    let cell = output.cells()[5];
    assert_eq!(cell.get_id(), 5);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(_cat, VD::SourceCategory::Segment);
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), false);
    assert_eq!(cell.contains_segment(), true);
    let cell = output.cells()[6];
    assert_eq!(cell.get_id(), 6);
    let (_source_index, _cat) = cell.source_index_2();
    assert_eq!(cell.is_degenerate(), false);
    assert_eq!(cell.contains_point(), true);
    assert_eq!(cell.contains_segment(), false);
    assert_eq!(output.vertices().len(), 5);
    assert_eq!(output.edges().len(), 22);
    let v = output.vertices()[0];
    assert!(almost_equal(v.x(), 1403829741.0000000, v.y(), 275.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 5);
    let v = output.vertices()[1];
    assert!(almost_equal(v.x(), 1403829807.4960756, v.y(), 74.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 13);
    let v = output.vertices()[2];
    assert!(almost_equal(v.x(), 1403829806.4826467, v.y(), 210.4826467));
    assert_eq!(v.get_incident_edge().unwrap().0, 15);
    let v = output.vertices()[3];
    assert!(almost_equal(v.x(), 1403829871.0000000, v.y(), 275.0000000));
    assert_eq!(v.get_incident_edge().unwrap().0, 19);
    let v = output.vertices()[4];
    assert!(almost_equal(v.x(), 1403829807.4965117, v.y(), 73.9430175));
    assert_eq!(v.get_incident_edge().unwrap().0, 21);
    Ok(())
}
