//! Bombs, a whole load of 'em.
//! 
//! This crate provides single-producer multi-consumer communication types, where
//! `Fuse`s are the producers and `Bomb`s are the consumers.
//! 
//! [`Bomb`]s are one-time use only, whereas [`MultiBomb`]s can be reused.
//! 
//! # Example 
//! 
//! If you want to send a "close program" signal to different threads,
//! such that each thread needs to clean up first before closing,
//! you can give each thread a [`Bomb<()>`](Bomb), and run cleanup code
//! once exploded.
//! 
//! ```
//! use std::thread;
//! use bombs::Bomb;
//! 
//! // Create a new fuse and bomb pair.
//! let (fuse, bomb) = Bomb::new(); 
//! 
//! // Clone `bomb` into thread.
//! let bomb_clone = bomb.clone();
//! thread::spawn(move || {
//!     loop {
//!         // Do some stuff...
//!     
//!         if let Some(_) = bomb_clone.exploded() {
//!             // Received close signal, break.
//! 
//!             // Clean up data values...
//! 
//!             break;
//!         }
//!     }
//! });
//! 
//! // Create another thread.
//! // Move original `bomb` into thread.
//! thread::spawn(move || {
//!     loop {
//!         // Do some other stuff...
//!     
//!         if let Some(_) = bomb.exploded() {
//!             // Received close signal, break.
//! 
//!             // Clean up data values...
//! 
//!             break;
//!         }
//!     }
//! });
//! 
//! // Do some different stuff...
//! 
//! // Send close signal.
//! let fire = fuse.light(());
//! 
//! // Wait for all inner threads to close safely (checked by `Bomb` drop).
//! while !fire.extinguished() { }
//! 
//! // Now safely quit the program.
//! ```

mod multi; pub use multi::*;

use std::{
    sync::{ 
        atomic::{ AtomicBool, Ordering },
        Arc, Weak
    },
    mem::MaybeUninit,
    cell::UnsafeCell,
};

/// A type alias for [`Fuse<T>`].
pub type Fuze<T> = Fuse<T>;

struct OneTime<T> {
    pub has_value: AtomicBool,
    pub cell: UnsafeCell<MaybeUninit<T>>,
}

impl<T> OneTime<T> {
    pub fn new() -> Self {
        Self {
            has_value: AtomicBool::new(false),
            cell: UnsafeCell::new(MaybeUninit::uninit()),
        }
    }
}

impl<T> Drop for OneTime<T> {
    fn drop(&mut self) {
        if self.has_value.load(Ordering::Acquire) {
            // SAFETY: Guaranteed to have a value, it is safe to drop.

            unsafe {
                // Replace with uninitialised memory, which is safe to not drop.
                let value = self.cell.get().replace(MaybeUninit::uninit());
                // Acquire value and drop.
                value.assume_init();
            }
        }
    }
}

/// A one-time use bomb. 
/// 
/// The [`Fuse<T>`] for this `Bomb` can be lit only once, 
/// and the bomb will remain in an exploded state for the 
/// remainder of the existence of this instance.
/// 
/// `Bomb`s can be safely cloned and sent between threads.
pub struct Bomb<T> {
    data: Arc<OneTime<T>>,

    /// Empty pointer to count strong references (bombs).
    /// Used by [`Fuse`] to check if all bombs have been dropped.
    _counter: Arc<()>,
}

unsafe impl<T> Send for Bomb<T> { }
unsafe impl<T> Sync for Bomb<T> { }

/// A one-time use fuse.
/// 
/// This `Fuse` can only be lit once, and will explode all
/// [`Bomb<T>`] instances associated with it. The `Fuse` 
/// instance is consumed on use.
/// 
/// `Fuse`s cannot be cloned, but may be moved between threads.
pub struct Fuse<T> {
    data: Arc<OneTime<T>>,

    /// Weak pointer to bombs, to check if all bombs 
    /// associated with this fuse have been dropped.
    /// Empty type since we only care about strong count.
    _counter: Weak<()>
}

unsafe impl<T> Send for Fuse<T> { }
// SAFETY: lighting a `Fuse` is consuming and requires ownership, which
// ensures that there is only one instance modifying the internal state.
unsafe impl<T> Sync for Fuse<T> { }

/// Result of a bomb explosion.
/// 
/// `Fire` is a handle returned by [`Fuse<T>::light()`], which
/// allows to check whether all bombs have been extinguished (dropped).
/// 
/// [`Fuse<T>::light()`]: ../struct.Fuse.html#method.light
pub struct Fire {
    counter: Weak<()>,
}

impl<T> Bomb<T> {
    /// Creates a new single producer [`Fuse<T>`], and a mutli-consumer `Bomb<T>`.
    /// 
    /// Instances of `Bomb<T>` may be safely cloned and sent between threads.
    pub fn new() -> (Fuse<T>, Bomb<T>) {
        let _counter = Arc::new(());

        let fuse = Fuse::new(&_counter);

        let bomb = Self {
            data: fuse.data.clone(),
            _counter,
        };

        (fuse, bomb)
    }

    /// Returns `Some` if the `Bomb` has exploded.
    /// 
    /// Once the `Bomb` has exploded, it will remain
    /// in an exploded state for the remainder of the
    /// existence of this `Bomb` instance.
    /// It is expected that the `Bomb` is dropped after explosion,
    /// to notify the original [`Fuse`] (now [`Fire`]) that this `Bomb`
    /// has completed processing the data.
    pub fn exploded(&self) -> Option<&T> {
        // Check if the `Bomb` has exploded yet.
        // Return early if it has not.
        // This avoids the need to grab a raw pointer to the data cell.
        if !self.data.has_value.load(Ordering::Acquire) {
            None
        }
        else {
            // The `Bomb` has exploded, we have data.

            // SAFETY: Due to the implementation of `Fuse`,
            // if `exploded` is true, then there are no other
            // mutable references for the pointer to the cell,
            // hence it is safe to read across multiple threads.
            unsafe { Some(self.data.cell.get().as_ref().unwrap().assume_init_ref()) }
        }
    }
}

impl<T> Fuse<T> {
    fn new(_counter: &Arc<()>) -> Self {
        Self {
            data: Arc::new(OneTime::new()),

            _counter: Arc::downgrade(&_counter),
        }
    }

    /// Ignites the fuse.
    /// 
    /// Explodes all [`Bomb`]s associated to this `Fuse`.
    /// Each [`Bomb`] receives `value`.
    /// 
    /// Alias to [`light`](#method.light)
    pub fn ignite(self, value: T) -> Fire {
        self.light(value)
    }

    /// Lights the fuse.
    /// 
    /// Explodes all [`Bomb`]s associated to this `Fuse`.
    /// Each [`Bomb`] receives `value`.
    pub fn light(self, value: T) -> Fire {
        // SAFETY: We know that there do not exist 
        // other references to the value of the cell,
        // since `has_value` is still set to false,
        // and therefore any `Bomb`s will not attempt to get
        // a reference.
        unsafe { *self.data.cell.get() = MaybeUninit::new(value); }

        // Now we can set `has_value` to true, and
        // allow other `Bomb`s to get a reference safely.
        self.data.has_value.store(true, Ordering::Release);

        // At this point it is safe to drop this `Fuse`,
        // since the inner containers will remain in memory
        // due to other `Arc`s existing in the `Bomb`s.

        // Create a `Fire` handle, with a weak pointer to the counter.
        Fire {
            counter: self._counter
        }
    }
}

impl Fire {
    /// Checks if the exploded bombs have been extinguished (dropped).
    /// 
    /// For [`MultiBomb`]s, this will return `true` as soon as each bomb
    /// has exploded, such that [`MultiBomb<T>::exploded()`] returns `Some(T)`,
    /// as the inner `Bomb` is dropped immediately (and replaced with a new one).
    /// 
    /// [`MultiBomb<T>::exploded()`]: ../struct.MultiBomb.html#method.exploded
    pub fn extinguished(&self) -> bool {
        self.counter.strong_count() == 0
    }
}

// was getting weird errors with derive.
impl<T> Clone for Bomb<T> {
    fn clone(&self) -> Self {
        Self { data: self.data.clone(), _counter: self._counter.clone() }
    }
}