use std::sync::Arc;

use crate::{ Fuse, Bomb, Fire };

/// A reuseable bomb. 
/// *How did they manufacture this?*
/// 
/// The [`MultiFuse<T>`] for this `Bomb` can be lit multiple times, 
/// to transfer multiple data values sequentially.
/// The `Bomb` remains in an exploded state for one call per
/// data value sent. This means you can only retreive sent data once,
/// as then the `Bomb` will wait for the next data. A one-time use [`Bomb<T>`]
/// instead repeatedly returns the same one data value for subsequent
/// calls to [`exploded()`] after the [`Fuse`] has been lit.
/// 
/// Like one-time use `Bomb`s, `MultiBomb`s can be safely cloned and sent between threads.
/// 
/// [`exploded()`]: ../struct.Bomb.html#method.exploded
pub struct MultiBomb<T> {
    inner: Bomb<(MultiBomb<T>, T)>
}

/// A reusable fuse. 
/// *It just grows back.*
/// 
/// This `MultiFuse` can be lit multiple times, and will explode all
/// [`MultiBomb<T>`] instances associated with it every time.
/// 
/// It is not necessary, but may be useful for your application, to wait for
/// all of the [`MultiBomb`]s to extinguish before lighting the `MultiFuse` again.
/// Slow [`MultiBomb`]s may be in an exploded state again immediately after
/// they have finished processing data, if the `MultiFuse` was lit again in between processing.
/// 
/// Like one-time use `Fuse`s, `MultiFuse`s cannot be cloned, but may be moved between threads.
/// 
/// # Dropping
/// 
/// Currently, the `MultiFuse` and [`MultiBomb`] do not have internal logic to deal with drops.
/// This means that if you no longer need a `MultiFuse`/[`MultiBomb`] pair, you must ensure that
/// you drop the values properly.
/// 
/// If the `MultiFuse` is dropped without additional logic, all active [`MultiBomb`]s will wait
/// indefinitely to explode, which will become impossible. Therefore you should encorporate a
/// "Drop" message into your application, which drops the [`MultiBomb`]s you use, if you intend
/// to only use the `MultiFuse` temporarily.
/// 
/// You could even use a one-time use [`Bomb<()>`](Bomb) as a drop signal if you wanted to, though it 
/// may be redundant since [`MultiBomb`]s already use [`Bomb`]s internally.
pub struct MultiFuse<T> {
    // TODO: Implement proper drop behaviour, read above for further info. 
    inner: Fuse<(MultiBomb<T>, T)>
}

impl<T> MultiBomb<T> {
    /// Creates a new single but reusable producer [`MultiFuse<T>`], and a mutli-consumer `MultiBomb<T>`.
    /// 
    /// Instances of `MultiBomb<T>` may be safely cloned and sent between threads.
    pub fn new() -> (MultiFuse<T>, MultiBomb<T>) {
        let _counter = Arc::new(());

        let fuse = MultiFuse::new(&_counter);

        let bomb = Self {
            inner: Bomb {
                data: fuse.inner.data.clone(),

                _counter,
            }
        };

        (fuse, bomb)
    }

    /// Returns `Some` if the `MultiBomb` has exploded.
    /// 
    /// Once the `MultiBomb` has exploded, this function
    /// will return the value only once, and subsequent
    /// calls will check for the next explosion/data value sent.
    /// 
    /// # Dropping
    /// 
    /// If the `MultiBomb` is only needed temporarily, and then the
    /// `MultiFuse` is dropped, this function will **always** return `None`,
    /// which indistinguishable to waiting for data.
    /// You must implement your own drop behaviour to avoid this.
    /// 
    /// See [`MultiFuse`] for more details.
    pub fn exploded(&mut self) -> Option<T> {
        match self.inner.exploded() {
            None => None,

            Some((new_bomb, value)) => {
                // Force a clone to increment strong counter.
                self.inner = new_bomb.inner.clone();

                // Forget read value, as it would have been
                // copied, and therefore would break the reference
                // counter if allowed to drop.
                std::mem::forget(new_bomb);

                Some(value)
            }
        }
    }
}

impl<T> MultiFuse<T> {
    fn new(_counter: &Arc<()>) -> Self {
        Self {
            inner: Fuse::new(_counter),
        }
    }

    /// Lights the fuse.
    /// 
    /// Explodes all [`MultiBomb`]s associated to this `MultiFuse`.
    /// Each [`MultiBomb`] receives `value`.
    pub fn light(&mut self, value: T) -> Fire {
        // Create new fuse and bomb.
        let (fuse, bomb) = MultiBomb::new();

        // Replace old fuse with new.
        let old_fuse = std::mem::replace(&mut self.inner, fuse.inner);

        // Light old fuse, passing a new bomb with it.
        old_fuse.light((bomb, value))
    }
}

impl<T> Clone for MultiBomb<T> {
    fn clone(&self) -> Self {
        Self { inner: self.inner.clone() }
    }
}