/*!
This crate provides flexible clone on write smart pointers
with Borrowed, Owned, Arc and Rc variants.
They are like std's [`Cow`][std::borrow::Cow]
but with additonal [`Arc`][std::sync::Arc]
and [`Rc`][std::rc::Rc] variants.
Unlike std's [`Cow`][std::borrow::Cow],
they can also be used with types
that don't implement [`Clone`] or [`ToOwned`].

The `boar` smart pointers allow you to:

- share data efficiently
- decide ownership at runtime
- avoid cloning the data until it becomes necessary

# Difference between [`Boa`] and [`Boar`]

[`Boar`] has a [`Rc`][std::rc::Rc] variant,
but does *not* implement [`Send`] or [`Sync`].

[`Boa`] does *not* have a [`Rc`][std::rc::Rc] variant
but does implement [`Send`] or [`Sync`].

# [`BoaStr`] and [`BoarStr`] type alias

[`Boa`] and [`Boar`] are often used with [`str`] or similar types.
Feel free to have a look at our handy [type alias](#types) below.

# features

Enable additional features or optional dependencies
by adding the following to your `Cargo.toml`,
replacing "x.y" with the version you want.

```toml
[dependencies]
boar = { version = "x.y", features = ["serde"] }
```

# Examples

```rust
use std::sync::Arc;
use boar::BoaStr;

// No need to allocate memory for a &'static str
let title = BoaStr::Borrowed("Dune");

// The author's name shouldn't be too long.
// So we can just store it in a normal String.
let author = BoaStr::Owned(read_file("dune/author.txt"));

// A whole book is really long.
// And we want to share this string with multiple threads.
// So it would be efficient to store in an Arc,
// since cloning the Arc does not clone the String data inside of it.
let book_text = BoaStr::Arc(Arc::new(read_file("dune/book.txt")));

// We can store a &str, String and Arc<String> inside the same Vec
// because we are using the BoaStr enum.
let book_infos: Vec<BoaStr> = vec![title, author, book_text];

fn read_file(path: &str) -> String {
    // ...
# String::new()
}
```
*/
#![forbid(unsafe_code)]

mod boa;
mod boar;

#[cfg(feature = "serde")]
mod serde_support;

pub use self::boa::Boa;
pub use self::boar::Boar;

/// **B**orrowed, **O**wned or **A**rc smart pointer for types implementing the [`ToOwned`] (**T**.**O**.) trait.
pub type BoaTo<'b, B> = Boa<'b, B, <B as ToOwned>::Owned>;

/// **B**orrowed, **O**wned, **A**rc or **Rc** smart pointer for types implementing the [`ToOwned`] (**T**.**O**.) trait.
pub type BoarTo<'b, B> = Boar<'b, B, <B as ToOwned>::Owned>;

/// **B**orrowed, **O**wned or **A**rc heap allocating smart pointer.
pub type BoaBox<'b, T> = Boa<'b, T, Box<T>, T>;

/// **B**orrowed, **O**wned, **A**rc or **Rc** heap allocating smart pointer.
pub type BoarBox<'b, T> = Boar<'b, T, Box<T>, T>;

/// **B**orrowed, **O**wned or **A**rc [`str`] smart pointer.
pub type BoaStr<'b> = BoaTo<'b, str>;

/// **B**orrowed, **O**wned, **A**rc or **Rc** [`str`] smart pointer.
pub type BoarStr<'b> = BoarTo<'b, str>;

/// **B**orrowed, **O**wned or **A**rc [`Vec`] smart pointer.
pub type BoaVec<'b, T> = Boa<'b, [T], Vec<T>>;

/// **B**orrowed, **O**wned, **A**rc or **Rc** [`Vec`] smart pointer.
pub type BoarVec<'b, T> = Boar<'b, [T], Vec<T>>;
