use std::{
    borrow::{Borrow, BorrowMut, Cow},
    cmp::Ordering,
    collections::HashMap,
    ffi::{CStr, CString, OsStr, OsString},
    fmt,
    hash::Hash,
    iter::FromIterator,
    ops::{Add, AddAssign, Deref},
    path::{Path, PathBuf},
    rc::Rc,
    sync::Arc,
};

/// **B**orrowed, **O**wned, **A**rc or **R**c smart pointer.
#[non_exhaustive]
pub enum Boar<'b, B, O = B, S = O>
where
    B: ?Sized,
    S: ?Sized,
{
    Borrowed(&'b B),

    Owned(O),

    // unstable reason:
    // Variant may not be needed.
    // Arc::clone may be good enough.
    // Upgrading and downgrading lifetimes may be used rarely.
    #[cfg(feature = "unstable")]
    BorrowedArc(&'b Arc<S>),

    Arc(Arc<S>),

    // unstable reason:
    // Variant may not be needed.
    // Rc::clone may be good enough.
    // Upgrading and downgrading lifetimes may be used rarely.
    #[cfg(feature = "unstable")]
    BorrowedRc(&'b Rc<S>),

    Rc(Rc<S>),
}

impl<'b, B: ?Sized, O, S: ?Sized> Boar<'b, B, O, S> {
    /// Extracts the owned data.
    ///
    /// Clones the data if it is not already owned.
    ///
    /// # Examples
    ///
    /// Calling `into_owned` on a `Boar::Borrowed` clones the inner data.
    /// ```
    /// use boar::{Boar, BoarStr};
    ///
    /// let boar: BoarStr = Boar::Borrowed("Hello!");
    /// let owned: String = boar.into_owned();
    /// assert_eq!(owned, "Hello!");
    /// ```
    ///
    /// Calling `into_owned` on a `Boar::Owned` doesn't clone anything.
    /// ```
    /// use boar::{Boar, BoarStr};
    ///
    /// let boar: BoarStr = Boar::Owned("Hello!".to_owned());
    /// let owned: String = boar.into_owned();
    /// assert_eq!(owned, "Hello!");
    /// ```
    ///
    /// Calling `into_owned` on a `Boar::Arc` or `Boar::Rc` only clones the inner data
    /// if the reference count is greater than one.
    /// ```
    /// use boar::{Boar, BoarStr};
    ///
    /// let boar: BoarStr = Boar::Rc("Hello!".to_owned().into());
    /// let owned: String = boar.into_owned();
    /// assert_eq!(owned, "Hello!");
    /// ```
    #[inline]
    pub fn into_owned(self) -> O
    where
        B: ToOwned<Owned = O>,
        S: Clone + Into<O>,
    {
        match self {
            Self::Borrowed(x) => x.to_owned(),
            Self::Owned(x) => x,
            #[cfg(feature = "unstable")]
            Self::BorrowedArc(x) => x.deref().clone().into(),
            Self::Arc(x) => Arc::try_unwrap(x)
                .unwrap_or_else(|x| x.deref().clone())
                .into(),
            #[cfg(feature = "unstable")]
            Self::BorrowedRc(x) => x.deref().clone().into(),
            Self::Rc(x) => Rc::try_unwrap(x)
                .unwrap_or_else(|x| x.deref().clone())
                .into(),
        }
    }

    /// Acquires a mutable reference to the owned form of the data.
    ///
    /// Clones the data if it is not already owned.
    ///
    /// # Examples
    ///
    /// Calling `to_mut` on a `Boar::Borrowed` clones the inner data.
    /// ```
    /// use boar::{Boar, BoarStr};
    ///
    /// let mut hello: BoarStr = Boar::Borrowed("Hello");
    /// hello.to_mut().push_str(", Barbara!");
    /// assert_eq!(hello, "Hello, Barbara!");
    /// ```
    ///
    /// Calling `to_mut` on a `Boar::Owned` doesn't clone anything.
    /// ```
    /// use boar::{Boar, BoarStr};
    ///
    /// let mut hello: BoarStr = Boar::Owned("Hello".to_owned());
    /// hello.to_mut().push_str(", Barbara!");
    /// assert_eq!(hello, "Hello, Barbara!");
    /// ```
    ///
    /// Calling `to_mut` on a `Boar::Arc` only clones the inner data
    /// if the reference count is greater than one.
    /// ```
    /// use boar::{Boar, BoarStr};
    ///
    /// let mut hello: BoarStr = Boar::Arc("Hello".to_owned().into());
    /// hello.to_mut().push_str(", Barbara!");
    /// assert_eq!(hello, "Hello, Barbara!");
    /// ```
    #[inline]
    pub fn to_mut(&mut self) -> &mut O
    where
        B: ToOwned<Owned = O>,
        S: Clone + BorrowMut<O>,
    {
        match self {
            Self::Borrowed(x) => {
                *self = Self::Owned(x.to_owned());
                self.to_mut()
            }
            Self::Owned(x) => x,
            #[cfg(feature = "unstable")]
            Self::BorrowedArc(x) => {
                // avoid changing the reference count by immediately cloning S
                *self = Self::Arc(S::clone(x).into());
                self.to_mut()
            }
            Self::Arc(x) => Arc::make_mut(x).borrow_mut(),
            #[cfg(feature = "unstable")]
            Self::BorrowedRc(x) => {
                // avoid changing the reference count by immediately cloning S
                *self = Self::Rc(S::clone(x).into());
                self.to_mut()
            }
            Self::Rc(x) => Rc::make_mut(x).borrow_mut(),
        }
    }

    /**
    Convert borrowed variants so that the new `Boar` has a `'static` lifetime.

    # Examples

    ```rust
    # use boar::BoarStr;
    let b = BoarStr::Borrowed("a");
    let s = b.into_static();
    assert_eq!(s, "a");
    assert_matches::assert_matches!(s, BoarStr::Owned(_));
    ```
    */
    #[inline]
    pub fn into_static(self) -> Boar<'static, B, O, S>
    where
        B: ToOwned<Owned = O>,
    {
        match self {
            Self::Borrowed(x) => Boar::Owned(x.to_owned()),
            Self::Owned(x) => Boar::Owned(x),
            #[cfg(feature = "unstable")]
            Self::BorrowedArc(x) => Boar::Arc(x.clone()),
            Self::Arc(x) => Boar::Arc(x),
            #[cfg(feature = "unstable")]
            Self::BorrowedRc(x) => Boar::Rc(x.clone()),
            Self::Rc(x) => Boar::Rc(x),
        }
    }

    /**
    Convert to a new `Boar` that borrows from `self`.

    # Examples

    ```rust
    # use boar::BoarStr;
    let o = BoarStr::Owned("a".into());
    let b = o.to_borrowed();
    assert_eq!(b, "a");
    assert_matches::assert_matches!(b, BoarStr::Borrowed(_));
    ```

    ```rust
    # use boar::BoarStr;
    let a = BoarStr::Rc(String::from("a").into());
    let b = a.to_borrowed();
    assert_eq!(b, "a");
    assert_matches::assert_matches!(b, BoarStr::BorrowedRc(_));
    ```
    */
    // unstable reason:
    // Downgrading the lifetime may be used rarely.
    // Compatibility may be broken in rare cases, when BorrowedArc is stabilized.
    #[cfg(feature = "unstable")]
    #[inline]
    pub fn to_borrowed(&self) -> Boar<'_, B, O, S>
    where
        O: Borrow<B>,
    {
        match self {
            Self::Borrowed(x) => Boar::Borrowed(x),
            Self::Owned(x) => Boar::Borrowed(x.borrow()),
            Self::BorrowedArc(x) => Boar::BorrowedArc(x),
            Self::Arc(x) => Boar::BorrowedArc(x),
            Self::BorrowedRc(x) => Boar::BorrowedRc(x),
            Self::Rc(x) => Boar::BorrowedRc(x),
        }
    }

    /**
    Convert [`Boar::Owned`] into [`Boar::Rc`].

    # Examples

    ```rust
    # use boar::BoarStr;
    let o = BoarStr::Owned("a".into());
    let s = o.into_shared();
    assert_eq!(s, "a");
    assert_matches::assert_matches!(s, BoarStr::Rc(_));
    ```
    */
    #[inline]
    pub fn into_shared(self) -> Self
    where
        O: Into<Rc<S>>,
    {
        match self {
            Self::Owned(x) => Self::Rc(x.into()),
            _ => self,
        }
    }

    /// This is **unstable** and only available with the `unstable` feature.
    #[cfg(feature = "unstable")] // reason: unclear if actually needed; `make_shared` and `clone` can be used instead.
    #[inline]
    pub fn clone_shared(&mut self) -> Self
    where
        O: Default + Clone + Into<Rc<S>>,
    {
        self.internal_make_shared();
        self.clone()
    }

    /**
    Change [`Boar::Owned`] into [`Boar::Rc`].

    # Examples

    ```rust
    # use boar::BoarStr;
    let mut a = BoarStr::Owned("a".into());
    a.make_shared();
    assert_eq!(a, "a");
    assert_matches::assert_matches!(a, BoarStr::Rc(_));
    ```
    */
    #[inline]
    pub fn make_shared(&mut self)
    where
        O: Default,
        O: Into<Rc<S>>,
    {
        self.internal_make_shared()
    }

    #[inline]
    fn internal_make_shared(&mut self)
    where
        O: Default,
        O: Into<Rc<S>>,
    {
        if let Self::Owned(x) = self {
            let x = std::mem::take(x);
            *self = Self::Rc(x.into());
        }
    }

    #[inline]
    fn internal_borrow(&self) -> &B
    where
        O: Borrow<B>,
        S: Borrow<B>,
    {
        match self {
            Self::Borrowed(x) => x,
            Self::Owned(x) => x.borrow(),
            #[cfg(feature = "unstable")]
            Self::BorrowedArc(x) => x.deref().deref().borrow(),
            Self::Arc(x) => x.deref().borrow(),
            #[cfg(feature = "unstable")]
            Self::BorrowedRc(x) => x.deref().deref().borrow(),
            Self::Rc(x) => x.deref().borrow(),
        }
    }
}

impl<'b, B: ?Sized, O, S: ?Sized, Rhs> Add<Rhs> for Boar<'b, B, O, S>
where
    O: Add<Rhs>,
    B: ToOwned<Owned = O>,
    S: Clone + Into<O>,
{
    type Output = <O as Add<Rhs>>::Output;

    #[inline]
    fn add(self, rhs: Rhs) -> <O as Add<Rhs>>::Output {
        self.into_owned() + rhs
    }
}

impl<'b, B: ?Sized, O, S: ?Sized, Rhs> AddAssign<Rhs> for Boar<'b, B, O, S>
where
    O: AddAssign<Rhs>,
    B: ToOwned<Owned = O>,
    S: Clone + BorrowMut<O>,
{
    #[inline]
    fn add_assign(&mut self, rhs: Rhs) {
        *self.to_mut() += rhs;
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> AsRef<B> for Boar<'b, B, O, S>
where
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn as_ref(&self) -> &B {
        self.internal_borrow()
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> Borrow<B> for Boar<'b, B, O, S>
where
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn borrow(&self) -> &B {
        self.internal_borrow()
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> Deref for Boar<'b, B, O, S>
where
    O: Borrow<B>,
    S: Borrow<B>,
{
    type Target = B;

    #[inline]
    fn deref(&self) -> &Self::Target {
        self.internal_borrow()
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> Clone for Boar<'b, B, O, S>
where
    O: Clone,
{
    #[inline]
    fn clone(&self) -> Self {
        match self {
            Self::Borrowed(x) => Self::Borrowed(x),
            Self::Owned(x) => Self::Owned(x.clone()),
            #[cfg(feature = "unstable")]
            Self::BorrowedArc(x) => Self::BorrowedArc(x),
            Self::Arc(x) => Self::Arc(x.clone()),
            #[cfg(feature = "unstable")]
            Self::BorrowedRc(x) => Self::BorrowedRc(x),
            Self::Rc(x) => Self::Rc(x.clone()),
        }
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> Default for Boar<'b, B, O, S>
where
    O: Default,
{
    /// Creates an owned Boar with the default value for the contained owned type.
    #[inline]
    fn default() -> Self {
        Self::Owned(O::default())
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> fmt::Debug for Boar<'b, B, O, S>
where
    B: fmt::Debug,
    O: fmt::Debug,
    S: fmt::Debug,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Boar::Borrowed(x) => f.debug_tuple("Boar::Borrowed").field(x).finish(),
            Boar::Owned(x) => f.debug_tuple("Boar::Owned").field(x).finish(),
            #[cfg(feature = "unstable")]
            Boar::BorrowedArc(x) => f.debug_tuple("Boar::BorrowedArc").field(x).finish(),
            Boar::Arc(x) => f.debug_tuple("Boar::Arc").field(x).finish(),
            #[cfg(feature = "unstable")]
            Boar::BorrowedRc(x) => f.debug_tuple("Boar::BorrowedRc").field(x).finish(),
            Boar::Rc(x) => f.debug_tuple("Boar::Rc").field(x).finish(),
        }
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> fmt::Display for Boar<'b, B, O, S>
where
    B: fmt::Display,
    O: fmt::Display,
    S: fmt::Display,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::Borrowed(x) => fmt::Display::fmt(x, f),
            Self::Owned(x) => fmt::Display::fmt(x, f),
            #[cfg(feature = "unstable")]
            Self::BorrowedArc(x) => fmt::Display::fmt(x, f),
            Self::Arc(x) => fmt::Display::fmt(x, f),
            #[cfg(feature = "unstable")]
            Self::BorrowedRc(x) => fmt::Display::fmt(x, f),
            Self::Rc(x) => fmt::Display::fmt(x, f),
        }
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> Hash for Boar<'b, B, O, S>
where
    B: Hash,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn hash<H: std::hash::Hasher>(&self, state: &mut H) {
        self.internal_borrow().hash(state)
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> PartialEq<&B> for Boar<'b, B, O, S>
where
    B: PartialEq,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn eq(&self, other: &&B) -> bool {
        self.internal_borrow() == *other
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> PartialEq<B> for Boar<'b, B, O, S>
where
    B: PartialEq,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn eq(&self, other: &B) -> bool {
        self.internal_borrow() == other
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> PartialEq for Boar<'b, B, O, S>
where
    B: PartialEq,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn eq(&self, other: &Self) -> bool {
        self.internal_borrow() == other.internal_borrow()
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> Eq for Boar<'b, B, O, S>
where
    B: Eq,
    O: Borrow<B>,
    S: Borrow<B>,
{
}

impl<'b, B: ?Sized, O, S: ?Sized> PartialOrd<&B> for Boar<'b, B, O, S>
where
    B: PartialOrd,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn partial_cmp(&self, other: &&B) -> Option<Ordering> {
        self.internal_borrow().partial_cmp(*other)
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> PartialOrd<B> for Boar<'b, B, O, S>
where
    B: PartialOrd,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn partial_cmp(&self, other: &B) -> Option<Ordering> {
        self.internal_borrow().partial_cmp(other)
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> PartialOrd for Boar<'b, B, O, S>
where
    B: PartialOrd,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        self.internal_borrow().partial_cmp(other.internal_borrow())
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> Ord for Boar<'b, B, O, S>
where
    B: Ord,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn cmp(&self, other: &Self) -> Ordering {
        self.internal_borrow().cmp(other.internal_borrow())
    }
}

impl<'b, A, B: ?Sized, O, S: ?Sized> FromIterator<A> for Boar<'b, B, O, S>
where
    O: FromIterator<A>,
{
    #[inline]
    fn from_iter<T: IntoIterator<Item = A>>(iter: T) -> Self {
        Self::Owned(iter.into_iter().collect())
    }
}

impl<'b, B: ToOwned + ?Sized + 'b> From<Cow<'b, B>> for Boar<'b, B, B::Owned, B::Owned> {
    #[inline]
    fn from(p: Cow<'b, B>) -> Self {
        match p {
            Cow::Borrowed(x) => Self::Borrowed(x),
            Cow::Owned(x) => Self::Owned(x),
        }
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> From<&'b B> for Boar<'b, B, O, S> {
    #[inline]
    fn from(x: &'b B) -> Self {
        Self::Borrowed(x)
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> From<Arc<S>> for Boar<'b, B, O, S> {
    #[inline]
    fn from(x: Arc<S>) -> Self {
        Self::Arc(x)
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> From<Rc<S>> for Boar<'b, B, O, S> {
    #[inline]
    fn from(x: Rc<S>) -> Self {
        Self::Rc(x)
    }
}

impl<'b, T> From<Vec<T>> for Boar<'b, [T], Vec<T>> {
    #[inline]
    fn from(x: Vec<T>) -> Self {
        Self::Owned(x)
    }
}

impl<'b, T> From<Boar<'b, [T], Vec<T>>> for Vec<T>
where
    T: Clone,
    [T]: ToOwned<Owned = Vec<T>>,
    Vec<T>: Clone,
{
    #[inline]
    fn from(x: Boar<'b, [T], Vec<T>>) -> Self {
        x.into_owned()
    }
}

impl<'b, K, V, S> From<HashMap<K, V, S>> for Boar<'b, HashMap<K, V, S>> {
    #[inline]
    fn from(x: HashMap<K, V, S>) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<String> for Boar<'b, str, String> {
    #[inline]
    fn from(x: String) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Boar<'b, str, String>> for String {
    #[inline]
    fn from(x: Boar<'b, str, String>) -> Self {
        x.into_owned()
    }
}

impl<'b> From<CString> for Boar<'b, CStr, CString> {
    #[inline]
    fn from(x: CString) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Boar<'b, CStr, CString>> for CString {
    #[inline]
    fn from(x: Boar<'b, CStr, CString>) -> Self {
        x.into_owned()
    }
}

impl<'b> From<OsString> for Boar<'b, OsStr, OsString> {
    #[inline]
    fn from(x: OsString) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Boar<'b, OsStr, OsString>> for OsString {
    #[inline]
    fn from(x: Boar<'b, OsStr, OsString>) -> Self {
        x.into_owned()
    }
}

impl<'b> From<PathBuf> for Boar<'b, Path, PathBuf> {
    #[inline]
    fn from(x: PathBuf) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Boar<'b, Path, PathBuf>> for PathBuf {
    #[inline]
    fn from(x: Boar<'b, Path, PathBuf>) -> Self {
        x.into_owned()
    }
}

impl<'b, B, O, S, T, const N: usize> From<[T; N]> for Boar<'b, B, O, S>
where
    B: ?Sized,
    O: From<[T; N]>,
    S: ?Sized,
{
    #[inline]
    fn from(x: [T; N]) -> Self {
        Self::Owned(x.into())
    }
}
