use std::{
    borrow::{Borrow, BorrowMut, Cow},
    cmp::Ordering,
    collections::HashMap,
    ffi::{CStr, CString, OsStr, OsString},
    fmt,
    hash::Hash,
    iter::FromIterator,
    ops::{Add, AddAssign, Deref},
    path::{Path, PathBuf},
    sync::Arc,
};

/// **B**orrowed, **O**wned or **A**rc smart pointer.
#[non_exhaustive]
pub enum Boa<'b, B, O = B, S = O>
where
    B: ?Sized,
    S: ?Sized,
{
    Borrowed(&'b B),

    Owned(O),

    // unstable reason:
    // Variant may not be needed.
    // Arc::clone may be good enough.
    // Upgrading and downgrading lifetimes may be used rarely.
    #[cfg(feature = "unstable")]
    BorrowedArc(&'b Arc<S>),

    Arc(Arc<S>),
}

impl<'b, B: ?Sized, O, S: ?Sized> Boa<'b, B, O, S> {
    /// Extracts the owned data.
    ///
    /// Clones the data if it is not already owned.
    ///
    /// # Examples
    ///
    /// Calling `into_owned` on a `Boa::Borrowed` clones the inner data.
    /// ```
    /// use boar::{Boa, BoaStr};
    ///
    /// let boa: BoaStr = Boa::Borrowed("Hello!");
    /// let owned: String = boa.into_owned();
    /// assert_eq!(owned, "Hello!");
    /// ```
    ///
    /// Calling `into_owned` on a `Boa::Owned` doesn't clone anything.
    /// ```
    /// use boar::{Boa, BoaStr};
    ///
    /// let boa: BoaStr = Boa::Owned("Hello!".to_owned());
    /// let owned: String = boa.into_owned();
    /// assert_eq!(owned, "Hello!");
    /// ```
    ///
    /// Calling `into_owned` on a `Boa::Arc` only clones the inner data
    /// if the reference count is greater than one.
    /// ```
    /// use boar::{Boa, BoaStr};
    ///
    /// let boa: BoaStr = Boa::Arc("Hello!".to_owned().into());
    /// let owned: String = boa.into_owned();
    /// assert_eq!(owned, "Hello!");
    /// ```
    #[inline]
    pub fn into_owned(self) -> O
    where
        B: ToOwned<Owned = O>,
        S: Clone + Into<O>,
    {
        match self {
            Self::Borrowed(x) => x.to_owned(),
            Self::Owned(x) => x,
            #[cfg(feature = "unstable")]
            Self::BorrowedArc(x) => x.deref().clone().into(),
            Self::Arc(x) => Arc::try_unwrap(x)
                .unwrap_or_else(|x| x.deref().clone())
                .into(),
        }
    }

    /// Acquires a mutable reference to the owned form of the data.
    ///
    /// Clones the data if it is not already owned.
    ///
    /// # Examples
    ///
    /// Calling `to_mut` on an `Boa::Borrowed` clones the inner data.
    /// ```
    /// use boar::{Boa, BoaStr};
    ///
    /// let mut hello: BoaStr = Boa::Borrowed("Hello");
    /// hello.to_mut().push_str(", Barbara!");
    /// assert_eq!(hello, "Hello, Barbara!");
    /// ```
    ///
    /// Calling `to_mut` on an `Boa::Owned` doesn't clone anything.
    /// ```
    /// use boar::{Boa, BoaStr};
    ///
    /// let mut hello: BoaStr = Boa::Owned("Hello".to_owned());
    /// hello.to_mut().push_str(", Barbara!");
    /// assert_eq!(hello, "Hello, Barbara!");
    /// ```
    ///
    /// Calling `to_mut` on an `Boa::Arc` only clones the inner data
    /// if the reference count is greater than one.
    /// ```
    /// use boar::{Boa, BoaStr};
    ///
    /// let mut hello: BoaStr = Boa::Arc("Hello".to_owned().into());
    /// hello.to_mut().push_str(", Barbara!");
    /// assert_eq!(hello, "Hello, Barbara!");
    /// ```
    #[inline]
    pub fn to_mut(&mut self) -> &mut O
    where
        B: ToOwned<Owned = O>,
        S: Clone + BorrowMut<O>,
    {
        match self {
            Self::Borrowed(x) => {
                *self = Boa::Owned(x.to_owned());
                self.to_mut()
            }
            Self::Owned(x) => x,
            #[cfg(feature = "unstable")]
            Self::BorrowedArc(x) => {
                // avoid changing the reference count by immediately cloning S
                *self = Self::Arc(S::clone(x).into());
                self.to_mut()
            }
            Self::Arc(x) => Arc::make_mut(x).borrow_mut(),
        }
    }

    /**
    Convert borrowed variants so that the new `Boa` has a `'static` lifetime.

    # Examples

    ```rust
    # use boar::BoaStr;
    let b = BoaStr::Borrowed("a");
    let s = b.into_static();
    assert_eq!(s, "a");
    assert_matches::assert_matches!(s, BoaStr::Owned(_));
    ```
    */
    #[cfg(feature = "unstable")] // reason: Upgrading the lifetime may be used rarely.
    #[inline]
    pub fn into_static(self) -> Boa<'static, B, O, S>
    where
        B: ToOwned<Owned = O>,
    {
        match self {
            Self::Borrowed(x) => Boa::Owned(x.to_owned()),
            Self::Owned(x) => Boa::Owned(x),
            #[cfg(feature = "unstable")]
            Self::BorrowedArc(x) => Boa::Arc(x.clone()),
            Self::Arc(x) => Boa::Arc(x),
        }
    }

    /// Extracts or converts to an [`Arc`].
    ///
    /// Converting `B` or `O` to `Arc<S>` may clone the data.
    #[inline]
    pub fn into_arc(self) -> Arc<S>
    where
        for<'a> &'a B: Into<Arc<S>>,
        O: Into<Arc<S>>,
    {
        match self {
            Self::Borrowed(x) => x.into(),
            Self::Owned(x) => x.into(),
            #[cfg(feature = "unstable")]
            Self::BorrowedArc(x) => x.clone(),
            Self::Arc(x) => x,
        }
    }

    /**
    Convert to a new `Boa` that borrows from `self`.

    # Examples

    ```rust
    # use boar::BoaStr;
    let o = BoaStr::Owned("a".into());
    let b = o.to_borrowed();
    assert_eq!(b, "a");
    assert_matches::assert_matches!(b, BoaStr::Borrowed(_));
    ```

    ```rust
    # use boar::BoaStr;
    let a = BoaStr::Arc(String::from("a").into());
    let b = a.to_borrowed();
    assert_eq!(b, "a");
    assert_matches::assert_matches!(b, BoaStr::BorrowedArc(_));
    ```
    */
    // unstable reason:
    // Upgrading the lifetime may be used rarely.
    // Compatibility may be broken in rare cases, when BorrowedArc is stabilized.
    #[cfg(feature = "unstable")]
    #[inline]
    pub fn to_borrowed(&self) -> Boa<'_, B, O, S>
    where
        O: Borrow<B>,
    {
        match self {
            Self::Borrowed(x) => Boa::Borrowed(x),
            Self::Owned(x) => Boa::Borrowed(x.borrow()),
            Self::BorrowedArc(x) => Boa::BorrowedArc(x),
            Self::Arc(x) => Boa::BorrowedArc(x),
        }
    }

    /**
    Convert [`Boa::Owned`] into [`Boa::Arc`].

    # Examples

    ```rust
    # use boar::BoaStr;
    let o = BoaStr::Owned("a".into());
    let s = o.into_shared();
    assert_eq!(s, "a");
    assert_matches::assert_matches!(s, BoaStr::Arc(_));
    ```
    */
    #[inline]
    pub fn into_shared(self) -> Self
    where
        O: Into<Arc<S>>,
    {
        match self {
            Self::Owned(x) => Self::Arc(x.into()),
            _ => self,
        }
    }

    /// This is **unstable** and only available with the `unstable` feature.
    #[cfg(feature = "unstable")] // reason: unclear if actually needed; `make_shared` and `clone` can be used instead.
    #[inline]
    pub fn clone_shared(&mut self) -> Self
    where
        O: Default + Clone + Into<Arc<S>>,
    {
        self.internal_make_shared();
        self.clone()
    }

    /**
    Change [`Boa::Owned`] into [`Boa::Arc`].

    # Examples

    ```rust
    # use boar::BoaStr;
    let mut a = BoaStr::Owned("a".into());
    a.make_shared();
    assert_eq!(a, "a");
    assert_matches::assert_matches!(a, BoaStr::Arc(_));
    ```
    */
    #[inline]
    pub fn make_shared(&mut self)
    where
        O: Default + Into<Arc<S>>,
    {
        self.internal_make_shared()
    }

    #[inline]
    fn internal_make_shared(&mut self)
    where
        O: Default + Into<Arc<S>>,
    {
        if let Self::Owned(x) = self {
            let x = std::mem::take(x);
            *self = Self::Arc(x.into());
        }
    }

    #[inline]
    fn internal_borrow(&self) -> &B
    where
        O: Borrow<B>,
        S: Borrow<B>,
    {
        match self {
            Self::Borrowed(x) => x,
            Self::Owned(x) => x.borrow(),
            #[cfg(feature = "unstable")]
            Self::BorrowedArc(x) => x.deref().deref().borrow(),
            Self::Arc(x) => x.deref().borrow(),
        }
    }
}

impl<'b, B: ?Sized, O, S: ?Sized, Rhs> Add<Rhs> for Boa<'b, B, O, S>
where
    O: Add<Rhs>,
    B: ToOwned<Owned = O>,
    S: Clone + Into<O>,
{
    type Output = <O as Add<Rhs>>::Output;

    #[inline]
    fn add(self, rhs: Rhs) -> <O as Add<Rhs>>::Output {
        self.into_owned() + rhs
    }
}

impl<'b, B: ?Sized, O, S: ?Sized, Rhs> AddAssign<Rhs> for Boa<'b, B, O, S>
where
    O: AddAssign<Rhs>,
    B: ToOwned<Owned = O>,
    S: Clone + BorrowMut<O>,
{
    #[inline]
    fn add_assign(&mut self, rhs: Rhs) {
        *self.to_mut() += rhs;
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> AsRef<B> for Boa<'b, B, O, S>
where
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn as_ref(&self) -> &B {
        self.internal_borrow()
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> Borrow<B> for Boa<'b, B, O, S>
where
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn borrow(&self) -> &B {
        self.internal_borrow()
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> Deref for Boa<'b, B, O, S>
where
    O: Borrow<B>,
    S: Borrow<B>,
{
    type Target = B;

    #[inline]
    fn deref(&self) -> &Self::Target {
        self.internal_borrow()
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> Clone for Boa<'b, B, O, S>
where
    O: Clone,
{
    #[inline]
    fn clone(&self) -> Self {
        match self {
            Self::Borrowed(x) => Self::Borrowed(x),
            Self::Owned(x) => Self::Owned(x.clone()),
            #[cfg(feature = "unstable")]
            Self::BorrowedArc(x) => Self::BorrowedArc(x),
            Self::Arc(x) => Self::Arc(x.clone()),
        }
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> Default for Boa<'b, B, O, S>
where
    O: Default,
{
    /// Creates an owned Boa with the default value for the contained owned type.
    #[inline]
    fn default() -> Self {
        Self::Owned(O::default())
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> fmt::Debug for Boa<'b, B, O, S>
where
    B: fmt::Debug,
    O: fmt::Debug,
    S: fmt::Debug,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Boa::Borrowed(x) => f.debug_tuple("Boa::Borrowed").field(x).finish(),
            Boa::Owned(x) => f.debug_tuple("Boa::Owned").field(x).finish(),
            #[cfg(feature = "unstable")]
            Boa::BorrowedArc(x) => f.debug_tuple("Boa::BorrowedArc").field(x).finish(),
            Boa::Arc(x) => f.debug_tuple("Boa::Arc").field(x).finish(),
        }
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> fmt::Display for Boa<'b, B, O, S>
where
    B: fmt::Display,
    O: fmt::Display,
    S: fmt::Display,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::Borrowed(x) => fmt::Display::fmt(x, f),
            Self::Owned(x) => fmt::Display::fmt(x, f),
            #[cfg(feature = "unstable")]
            Self::BorrowedArc(x) => fmt::Display::fmt(x, f),
            Self::Arc(x) => fmt::Display::fmt(x, f),
        }
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> Hash for Boa<'b, B, O, S>
where
    B: Hash,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn hash<H: std::hash::Hasher>(&self, state: &mut H) {
        self.internal_borrow().hash(state)
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> PartialEq<&B> for Boa<'b, B, O, S>
where
    B: PartialEq,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn eq(&self, other: &&B) -> bool {
        self.internal_borrow() == *other
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> PartialEq<B> for Boa<'b, B, O, S>
where
    B: PartialEq,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn eq(&self, other: &B) -> bool {
        self.internal_borrow() == other
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> PartialEq for Boa<'b, B, O, S>
where
    B: PartialEq,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn eq(&self, other: &Self) -> bool {
        self.internal_borrow() == other.internal_borrow()
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> Eq for Boa<'b, B, O, S>
where
    B: Eq,
    O: Borrow<B>,
    S: Borrow<B>,
{
}

impl<'b, B: ?Sized, O, S: ?Sized> PartialOrd<&B> for Boa<'b, B, O, S>
where
    B: PartialOrd,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn partial_cmp(&self, other: &&B) -> Option<Ordering> {
        self.internal_borrow().partial_cmp(*other)
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> PartialOrd<B> for Boa<'b, B, O, S>
where
    B: PartialOrd,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn partial_cmp(&self, other: &B) -> Option<Ordering> {
        self.internal_borrow().partial_cmp(other)
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> PartialOrd for Boa<'b, B, O, S>
where
    B: PartialOrd,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        self.internal_borrow().partial_cmp(other.internal_borrow())
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> Ord for Boa<'b, B, O, S>
where
    B: Ord,
    O: Borrow<B>,
    S: Borrow<B>,
{
    #[inline]
    fn cmp(&self, other: &Self) -> Ordering {
        self.internal_borrow().cmp(other.internal_borrow())
    }
}

impl<'b, A, B: ?Sized, O, S: ?Sized> FromIterator<A> for Boa<'b, B, O, S>
where
    O: FromIterator<A>,
{
    #[inline]
    fn from_iter<T: IntoIterator<Item = A>>(iter: T) -> Self {
        Self::Owned(iter.into_iter().collect())
    }
}

impl<'b, B: ToOwned + ?Sized + 'b> From<Cow<'b, B>> for Boa<'b, B, B::Owned, B::Owned> {
    #[inline]
    fn from(p: Cow<'b, B>) -> Self {
        match p {
            Cow::Borrowed(x) => Self::Borrowed(x),
            Cow::Owned(x) => Self::Owned(x),
        }
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> From<&'b B> for Boa<'b, B, O, S> {
    #[inline]
    fn from(x: &'b B) -> Self {
        Self::Borrowed(x)
    }
}

impl<'b, B: ?Sized, O, S: ?Sized> From<Arc<S>> for Boa<'b, B, O, S> {
    #[inline]
    fn from(x: Arc<S>) -> Self {
        Self::Arc(x)
    }
}

impl<'b, T> From<Vec<T>> for Boa<'b, [T], Vec<T>> {
    #[inline]
    fn from(x: Vec<T>) -> Self {
        Self::Owned(x)
    }
}

impl<'b, T> From<Boa<'b, [T], Vec<T>>> for Vec<T>
where
    T: Clone,
    [T]: ToOwned<Owned = Vec<T>>,
    Vec<T>: Clone,
{
    #[inline]
    fn from(x: Boa<'b, [T], Vec<T>>) -> Self {
        x.into_owned()
    }
}

impl<'b, K, V, S> From<HashMap<K, V, S>> for Boa<'b, HashMap<K, V, S>> {
    #[inline]
    fn from(x: HashMap<K, V, S>) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<String> for Boa<'b, str, String> {
    #[inline]
    fn from(x: String) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Boa<'b, str, String>> for String {
    #[inline]
    fn from(x: Boa<'b, str, String>) -> Self {
        x.into_owned()
    }
}

impl<'b> From<CString> for Boa<'b, CStr, CString> {
    #[inline]
    fn from(x: CString) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Boa<'b, CStr, CString>> for CString {
    #[inline]
    fn from(x: Boa<'b, CStr, CString>) -> Self {
        x.into_owned()
    }
}

impl<'b> From<OsString> for Boa<'b, OsStr, OsString> {
    #[inline]
    fn from(x: OsString) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Boa<'b, OsStr, OsString>> for OsString {
    #[inline]
    fn from(x: Boa<'b, OsStr, OsString>) -> Self {
        x.into_owned()
    }
}

impl<'b> From<PathBuf> for Boa<'b, Path, PathBuf> {
    #[inline]
    fn from(x: PathBuf) -> Self {
        Self::Owned(x)
    }
}

impl<'b> From<Boa<'b, Path, PathBuf>> for PathBuf {
    #[inline]
    fn from(x: Boa<'b, Path, PathBuf>) -> Self {
        x.into_owned()
    }
}
