use crate::Error;
use std::collections::{BTreeMap, btree_map};
use std::sync::Arc;
use std::time::Duration;
use tokio::sync::{mpsc, Mutex};
use tokio::task;
use uuid::Uuid;

const ERR_LOCK_NOT_DEFINED: &str = "Lock not defined";
const ERR_INVALID_LOCK_TOKEN: &str = "Invalid lock token";

#[derive(Debug)]
pub struct SharedLock {
    lock: Arc<Mutex<Option<bool>>>,
}

#[derive(Debug, Clone)]
pub struct Lock {
    unlock_trigger: mpsc::Sender<bool>,
}

impl Lock {
    pub async fn release(&self) -> bool {
        // true if released, false if not locked
        self.unlock_trigger.send(true).await.is_ok()
    }
}

impl Default for SharedLock {
    fn default() -> Self {
        Self::new()
    }
}

impl SharedLock {
    #[must_use]
    pub fn new() -> Self {
        Self {
            lock: Arc::new(Mutex::new(None)),
        }
    }

    pub async fn acquire(&mut self, expire: Duration) -> Lock {
        let lock = self.lock.clone();
        let (lock_trigger, lock_listener) = triggered::trigger();
        let (unlock_trigger, mut unlock_listener) = mpsc::channel(1);
        task::spawn(async move {
            // guard moved here
            let _g = lock.lock().await;
            // triggered as soon as the lock is acquired
            lock_trigger.trigger();
            // exited as soon as unlocked or expired or unlock_trigger dropped
            let _ = tokio::time::timeout(expire, unlock_listener.recv()).await;
        });
        // want lock to be acquired
        lock_listener.await;
        Lock { unlock_trigger }
    }
}

#[derive(Debug)]
pub struct SharedLockFactory {
    shared_locks: BTreeMap<String, Mutex<SharedLock>>,
    locks: Mutex<BTreeMap<String, (Uuid, Lock)>>,
}

impl Default for SharedLockFactory {
    fn default() -> Self {
        Self::new()
    }
}

impl SharedLockFactory {
    #[must_use]
    pub fn new() -> Self {
        Self {
            shared_locks: BTreeMap::new(),
            locks: Mutex::new(BTreeMap::new()),
        }
    }

    /// # Errors
    ///
    /// Will return `Err` if the lock already exists
    pub fn create(&mut self, lock_id: &str) -> Result<(), Error> {
        if let btree_map::Entry::Vacant(x) =
            self.shared_locks.entry(lock_id.to_owned())
        {
            x.insert(Mutex::new(SharedLock::new()));
            Ok(())
        } else {
            Err(Error::duplicate(format!(
                "Shared lock {} already exists",
                lock_id
            )))
        }
    }

    /// # Errors
    ///
    /// Will return `Err` if the lock is not defined or the operation is timed out
    pub async fn safe_acquire(
        &self,
        lock_id: &str,
        expire: Duration,
        timeout: Duration,
    ) -> Result<Uuid, Error> {
        async fn acquire_lock(shared_lock: &Mutex<SharedLock>, expire: Duration) -> Lock {
            let mut lock = shared_lock.lock().await;
            lock.acquire(expire).await
        }

        match self.shared_locks.get(lock_id) {
            Some(shared_lock) => {
                match tokio::time::timeout(timeout, acquire_lock(shared_lock, expire)).await {
                    Ok(lock) => {
                        let token = Uuid::new_v4();
                        self.locks
                            .lock()
                            .await
                            .insert(lock_id.to_owned(), (token, lock));
                        Ok(token)
                    }
                    Err(_) => Err(Error::timeout()),
                }
            }
            None => Err(Error::not_found(ERR_LOCK_NOT_DEFINED)),
        }
    }

    /// # Errors
    ///
    /// Will return `Err` if the lock is not defined
    pub async fn acquire(&self, lock_id: &str, expire: Duration) -> Result<Uuid, Error> {
        match self.shared_locks.get(lock_id) {
            Some(v) => {
                let lock = v.lock().await.acquire(expire).await;
                let token = Uuid::new_v4();
                self.locks
                    .lock()
                    .await
                    .insert(lock_id.to_owned(), (token, lock));
                Ok(token)
            }
            None => Err(Error::not_found(ERR_LOCK_NOT_DEFINED)),
        }
    }

    /// # Errors
    ///
    /// Will return `Err` if the lock is not defined or the token is invalid
    pub async fn release(&self, lock_id: &str, token: Option<Uuid>) -> Result<bool, Error> {
        match self.locks.lock().await.get(lock_id) {
            Some((tok, lock)) => {
                if let Some(ref t) = token {
                    if t != tok {
                        return Err(Error::not_found(ERR_INVALID_LOCK_TOKEN));
                    }
                }
                Ok(lock.release().await)
            }
            None => Err(Error::not_found(ERR_LOCK_NOT_DEFINED)),
        }
    }
}
