#[derive(Debug, Clone)]
pub struct ScriptPubKey {
  pub asm: String,
  pub hex: String,
  pub script_type: String,
  pub address: String
}

impl ScriptPubKey {
  pub fn new(asm: &str, hex: &str, script_type: &str, address: &str) -> ScriptPubKey {
    return ScriptPubKey {
      asm: String::from(asm),
      hex: String::from(hex),
      script_type: String::from(script_type),
      address: String::from(address)
    };
  }
}

#[derive(Debug, Clone)]
pub struct Vout {
  pub value: f32,
  pub n: u32,
  pub scriptPubKey: ScriptPubKey
}

impl Vout {
  pub fn new(json_obj: &json::JsonValue) -> Vout {
    let script_pub_key = ScriptPubKey::new(
      json_obj["scriptPubKey"]["asm"].as_str().unwrap(),
      json_obj["scriptPubKey"]["hex"].as_str().unwrap(),
      json_obj["scriptPubKey"]["type"].as_str().unwrap(),
      if (json_obj["scriptPubKey"].has_key("address")) {
        json_obj["scriptPubKey"]["address"].as_str().unwrap()
      } else {
        ""
      }
    );

    return Vout {
      value: json_obj["value"].as_f32().unwrap(),
      n: json_obj["n"].as_u32().unwrap(),
      scriptPubKey: script_pub_key
    };
  }
}

#[derive(Debug, Clone)]
pub struct ScriptSig {
  pub asm: String,
  pub hex: String
}

impl ScriptSig {
  pub fn new(asm: &str, hex: &str) -> ScriptSig {
    return ScriptSig {
      asm: String::from(asm),
      hex: String::from(hex)
    };
  }
}

#[derive(Debug, Clone)]
pub struct Vin {
  pub coinbase: String,
  pub txid: String,
  pub vout: i32,
  pub scriptSig: ScriptSig,
  pub sequence: u64,
  pub txinwitness: Vec<String>
}

impl Vin {
  pub fn new(json_obj: &json::JsonValue) -> Vin {
    return Vin {
      coinbase: if (json_obj.has_key("coinbase")) {
        String::from(json_obj["coinbase"].as_str().unwrap())
      } else {
        String::from("")
      },
      txid: if (json_obj.has_key("txid")) {
        String::from(json_obj["txid"].as_str().unwrap())
      } else {
        String::from("")
      },
      vout: if (json_obj.has_key("vout")) {
        json_obj["vout"].as_i32().unwrap()
      } else {
        -1
      },
      scriptSig: if (json_obj.has_key("scriptSig")) {
        ScriptSig::new(
          json_obj["scriptSig"]["asm"].as_str().unwrap(),
          json_obj["scriptSig"]["hex"].as_str().unwrap()
        )
      } else {
        ScriptSig::new("", "")
      },
      sequence: if (json_obj.has_key("sequence")) {
        json_obj["sequence"].as_u64().unwrap()
      } else {
        0
      },
      txinwitness: if (json_obj.has_key("txinwitness")) {
        json_obj["txinwitness"]
          .members()
          .map(|x| String::from(x.as_str().unwrap()))
          .collect()
      } else {
        vec![String::from("")]
      }
    };
  }
}

#[derive(Debug, Clone)]
pub struct RawTransaction {
  pub txid: String,
  pub hash: String,
  pub version: u32,
  pub size: u32,
  pub vsize: u32,
  pub weight: u32,
  pub locktime: u32,
  pub vin: Vec<Vin>,
  pub vout: Vec<Vout>,
  pub hex: String,
  pub blockhash: String,
  pub confirmations: u32,
  pub time: u32,
  pub blocktime: u32
}

impl RawTransaction {
  pub fn new(json_str: &str) -> RawTransaction {
    let json_val = json::parse(json_str).unwrap();

    return RawTransaction {
      txid: String::from(json_val["txid"].as_str().unwrap()),
      hash: String::from(json_val["hash"].as_str().unwrap()),
      version: json_val["version"].as_u32().unwrap(),
      size: json_val["size"].as_u32().unwrap(),
      vsize: json_val["vsize"].as_u32().unwrap(),
      weight: json_val["weight"].as_u32().unwrap(),
      locktime: json_val["locktime"].as_u32().unwrap(),
      vin: json_val["vin"].members().map(|x| Vin::new(x)).collect(),
      vout: json_val["vout"].members().map(|x| Vout::new(x)).collect(),
      hex: String::from(json_val["hex"].as_str().unwrap()),
      blockhash: String::from(json_val["blockhash"].as_str().unwrap()),
      confirmations: json_val["confirmations"].as_u32().unwrap(),
      time: json_val["time"].as_u32().unwrap(),
      blocktime: json_val["blocktime"].as_u32().unwrap()
    };
  }
}
