use crate::api::BlockPulsarAPI;
use crate::btc::address::ValidateAddress;
use crate::btc::block::Block;
use crate::btc::block_header::BlockHeader;
use crate::btc::block_stats::BlockStats;
use crate::btc::blockchain_info::BlockchainInfo;
use crate::btc::chain_tips::ChainTip;
use crate::btc::chain_tx_stats::ChainTxStats;
use crate::btc::mempool_entry::MempoolEntry;
use crate::btc::mempool_info::MempoolInfo;
use crate::btc::mining_info::MiningInfo;
use crate::btc::raw_transaction::RawTransaction;

pub struct BlockPulsarBtcClient {
  api: BlockPulsarAPI
}

impl BlockPulsarBtcClient {
  pub fn new(api: BlockPulsarAPI) -> BlockPulsarBtcClient {
    return BlockPulsarBtcClient { api };
  }

  pub async fn get_best_block_hash(&self) -> String {
    let res = self.api.request("btc/blockchain/getbestblockhash", &[]).await;
    let json_val = json::parse(&res.text().await.unwrap()).unwrap();
    let data = String::from(json_val["data"].as_str().unwrap());
    return data.replace("\"", "");
  }

  pub async fn get_block(&self, block_hash: &str) -> Block {
    let params: [(&str, &str); 1] = [("blockhash", block_hash)];
    let res = self.api.request("btc/blockchain/getblock", &params).await;
    return Block::new(&res.text().await.unwrap());
  }

  pub async fn get_blockchain_info(&self) -> BlockchainInfo {
    let res = self.api.request("btc/blockchain/getblockchaininfo", &[]).await;
    return BlockchainInfo::new(&res.text().await.unwrap());
  }

  pub async fn get_block_count(&self) -> u32 {
    let res = self.api.request("btc/blockchain/getblockcount", &[]).await;
    let json_val = json::parse(&res.text().await.unwrap()).unwrap();
    let data = json_val["data"].as_u32().unwrap();
    return data;
  }

  pub async fn get_block_hash(&self, height: u32) -> String {
    let params: [(&str, &str); 1] = [("height", &height.to_string())];
    let res = self.api.request("btc/blockchain/getblockhash", &params).await;
    let json_val = json::parse(&res.text().await.unwrap()).unwrap();
    let data = String::from(json_val["data"].as_str().unwrap());
    return data.replace("\"", "");
  }

  pub async fn get_block_header(&self, block_hash: &str) -> BlockHeader {
    let params: [(&str, &str); 1] = [("blockhash", block_hash)];
    let res = self.api.request("btc/blockchain/getblockheader", &params).await;
    return BlockHeader::new(&res.text().await.unwrap());
  }

  pub async fn get_block_stats(&self, block_hash: &str) -> BlockStats {
    let params: [(&str, &str); 1] = [("hash_or_height", block_hash)];
    let res = self.api.request("btc/blockchain/getblockstats", &params).await;
    return BlockStats::new(&res.text().await.unwrap());
  }

  pub async fn get_chain_tips(&self) -> Vec<ChainTip> {
    let res = self.api.request("btc/blockchain/getchaintips", &[]).await;
    let arr = json::parse(&res.text().await.unwrap()).unwrap();
    let ret_val: Vec<ChainTip> = arr
      .members()
      .map(|x| {
        ChainTip::new(
          x["height"].as_u32().unwrap(),
          x["hash"].as_str().unwrap(),
          x["branchlen"].as_u32().unwrap(),
          x["status"].as_str().unwrap()
        )
      })
      .collect();
    return ret_val;
  }

  pub async fn get_chain_tx_stats(&self, block_hash: &str, nblocks: u32) -> ChainTxStats {
    let params: [(&str, &str); 2] = [("blockhash", block_hash), ("nblocks", &nblocks.to_string())];
    let res = self
      .api
      .request("btc/transaction/getchaintxstats", &params)
      .await;
    return ChainTxStats::new(&res.text().await.unwrap());
  }

  pub async fn get_difficulty(&self) -> f64 {
    let res = self.api.request("btc/mining/getdifficulty", &[]).await;
    let json_val = json::parse(&res.text().await.unwrap()).unwrap();
    let data = json_val["data"].as_f64().unwrap();
    return data;
  }

  pub async fn get_mining_info(&self) -> MiningInfo {
    let res = self.api.request("btc/mining/getmininginfo", &[]).await;
    return MiningInfo::new(&res.text().await.unwrap());
  }

  pub async fn get_mempool_info(&self) -> MempoolInfo {
    let res = self.api.request("btc/transaction/getmempoolinfo", &[]).await;
    return MempoolInfo::new(&res.text().await.unwrap());
  }

  pub async fn get_raw_mempool(&self) -> Vec<String> {
    let res = self.api.request("btc/transaction/getrawmempool", &[]).await;
    let arr = json::parse(&res.text().await.unwrap()).unwrap();
    let ret_val = arr
      .members()
      .map(|x| String::from(x.as_str().unwrap()))
      .collect();
    return ret_val;
  }

  pub async fn get_mempool_ancestors(&self, txid: &str) -> Vec<String> {
    let params: [(&str, &str); 1] = [("txid", txid)];
    let res = self
      .api
      .request("btc/transaction/getmempoolancestors", &params)
      .await;
    let arr = json::parse(&res.text().await.unwrap()).unwrap();
    let ret_val = arr
      .members()
      .map(|x| String::from(x.as_str().unwrap()))
      .collect();
    return ret_val;
  }

  pub async fn get_mempool_descendants(&self, txid: &str) -> Vec<String> {
    let params: [(&str, &str); 1] = [("txid", txid)];
    let res = self
      .api
      .request("btc/transaction/getmempooldescendants", &params)
      .await;
    let arr = json::parse(&res.text().await.unwrap()).unwrap();
    let ret_val = arr
      .members()
      .map(|x| String::from(x.as_str().unwrap()))
      .collect();
    return ret_val;
  }

  pub async fn get_mempool_entry(&self, txid: &str) -> MempoolEntry {
    let params: [(&str, &str); 1] = [("txid", txid)];
    let res = self
      .api
      .request("btc/transaction/getmempoolentry", &params)
      .await;
    return MempoolEntry::new(&res.text().await.unwrap());
  }

  pub async fn get_network_hash_ps(&self, nblocks: u32, height: i32) -> f64 {
    let params: [(&str, &str); 2] = [
      ("nblocks", &nblocks.to_string()),
      ("height", &height.to_string())
    ];
    let res = self.api.request("btc/mining/getnetworkhashps", &params).await;
    let json_val = json::parse(&res.text().await.unwrap()).unwrap();
    let data = json_val["data"].as_f64().unwrap();
    return data
  }

  pub async fn get_raw_transaction(&self, txid: &str, block_hash: &str) -> RawTransaction {
    let params: [(&str, &str); 3] = [
      ("txid", txid),
      ("blockhash", block_hash),
      ("verbose", "true")
    ];
    let res = self
      .api
      .request("btc/transaction/getrawtransaction", &params)
      .await;
    return RawTransaction::new(&res.text().await.unwrap());
  }

  pub async fn validate_address(&self, address: &str) -> ValidateAddress {
    let params: [(&str, &str); 1] = [("address", address)];
    let res = self.api.request("btc/address/validateaddress", &params).await;
    return ValidateAddress::new(&res.text().await.unwrap());
  }
}
