use reqwest::header::{HeaderMap, HeaderName, HeaderValue};

pub struct BlockPulsarAPI {
  client: reqwest::Client,
  api_host: String,
  api_key: String,
  api_secret: String
}

static DEFAULT_API_HOST: &str = "https://api.blockpulsar.com/";

impl BlockPulsarAPI {
  pub fn new(api_key: &str, api_secret: &str) -> BlockPulsarAPI {
    return BlockPulsarAPI {
      client: reqwest::Client::new(),
      api_host: String::from(DEFAULT_API_HOST),
      api_key: String::from(api_key),
      api_secret: String::from(api_secret)
    };
  }

  fn construct_headers(&self) -> HeaderMap {
    let mut headers = HeaderMap::new();
    headers.insert(
        HeaderName::from_static("x-api-key"),
        HeaderValue::from_str(&self.api_key).unwrap(),
    );
    headers.insert(
        HeaderName::from_static("x-api-secret"),
        HeaderValue::from_str(&self.api_secret).unwrap(),
    );
    headers.insert(
      HeaderName::from_static("user-agent"),
      HeaderValue::from_str("blockpulsar_client_rust 1.0.0").unwrap()
    );

    return headers;
  }

  pub(crate) async fn request(&self, route: &str, params: &[(&str, &str)]) -> reqwest::Response {
    let api_url = format!("{}{}", self.api_host, route);
    let res = self
      .client
      .get(api_url)
      .headers(self.construct_headers())
      .query(&params)
      .send()
      .await;
    return res.unwrap();
  }
}
