# Blockpulsar Rust Client
Official Rust client for Blockpulsar API

- Website: https://blockpulsar.com
- API documentation: https://blockpulsar-docs.web.app/
- API version: 1.0.0
- Package version: 1.0.0


## Install
**1.** Edit the generated `Cargo.toml` file to add `blockpulsar_client_rust` as a dependency library:
```
[dependencies]
blockpulsar_client_rust="1.0.0"
```
**2.** Access the library in your code:
```rust
use blockpulsar_client_rust as bp;
```
**3.** Build the project with `cargo build --release`

Cargo will take care of managing the versions, building the dependencies when needed, and passing the correct arguments to the compiler to link together all of the dependencies.


## Example
Set the `api_key` and `secret_key` keys obtaned from the website.
```rust
use blockpulsar_client_rust as bp;

#[tokio::main]
async fn main() -> Result<(), Box<dyn std::error::Error>> {
  let bp_client = bp::client::BlockpulsarClient::new("api_key", "secret_key");

  let bb_hash = bp_client.btc.get_best_block_hash().await;
  let block = bp_client.btc.get_block(&bb_hash).await;

  println!("{:?}", block);

  return Ok(());
}
```


## Methods
```rust
// Returns the hash of the best (tip) block in the most-work fully-validated chain.
get_best_block_hash() -> String
```
```rust
// Get block.
//
// Args:
// -block_hash: The block hash
get_block(block_hash: &str) -> Block
```
```rust
// Returns an object containing various state info regarding blockchain processing.
get_blockchain_info() -> BlockchainInfo 
```
```rust
// Returns the height of the most-work fully-validated chain. The genesis block has height 0.
get_block_count() -> u32
```
```rust
// Returns hash of block in best-block-chain at height provided.
//
// Args:
// -height: The height index. The genesis block has height 0
get_block_hash(height: u32) -> String
```
```rust
// Get block header
//
// Args:
// -block_hash: The block hash
get_block_header(block_hash: &str) -> BlockHeader
```
```rust
// Compute per block statistics for a given window. All amounts are in satoshis.
//
// Args:
// -block_hash: The block hash
get_block_stats(block_hash: &str) -> BlockStats
```
```rust
// Return information about all known tips in the block tree, including the main chain
// as well as orphaned branches.
get_chain_tips() -> Vec<ChainTip>
```
```rust
// Compute statistics about the total number and rate of transactions in the chain.
//
// Args:
// -blockhash: (optional) The hash of the block that ends the window. Default is chain tip
// -nblocks: (optional) Size of the window in number of blocks.
get_chain_tx_stats(block_hash: &str, nblocks: u32) -> ChainTxStats
```
```rust
// Returns the proof-of-work difficulty as a multiple of the minimum difficulty.
get_difficulty() -> f64
```
```rust
// Get memory pool ancestors.
//
// Args:
// -txid: The transaction id (must be in mempool)
get_mempool_ancestors(txid: &str) -> Vec<String>
```
```rust
// Get memory pool descendants.
//
// Args:
// -txid: The transaction id (must be in mempool)
get_mempool_descendants(txid: &str) -> Vec<String>
```
```rust
// Returns mempool data for given transaction.
//
// Args:
// -txid: The transaction id (must be in mempool)
get_mempool_entry(txid: &str) -> MempoolEntry
```
```rust
// Returns details on the active state of the transaction memory pool.
get_mempool_info() -> MempoolInfo
```
```rust
// Get raw memory pool data. A vector of TX IDs.
get_raw_mempool() -> Vec<String>
```
```rust
// Returns a json object containing mining-related information.
get_mining_info() -> MiningInfo
```
```rust
// Returns the estimated network hashes per second based on the last n blocks.
//
// Args:
// -nblocks: (optional) The number of blocks, or -1 for blocks since last difficulty change.
// -height: (optional) To estimate at the time of the given height.
get_network_hash_ps(nblocks: u32, height: i32) -> u128
```
```rust
// Get raw transaction. When called with a blockhash argument, getrawtransaction will
// return the transaction if the specified block is available and the transaction is found
// in that block.
//
// Args:
// -txid: The transaction ID
// -block_hash: The block in which to look for the transaction.
get_raw_transaction(txid: &str, block_hash: &str) -> RawTransaction
```
```rust
// Return information about the given bitcoin address.
//
// Args:
// -address: BTC address to be checked
validate_address(address: &str) -> ValidateAddress
```


## Contributions
Contributions are welcome and can be made by submitting GitHub pull requests to this repository.  
The `blockpulsar_client_rs` source code follows the [Rust Style Guide](https://github.com/rust-dev-tools/fmt-rfcs/blob/master/guide/guide.md)
and some modifed rules specified in [`.rustfmt.toml`](https://github.com/blockpulsar/blockpulsar_client_rust/blob/master/.rustfmt.toml) file.

Before making a commit or creating a PR please make sure to run the following command on the root directory
```bash
cargo +nightly fmt
```


## License
This source code is available to everyone under the standard [Apache-2.0 License](https://github.com/blockpulsar/blockpulsar_client_rust/blob/master/LICENSE).
