use std::collections::HashMap;
use std::mem;
use structopt::StructOpt;
// blend_info
use blend_info::{calc_mem_tlen, read_dna, use_dna, DnaStrC};

/// Print some information about a Blender scene file.
#[derive(StructOpt)]
struct Cli {
    /// The path to the file to read
    #[structopt(parse(from_os_str))]
    path: std::path::PathBuf,
    /// Print information about Blender's DNA
    #[structopt(long = "dna")]
    dna: bool,
    /// Print information about a particular struct
    #[structopt(short = "n", long = "struct_name")]
    struct_name: Option<String>,
}

fn main() -> std::io::Result<()> {
    let args = Cli::from_args();
    let print_dna: bool = args.dna;
    let struct_name_opt: Option<String> = args.struct_name;
    // read DNA
    let mut dna_types_hm: HashMap<String, u16> = HashMap::new();
    let mut dna_structs_hm: HashMap<String, DnaStrC> = HashMap::new();
    let mut dna_2_type_id: Vec<u16> = Vec::new();
    let mut types: Vec<String> = Vec::new();
    let mut num_bytes_read: usize = 0;
    read_dna(
        print_dna,
        &args.path,
        &mut dna_types_hm,
        &mut dna_structs_hm,
        &mut dna_2_type_id,
        &mut types,
        &mut num_bytes_read,
    )?;
    // use DNA for something, e.g. find out about cameras
    if let Some(struct_name) = struct_name_opt {
        if struct_name.contains(".") {
            let mut names: Vec<String> = struct_name.split(".").map(|s| s.to_string()).collect();
            let mut names2 = names.split_off(1);
            let mut bytes_read: Vec<u8> = Vec::with_capacity(num_bytes_read);
            let mut structs_read: Vec<String> = Vec::with_capacity(names.len());
            let mut data_read: Vec<u32> = Vec::with_capacity(names.len());
            use_dna(
                print_dna,
                &args.path,
                &dna_types_hm,
                &dna_structs_hm,
                &names,
                &dna_2_type_id,
                &types,
                &mut bytes_read,
                &mut structs_read,
                &mut data_read,
            )?;
            let names3 = names2.split_off(1);
            let mut byte_index: usize = 0;
            for _struct_read in structs_read {
                if let Some(struct_found) = dna_structs_hm.get(&names[0]) {
                    for member in &struct_found.members {
                        if member.mem_type == names2[0] {
                            if let Some(struct_found2) = dna_structs_hm.get(&names2[0]) {
                                for member2 in &struct_found2.members {
                                    if let Some(type_found2) = dna_types_hm.get(&member2.mem_type) {
                                        let mem_tlen2: u16 = calc_mem_tlen(member2, *type_found2);
                                        if member2.mem_name.contains(&names3[0]) {
                                            let mut id = String::with_capacity(mem_tlen2 as usize);
                                            for i in 0..mem_tlen2 as usize {
                                                if bytes_read[byte_index + i] == 0 {
                                                    break;
                                                }
                                                if (bytes_read[byte_index + i] as char)
                                                    .is_ascii_alphanumeric()
                                                {
                                                    id.push(bytes_read[byte_index + i] as char);
                                                }
                                            }
                                            println!("{} = {:?}", struct_name, id);
                                            byte_index += mem_tlen2 as usize;
                                        } else {
                                            byte_index += mem_tlen2 as usize;
                                        }
                                    }
                                }
                            }
                        } else {
                            if member.mem_name.ends_with("[4][4]") {
                                if member.mem_name.starts_with(&names2[0]) {
                                    let mut mat_values: [f32; 16] = [0.0_f32; 16];
                                    let mut values_read: usize = 0;
                                    for i in 0..4 {
                                        for j in 0..4 {
                                            let mut float_buf: [u8; 4] = [0_u8; 4];
                                            for b in 0..4 as usize {
                                                float_buf[b] =
                                                    bytes_read[byte_index + values_read * 4 + b];
                                            }
                                            let some_float: f32 =
                                                unsafe { mem::transmute(float_buf) };
                                            mat_values[i * 4 + j] = some_float;
					    values_read += 1;
                                        }
                                    }
                                    println!("{} = {:#?}", struct_name, mat_values);
                                }
                            } else if member.mem_name == names2[0] {
                                // some basic type?
                                match member.mem_type.as_str() {
                                    "float" => {
                                        let mut float_buf: [u8; 4] = [0_u8; 4];
                                        for i in 0..4 as usize {
                                            float_buf[i] = bytes_read[byte_index + i];
                                        }
                                        let some_float: f32 = unsafe { mem::transmute(float_buf) };
                                        println!("{} = {}_{}", struct_name, some_float, "f32");
                                    }
                                    _ => {
                                        println!("TODO: {} = {}", member.mem_name, member.mem_type);
                                    }
                                }
                            }
                            // find mem_type in dna_types.names
                            if let Some(type_found) = dna_types_hm.get(&member.mem_type) {
                                let mem_tlen: u16 = calc_mem_tlen(member, *type_found);
                                byte_index += mem_tlen as usize;
                            }
                        }
                    }
                }
            }
        } else {
            let mut type_tlen: u16 = 0;
            // get expected tlen from dna_types
            if let Some(tlen) = dna_types_hm.get(&struct_name) {
                println!("{} {}", struct_name, tlen);
                type_tlen = *tlen;
            }
            // use dna_struct
            let mut counter: u16 = 0;
            if let Some(struct_found) = dna_structs_hm.get(&struct_name) {
                let sdna_nr = struct_found.sdna_nr;
                println!("struct {} {{ // SDNAnr = {}", struct_name, sdna_nr);
                for index in 0..struct_found.members.len() {
                    let member = &struct_found.members[index];
                    // find mem_type in dna_types.names
                    if let Some(type_found) = dna_types_hm.get(&member.mem_type) {
                        let mem_tlen: u16 = calc_mem_tlen(member, *type_found);
                        println!("  {} {}; // {}", member.mem_type, member.mem_name, mem_tlen);
                        counter += mem_tlen;
                    }
                }
                println!("}}; // {}", counter);
            }
            assert!(counter == type_tlen);
        }
    } else if !print_dna {
        // example usage
        let names: Vec<String> = vec![
            "Object".to_string(),
            "Camera".to_string(),
            "Lamp".to_string(),
            "Mesh".to_string(),
            "MPoly".to_string(),
            "MVert".to_string(),
            "MLoop".to_string(),
            "MLoopUV".to_string(),
            "MLoopCol".to_string(),
        ];
        let mut bytes_read: Vec<u8> = Vec::with_capacity(num_bytes_read);
        let mut structs_read: Vec<String> = Vec::with_capacity(names.len());
        let mut data_read: Vec<u32> = Vec::with_capacity(names.len());
        use_dna(
            print_dna,
            &args.path,
            &dna_types_hm,
            &dna_structs_hm,
            &names,
            &dna_2_type_id,
            &types,
            &mut bytes_read,
            &mut structs_read,
            &mut data_read,
        )?;
        println!(
            "TODO: Do something with the {} bytes returned by use_dna(...).",
            bytes_read.len()
        );
        println!("{:?}", structs_read);
        let sum: u32 = data_read.iter().sum();
        println!("{:?} = {}", data_read, sum);
        println!("=== EXTRACT INFO ===");
        let mut data_read_index: usize = 0;
        let mut index: usize = 0;
        for struct_name in structs_read {
            println!("{} ({})", struct_name, data_read_index);
            if let Some(tlen) = dna_types_hm.get(&struct_name) {
                if data_read[index] == *tlen as u32 {
                    data_read_index += data_read[index] as usize;
                } else {
                    let num_structs: u32 = data_read[index] / (*tlen as u32);
                    println!("{} * {}", num_structs, struct_name);
                    data_read_index += data_read[index] as usize;
                }
            } else {
                data_read_index += data_read[index] as usize;
            }
            index += 1;
        }
        println!("({})", data_read_index);
        println!("=== EXTRACT INFO ===");
    }
    Ok(())
}
