use anyhow::Result;

use self::{elearning::ELearning, stinfo::StInfo};

mod elearning;
mod stinfo;

pub struct MyBK {
    client: reqwest::Client,
}

trait HtmlExt {
    fn get_named_input(&self, name: &str) -> Option<&str>;
}

impl HtmlExt for scraper::Html {
    fn get_named_input(&self, name: &str) -> Option<&str> {
        let selector = scraper::Selector::parse(&format!(r#"[name="{name}"]"#)).ok()?;
        let mut element_ref =
            self.select(&selector)
                .map(|e| e.value())
                .filter_map(|e| match e.name() {
                    "input" => e.attr("value"),
                    "meta" => e.attr("content"),
                    _ => panic!("unknow named input"),
                });
        element_ref.next()
    }
}

impl MyBK {
    pub async fn with_credential(username: &str, password: &str) -> Self {
        let client = reqwest::ClientBuilder::new()
            .cookie_store(true)
            .build()
            .expect("cannot build reqwest client");

        const LOGIN_URL: &str = "https://sso.hcmut.edu.vn/cas/login";
        let res = client.get(LOGIN_URL).send().await.unwrap();
        let html = scraper::Html::parse_document(&res.text().await.unwrap());
        let lt = html.get_named_input("lt").expect("lt not found");
        let exec_id = html
            .get_named_input("execution")
            .expect("execution not found");
        let event_id = html
            .get_named_input("_eventId")
            .expect("event id not found");

        client
            .post(LOGIN_URL)
            .form(&[
                ("lt", lt),
                ("username", username),
                ("password", password),
                ("execution", exec_id),
                ("_eventId", event_id),
                ("submit", "Login"),
            ])
            .send()
            .await
            .expect("login unsuccessfully");

        Self { client }
    }

    pub async fn get_stinfo(&self) -> Result<StInfo<'_>> {
        StInfo::from_mybk(self).await
    }

    pub async fn get_elearning(&self) -> Result<ELearning<'_>> {
        ELearning::from_mybk(self).await
    }

    pub async fn logout(self) {
        const LOGOUT_URL: &str = "https://sso.hcmut.edu.vn/cas/logout";
        self.client
            .get(LOGOUT_URL)
            .send()
            .await
            .expect("log out failed");
    }
}
