use std::collections::HashMap;

use diff::Diff;
use serde::{ser::SerializeSeq, Deserialize, Serialize};

#[derive(Serialize, Deserialize, Debug, Diff, PartialEq, Eq)]
pub struct Grade {
    #[serde(rename = "ma_mh")]
    pub id: String,
    #[serde(rename = "diem_thanhphan")]
    pub mid: Option<String>,
    #[serde(rename = "diem_thi")]
    pub fin: Option<String>,
    #[serde(rename = "diem_tong_ket")]
    pub tot: Option<String>,
}

/// Due to inconsistencies in response data format,
/// we have to account for array data and indexed object data.
/// Hence the manually implemented deserializer.
#[derive(Debug, Diff, PartialEq, Eq)]
pub struct GradeSheet(HashMap<String, Grade>);
struct GradeSheetVisitor;

impl<'de> serde::de::Visitor<'de> for GradeSheetVisitor {
    type Value = GradeSheet;

    fn expecting(&self, formatter: &mut std::fmt::Formatter) -> std::fmt::Result {
        formatter.write_str("a sequence or an indexed hashmap")
    }

    fn visit_map<A>(self, mut map: A) -> Result<Self::Value, A::Error>
    where
        A: serde::de::MapAccess<'de>,
    {
        let mut result: Vec<Grade> = Vec::new();
        while let Some((_, v)) = map.next_entry::<usize, Grade>()? {
            result.push(v);
        }
        Ok(result.into_iter().collect())
    }

    fn visit_seq<A>(self, mut seq: A) -> Result<Self::Value, A::Error>
    where
        A: serde::de::SeqAccess<'de>,
    {
        let mut result: Vec<Grade> = Vec::new();
        while let Some(v) = seq.next_element::<Grade>()? {
            result.push(v);
        }
        Ok(result.into_iter().collect())
    }
}

impl<'de> Deserialize<'de> for GradeSheet {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        deserializer.deserialize_any(GradeSheetVisitor)
    }
}

impl Serialize for GradeSheet {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        let mut seq = serializer.serialize_seq(Some(self.0.len()))?;
        for v in self.0.values() {
            seq.serialize_element(v)?;
        }
        seq.end()
    }
}

impl std::iter::IntoIterator for GradeSheet {
    type Item = Grade;

    type IntoIter = std::collections::hash_map::IntoValues<String, Grade>;

    fn into_iter(self) -> Self::IntoIter {
        self.0.into_values()
    }
}

impl std::iter::FromIterator<Grade> for GradeSheet {
    fn from_iter<T: IntoIterator<Item = Grade>>(iter: T) -> Self {
        let map = iter
            .into_iter()
            .map(|grade| (grade.id.clone(), grade))
            .collect::<HashMap<String, Grade>>();
        Self(map)
    }
}

impl GradeSheet {
    pub fn save(&self) {
        let file = std::fs::OpenOptions::new()
            .write(true)
            .truncate(true)
            .create(true)
            .open("data/gradesheet.json")
            .expect("error opening file");

        serde_json::to_writer_pretty(file, &self).expect("save failed");
    }

    pub fn load() -> Self {
        let file = std::fs::OpenOptions::new()
            .read(true)
            .open("data/gradesheet.json")
            .expect("error opening file");

        serde_json::from_reader(file).expect("load failed")
    }

    pub fn is_empty(&self) -> bool {
        self.0.is_empty()
    }
}
