use cipher::generic_array::GenericArray;
use std::io::{Read, Write};
use std::mem::size_of;
use std::path::PathBuf;
use crate::bottle::{BottleReader, BottleStream, BottleWriter};
use crate::bottle_cap::BottleType;
use crate::bottle_error::{BottleError, BottleResult};
use crate::file_atlas::FileAtlas;
use crate::file_list::{Block, BlockHash, FileList};
use crate::hashing::HashType;
use crate::header::Header;


const DEFAULT_BLOCK_SIZE_BITS: usize = 20;

// fields for FileList
const FIELD_TOTAL_FILE_COUNT_INT: u8 = 0;
const FIELD_TOTAL_BLOCK_COUNT_INT: u8 = 1;
const FIELD_DIGEST_TYPE_INT: u8 = 2;

// fields for File:
const FIELD_SIZE_INT: u8 = 0;
const FIELD_POSIX_MODE_INT: u8 = 1;
const FIELD_CREATED_NSEC_INT: u8 = 2;
const FIELD_MODIFIED_NSEC_INT: u8 = 3;
const FIELD_BLOCK_COUNT_INT: u8 = 4;
const FIELD_PATH_STRING: u8 = 0;
// 1 = mime type (unused for now)
const FIELD_POSIX_USER_STRING: u8 = 2;
const FIELD_POSIX_GROUP_STRING: u8 = 3;
const FIELD_IS_FOLDER_FLAG: u8 = 0;
const FIELD_DIGEST_BYTES: u8 = 0;
const FIELD_SYMLINK_TARGET_STRING: u8 = 4;

// fields for the block table in a File:
const FIELD_BLOCK_SIZE_INT: u8 = 0;
const FIELD_BLOCK_DIGEST_BYTES: u8 = 0;


fn write_file_bottle<W: Write>(writer: W, atlas: &FileAtlas) -> BottleResult<()> {
    let mut header = Header::new();
    header.add_string(FIELD_PATH_STRING, atlas.normalized_path.to_str().unwrap())?;
    header.add_int(FIELD_POSIX_MODE_INT, atlas.perms as u64)?;
    header.add_int(FIELD_CREATED_NSEC_INT, atlas.ctime_ns as u64)?;
    header.add_int(FIELD_MODIFIED_NSEC_INT, atlas.mtime_ns as u64)?;
    header.add_string(FIELD_POSIX_USER_STRING, &atlas.user)?;
    header.add_string(FIELD_POSIX_GROUP_STRING, &atlas.group)?;
    if atlas.is_folder {
        header.add_flag(FIELD_IS_FOLDER_FLAG)?;
    } else {
        header.add_int(FIELD_SIZE_INT, atlas.size)?;
        header.add_bytes(FIELD_DIGEST_BYTES, &atlas.contents.hash)?;
        header.add_int(FIELD_BLOCK_COUNT_INT, atlas.contents.blocks.len() as u64)?;
    }
    if let Some(symlink_target) = &atlas.symlink_target {
        header.add_string(FIELD_SYMLINK_TARGET_STRING, symlink_target.to_str().unwrap())?;
    }

    let mut bottle_writer = BottleWriter::new(writer, BottleType::File, header, DEFAULT_BLOCK_SIZE_BITS)?;
    if atlas.contents.blocks.len() > 1 {
        for block in &atlas.contents.blocks {
            bottle_writer.write_data_stream()?;
            let mut header = Header::new();
            header.add_int(FIELD_BLOCK_SIZE_INT, block.size as u64)?;
            header.add_bytes(FIELD_BLOCK_DIGEST_BYTES, &block.hash)?;
            bottle_writer.write_all(header.pack())?;
            bottle_writer.close_stream()?;
        }
    }
    bottle_writer.close()?;
    Ok(())
}

fn read_file_bottle<R: Read>(bottle_reader: &mut BottleReader<R>) -> BottleResult<FileAtlas> {
    let header = &bottle_reader.bottle_cap.header;
    bottle_reader.expect_type(BottleType::File)?;

    let mut atlas = FileAtlas {
        path: PathBuf::from(header.get_string(FIELD_PATH_STRING).ok_or(BottleError::MissingHeader)?),
        ..Default::default()
    };
    atlas.normalized_path = atlas.path.clone();
    if header.get_flag(FIELD_IS_FOLDER_FLAG) { atlas.is_folder = true; }
    if !atlas.is_folder {
        atlas.contents.hash.copy_from_slice(header.get_bytes(FIELD_DIGEST_BYTES).ok_or(BottleError::MissingHeader)?);
    }
    if let Some(size) = header.get_int(FIELD_SIZE_INT) { atlas.size = size; }
    if let Some(perms) = header.get_int(FIELD_POSIX_MODE_INT) { atlas.perms = perms as u32; }
    if let Some(ctime) = header.get_int(FIELD_CREATED_NSEC_INT) { atlas.ctime_ns = ctime as i64; }
    if let Some(mtime) = header.get_int(FIELD_MODIFIED_NSEC_INT) { atlas.mtime_ns = mtime as i64; }
    if let Some(user) = header.get_string(FIELD_POSIX_USER_STRING) { atlas.user = user.to_string(); }
    if let Some(group) = header.get_string(FIELD_POSIX_GROUP_STRING) { atlas.group = group.to_string(); }
    if let Some(symlink_target) = header.get_string(FIELD_SYMLINK_TARGET_STRING) {
        atlas.symlink_target = Some(PathBuf::from(symlink_target));
    }

    let block_count = header.get_int(FIELD_BLOCK_COUNT_INT).unwrap_or(if atlas.is_folder { 0 } else { 1 });
    if block_count == 1 {
        // only one block: the whole file.
        let block = Block {
            size: atlas.size as usize,
            hash: atlas.contents.hash,
        };
        atlas.contents.blocks.push(block);
    } else {
        for _ in 0..block_count {
            bottle_reader.expect_next_stream(BottleStream::Data)?;
            let mut buffer = Vec::new();
            bottle_reader.data_stream()?.read_to_end(&mut buffer)?;
            bottle_reader.close_stream()?;
            let header = Header::from(&buffer[..]);

            let block = Block {
                size: header.get_int(FIELD_BLOCK_SIZE_INT).unwrap_or(0) as usize,
                hash: GenericArray::clone_from_slice(
                    header.get_bytes(FIELD_BLOCK_DIGEST_BYTES).unwrap_or(&[0u8; size_of::<BlockHash>()])
                ),
            };
            atlas.contents.blocks.push(block);
        }
    }

    bottle_reader.expect_end()?;
    Ok(atlas)
}

fn write_block_bottle<W: Write>(writer: W, block: &Block, data: &[u8]) -> BottleResult<()> {
    let mut header = Header::new();
    header.add_int(FIELD_BLOCK_SIZE_INT, block.size as u64)?;
    header.add_bytes(FIELD_BLOCK_DIGEST_BYTES, &block.hash)?;
    let mut bottle_writer = BottleWriter::new(writer, BottleType::FileBlock, header, DEFAULT_BLOCK_SIZE_BITS)?;
    bottle_writer.write_data_stream()?;
    bottle_writer.write_all(data)?;
    bottle_writer.close_stream()?;
    bottle_writer.close()?;
    Ok(())
}

fn read_block_bottle<R: Read>(bottle_reader: &mut BottleReader<R>) -> BottleResult<(Block, Vec<u8>)> {
    let header = &bottle_reader.bottle_cap.header;
    bottle_reader.expect_type(BottleType::FileBlock)?;

    let size = header.get_int(FIELD_BLOCK_SIZE_INT).ok_or(BottleError::MissingHeader)? as usize;
    let hash = GenericArray::clone_from_slice(
        header.get_bytes(FIELD_BLOCK_DIGEST_BYTES).ok_or(BottleError::MissingHeader)?
    );
    let block = Block { size, hash };

    bottle_reader.expect_next_stream(BottleStream::Data)?;
    let mut buffer = Vec::new();
    bottle_reader.data_stream()?.read_to_end(&mut buffer)?;
    bottle_reader.close_stream()?;
    bottle_reader.expect_end()?;
    Ok((block, buffer))
}


pub struct FileListBottleWriter<W: Write> {
    bottle_writer: BottleWriter<W>,
}

impl<W: Write> FileListBottleWriter<W> {
    pub fn new<Callback: FnMut (usize)>(
        writer: W,
        hash_type: HashType,
        file_list: &FileList,
        mut updater: Callback,
    ) -> BottleResult<FileListBottleWriter<W>> {
        let mut header = Header::new();
        header.add_int(FIELD_TOTAL_FILE_COUNT_INT, file_list.files.len() as u64)?;
        header.add_int(FIELD_TOTAL_BLOCK_COUNT_INT, file_list.blocks.len() as u64)?;
        header.add_int(FIELD_DIGEST_TYPE_INT, hash_type as u64)?;

        let mut bottle_writer = BottleWriter::new(writer, BottleType::FileList, header, DEFAULT_BLOCK_SIZE_BITS)?;

        // first, write each file atlas as a stream
        let mut count = 0;
        for atlas in &file_list.files {
            bottle_writer.write_bottle()?;
            write_file_bottle(&mut bottle_writer, &atlas.borrow())?;
            bottle_writer.close_stream()?;
            count += 1;
            updater(count);
        }

        Ok(FileListBottleWriter { bottle_writer })
    }

    pub fn add_block(&mut self, block: &Block, data: &[u8]) -> BottleResult<()> {
        self.bottle_writer.write_bottle()?;
        write_block_bottle(&mut self.bottle_writer, block, data)?;
        self.bottle_writer.close_stream()?;
        Ok(())
    }

    pub fn close(self) -> BottleResult<W> {
        self.bottle_writer.close()
    }
}


pub struct FileListBottleReader<R: Read> {
    bottle_reader: BottleReader<R>,
    pub hash_type: HashType,
    pub file_list: FileList,
    pub block_count: usize,
}

impl<R: Read> FileListBottleReader<R> {
    /// `updater` receives the number of files read so far, and the total we
    /// expect.
    pub fn new<Callback: FnMut (usize, usize, HashType, &FileList)>(
        mut bottle_reader: BottleReader<R>,
        mut updater: Callback,
    ) -> BottleResult<FileListBottleReader<R>> {
        let header = &bottle_reader.bottle_cap.header;
        let file_count = header.get_int(FIELD_TOTAL_FILE_COUNT_INT).ok_or(BottleError::MissingHeader)?;
        let block_count = header.get_int(FIELD_TOTAL_BLOCK_COUNT_INT).ok_or(BottleError::MissingHeader)? as usize;
        let hash_type: u64 = header.get_int(FIELD_DIGEST_TYPE_INT).unwrap_or(HashType::SHA256 as u64);
        let hash_type: HashType = (hash_type as u8).try_into().map_err(|_| BottleError::UnknownHashType)?;

        let mut file_list = FileList::default();
        for i in 0..file_count {
            bottle_reader.expect_next_stream(BottleStream::Bottle)?;
            let atlas = read_file_bottle(bottle_reader.bottle_reader()?)?;
            file_list.files.push(atlas.bobble());
            bottle_reader.close_stream()?;
            // build file map before the last update:
            if i == file_count - 1 {
                file_list.build_file_map();
            }
            updater((i + 1) as usize, file_count as usize, hash_type, &file_list);
        }

        Ok(FileListBottleReader { bottle_reader, hash_type, file_list, block_count })
    }

    pub fn next_block(&mut self) -> BottleResult<Option<(Block, Vec<u8>)>> {
        let stream_kind = self.bottle_reader.next_stream()?;
        if stream_kind == BottleStream::End { return Ok(None); }
        if stream_kind != BottleStream::Bottle {
            return Err(BottleError::WrongStreamType);
        }
        let (block, data) = read_block_bottle(self.bottle_reader.bottle_reader()?)?;
        self.bottle_reader.close_stream()?;
        Ok(Some((block, data)))
    }

    pub fn close(self) -> BottleResult<R> {
        self.bottle_reader.close()
    }
}


#[cfg(test)]
mod test {
    use std::path::PathBuf;
    use crate::bottle::BottleReader;
    use crate::file_bottle::FileListBottleWriter;
    use crate::file_atlas::FileAtlas;
    use crate::file_list::{Block, FileList};
    use crate::hashing::HashType;
    use super::{FileListBottleReader, read_file_bottle, write_file_bottle};

    fn sample_atlas() -> FileAtlas {
        let mut atlas = FileAtlas::default();
        atlas.path = PathBuf::from("test.txt");
        atlas.normalized_path = atlas.path.clone();
        atlas.size = 1456;
        atlas.perms = 0o644;
        atlas.mtime_ns = 1631823097070057800;
        atlas.ctime_ns = 1631823097070057800;
        atlas.user = "aku".to_string();
        atlas.group = "evil".to_string();
        atlas.contents.hash.copy_from_slice(
            &hex::decode("d347b2e0a27bb2a71ae56f313ee05c3dd499dc5d08aa62f901983dfc665bf0b8").unwrap()
        );

        let mut block1 = Block::default();
        block1.size = 1_234_988;
        block1.hash.copy_from_slice(
            &hex::decode("55e0c9e957f0d056e0acd332c35d8d67a279ed937f20f7d78d2db7c699c32ec1").unwrap()
        );
        let mut block2 = Block::default();
        block2.size = 910_812;
        block2.hash.copy_from_slice(
            &hex::decode("97fd21d292a8937c455e5ffcbc1da58297a93f349383cb442a2784d5f64c71f2").unwrap()
        );
        atlas.contents.blocks.push(block1);
        atlas.contents.blocks.push(block2);
        atlas
    }

    fn sample_atlas_folder() -> FileAtlas {
        let mut atlas = FileAtlas::default();
        atlas.path = PathBuf::from("warez");
        atlas.normalized_path = atlas.path.clone();
        atlas.perms = 0o755;
        atlas.mtime_ns = 1631823097070057800;
        atlas.ctime_ns = 1631823097070057800;
        atlas.user = "aku".to_string();
        atlas.group = "evil".to_string();
        atlas.is_folder = true;
        atlas
    }


    #[test]
    fn read_and_write_file_bottle() {
        let mut buffer = Vec::new();
        let atlas = sample_atlas();
        write_file_bottle(&mut buffer, &atlas).unwrap();
        // make sure it's small:
        assert_eq!(buffer.len(), 176);

        let atlas2 = read_file_bottle(&mut BottleReader::new(&buffer[..]).unwrap()).unwrap();
        assert_eq!(atlas, atlas2);
    }

    #[test]
    fn read_and_write_folder_bottle() {
        let mut buffer = Vec::new();
        let atlas = sample_atlas_folder();
        write_file_bottle(&mut buffer, &atlas).unwrap();
        // make sure it's small:
        assert_eq!(buffer.len(), 53);

        let atlas2 = read_file_bottle(&mut BottleReader::new(&buffer[..]).unwrap()).unwrap();
        assert_eq!(atlas, atlas2);
    }

    #[test]
    fn file_list_with_one_file() {
        let mut buffer = Vec::new();
        let atlas = sample_atlas();
        let block1 = atlas.contents.blocks[0].clone();
        let block2 = atlas.contents.blocks[1].clone();

        let mut file_list = FileList::default();
        file_list.files.push(atlas.bobble());
        file_list.blocks.insert(block1.hash, block1.clone());
        file_list.blocks.insert(block2.hash, block2.clone());

        let mut writer = FileListBottleWriter::new(&mut buffer, HashType::BLAKE3, &file_list, |_| {}).unwrap();
        writer.add_block(&block1, b"Some random data").unwrap();
        writer.add_block(&block2, b"FRIDAY AGAIN GARFIE BABY").unwrap();
        writer.close().unwrap();

        let bottle_reader = BottleReader::new(&buffer[..]).unwrap();
        let mut reader = FileListBottleReader::new(bottle_reader, |_, _, _, _| {}).unwrap();
        assert_eq!(reader.block_count, 2);
        assert_eq!(reader.file_list.files, file_list.files);
        assert_eq!(reader.next_block().unwrap(), Some((block1, b"Some random data".to_vec())));
        assert_eq!(reader.next_block().unwrap(), Some((block2, b"FRIDAY AGAIN GARFIE BABY".to_vec())));
        assert_eq!(reader.next_block().unwrap(), None);
        reader.close().unwrap();
    }
}
