use std::io::{Read, Write};

pub struct CountingWriter<W: Write, F: FnMut (u64, bool)> {
    writer: W,
    callback: F,
    pub count: u64,
}

impl<W: Write, F: FnMut (u64, bool)> CountingWriter<W, F> {
    pub fn new(writer: W, callback: F) -> CountingWriter<W, F> {
        CountingWriter { writer, callback, count: 0 }
    }
}

impl<W: Write, F: FnMut (u64, bool)> Write for CountingWriter<W, F> {
    fn write(&mut self, data: &[u8]) -> std::io::Result<usize> {
        self.count += data.len() as u64;
        (self.callback)(self.count, false);
        self.writer.write(data)
    }

    fn flush(&mut self) -> std::io::Result<()> {
        (self.callback)(self.count, true);
        self.writer.flush()
    }
}


pub struct CountingReader<R: Read, F: FnMut (u64, bool)> {
    reader: R,
    callback: F,
    pub count: u64,
}

impl<R: Read, F: FnMut (u64, bool)> CountingReader<R, F> {
    pub fn new(reader: R, callback: F) -> CountingReader<R, F> {
        CountingReader { reader, callback, count: 0 }
    }

    pub fn into_inner(self) -> R {
        self.reader
    }
}

impl<R: Read, F: FnMut (u64, bool)> Read for CountingReader<R, F> {
    fn read(&mut self, data: &mut [u8]) -> std::io::Result<usize> {
        let n = self.reader.read(data)?;
        self.count += n as u64;
        (self.callback)(self.count, false);
        Ok(n)
    }
}
