use clap::{App, AppSettings};
use rpassword::prompt_password_stderr;
use std::collections::HashSet;
use std::fs::File;
use std::io::Read;
use std::path::PathBuf;
use std::process;
use bitbottle::*;
use bitbottle::cli::*;

const ABOUT: &str =
    "Read or unpack a bitbottle file archive.";

const ARGS: &str = "
-s --secret [FILE]          'decrypt the archive using an ssh-ed2519 private key file'
-p --password               'decrypt the archive with a password'

-v --verbose                'describe what's happening as the bitbottle is read'
-q --quiet                  'don't display progress bars or explanatory information'

-i --info                   'list the contents without unpacking'
--check                     'validate that the block storage is correct and complete without unpacking'
--dump                      'display bottle structure without unpacking'

-d --dest [FOLDER]          'unpack archive into a new folder (default is the archive name)'
[FILE]                      'read archive from file instead of stdin'
";

static mut VERBOSE: bool = false;
static mut QUIET: bool = false;

macro_rules! verbose {
    ($($arg:tt)*) => ({
        if unsafe { VERBOSE } {
            eprintln!($($arg)*);
        }
    })
}

macro_rules! info {
    ($($arg:tt)*) => ({
        if !unsafe { QUIET } {
            eprintln!($($arg)*);
        }
    })
}

fn is_verbose() -> bool {
    unsafe { VERBOSE }
}

fn is_quiet() -> bool {
    unsafe { QUIET }
}

#[derive(Clone, Copy, Debug, Eq, PartialEq)]
enum Mode {
    Unpack,
    Info,
    CheckOnly,
    Dump,
}


fn drain<R: Read>(mut reader: R) -> std::io::Result<usize> {
    let mut byte_count = 0;
    let mut buffer = vec![0u8; 0x1_0000];
    loop {
        let n = reader.read(&mut buffer)?;
        if n == 0 { return Ok(byte_count); }
        byte_count += n;
    }
}

fn dump_file_archive<R: Read>(bottle_reader: &mut BottleReader<R>, indent_size: usize) -> BottleResult<()> {
    for line in indent(bottle_reader.bottle_cap.dump(), indent_size) { info!("{}", line); }
    loop {
        match bottle_reader.next_stream()? {
            BottleStream::Data => {
                let size = drain(bottle_reader.data_stream()?)?;
                let size_str = format!("{} bytes ({})", size, to_binary_si(size as f64));
                info!("{}", indent_line(format!("Data stream: {}", size_str), indent_size + 4));
                bottle_reader.close_stream()?;
            },
            BottleStream::Bottle => {
                let r = bottle_reader.bottle_reader()?;
                dump_file_archive(r, indent_size + 4)?;
                bottle_reader.close_stream()?;
            },
            BottleStream::End => break,
        }
    }
    Ok(())
}

fn extract_file_archive<R: Read>(
    bottle_reader: BottleReader<R>,
    dest_path: PathBuf,
    orig_size: Option<u64>,
) -> BottleResult<R> {
    if bottle_reader.bottle_cap.bottle_type != BottleType::FileList {
        eprintln!("ERROR: This is not a bitbottle archive -- it has no file list.");
        process::exit(1);
    }

    let progress = ProgressLine::new().bobble();
    progress.borrow_mut().show_ever = !is_quiet();
    progress.borrow_mut().show_bar = true;
    progress.borrow_mut().update(0f64, "Reading file list".to_string());

    let mut total_file_count = 0;
    let mut total_size = 0u64;
    let mut path_warning_count = 0;

    let options = ReadArchiveOptions::default();
    let reader = expand_archive(bottle_reader, options, &dest_path, |state, file_list| {
        let mut progress = progress.borrow_mut();
        match state {
            ReadArchiveState::TableOfContents { file_count, total, hash_type } => {
                if is_verbose() {
                    progress.clear();
                    let atlas = file_list.files.last().unwrap();
                    verbose!("{}", display_atlas_filename(&atlas.borrow()));
                    progress.force_update();
                }

                let percent = file_count as f64 / total as f64;
                progress.complete(percent);
                if file_count == total {
                    total_file_count = file_list.total_file_count();
                    total_size = file_list.total_size();

                    progress.clear();
                    verbose!("{}", display_file_header(hash_type, file_list, orig_size));
                    progress.force_update();
                }
            },
            ReadArchiveState::SymlinkIgnoredWarning { symlink } => {
                info!("");
                info!("WARNING: ignoring invalid symlink {:?} -> {:?}", symlink.source, symlink.target);
                info!("");
            },
            ReadArchiveState::BadPathWarning { bad_path, new_path } => {
                if path_warning_count == 0 {
                    info!("");
                    info!("WARNING: invalid path {:?} -- defanging to {:?}", bad_path, new_path);
                    info!("");
                }
                path_warning_count += 1;
            },
            ReadArchiveState::BlockData { bytes_read, bytes_written, .. } => {
                let percent = bytes_written as f64 / file_list.total_size() as f64;
                progress.update(percent, format!("Extracting: {} -> {}/{}",
                    to_binary_si(bytes_read as f64), to_binary_si(bytes_written as f64),
                    to_binary_si(file_list.total_size() as f64)));
            },
        }
        progress.display();
    })?;
    progress.borrow_mut().clear();

    if path_warning_count > 1 {
        info!("({} other similar path warnings)", path_warning_count - 1);
    }
    info!("Extracted {} file(s) ({} bytes) to {}",
        total_file_count, to_binary_si(total_size as f64), dest_path.to_str().unwrap());

    Ok(reader)
}

fn info_file_archive<R: Read>(
    bottle_reader: BottleReader<R>,
    orig_size: Option<u64>,
    show_info: bool,
    check: bool,
) -> BottleResult<R> {
    if bottle_reader.bottle_cap.bottle_type != BottleType::FileList {
        eprintln!("ERROR: This is not a bitbottle archive -- it has no file list.");
        process::exit(1);
    }

    let progress = ProgressLine::new().bobble();
    progress.borrow_mut().show_ever = !is_quiet();
    progress.borrow_mut().show_bar = true;
    progress.borrow_mut().update(0f64, "Reading file list".to_string());
    let mut path_warning_count = 0;

    let mut bottle_reader = read_archive(bottle_reader, &mut |state, _file_list| {
        let mut progress = progress.borrow_mut();
        match state {
            ReadArchiveState::TableOfContents { file_count, total, .. } => {
                let percent = file_count as f64 / total as f64;
                progress.complete(percent);
            },
            ReadArchiveState::SymlinkIgnoredWarning { symlink } => {
                info!("");
                info!("*** WARNING: ignoring invalid symlink {:?} -> {:?}", symlink.source, symlink.target);
                info!("");
            },
            ReadArchiveState::BadPathWarning { bad_path, new_path } => {
                if path_warning_count == 0 {
                    info!("");
                    info!("*** WARNING: invalid path {:?} -- defanging to {:?}", bad_path, new_path);
                    info!("");
                }
                path_warning_count += 1;
            },
            ReadArchiveState::BlockData { .. } => (),
        }
        progress.display();
    })?;
    progress.borrow_mut().clear();
    if path_warning_count > 1 {
        info!("({} other similar path warnings)", path_warning_count - 1);
    }

    if show_info {
        for line in display_file_list(bottle_reader.hash_type, &bottle_reader.file_list, orig_size) {
            eprintln!("{}", line);
        }
    } else {
        info!("{}", display_file_header(bottle_reader.hash_type, &bottle_reader.file_list, orig_size));
    }

    if check {
        // internal consistency: do the block sizes line up with the file sizes?
        for file in &bottle_reader.file_list.files {
            let file = file.borrow();
            let size = file.contents.blocks.iter().fold(0u64, |sum, block| sum + block.size as u64);
            if size != file.size {
                eprintln!("ERROR: File size {} doesn't match file atlas block size {}", file.size, size);
            }
        }

        progress.borrow_mut().update(0f64, "Checking block storage".to_string());
        let mut block_list: HashSet<Block> = bottle_reader.file_list.blocks.values().cloned().collect();
        let total_blocks = block_list.len();
        let mut digest = Hashing::new(bottle_reader.hash_type);

        while let Some((block, data)) = bottle_reader.next_block()? {
            digest.update(&data);
            let hash = digest.finalize_reset();
            if hash == block.hash {
                progress.borrow_mut().clear();
                verbose!("OK  {}", hex::encode(hash));
            } else {
                eprintln!("ERROR: Bad block: {} != {}", hex::encode(hash), hex::encode(block.hash));
                process::exit(1);
            }
            if !block_list.remove(&block) {
                eprintln!("ERROR: Extra block: {}", hex::encode(hash));
                process::exit(1);
            }

            let so_far = total_blocks - block_list.len();
            let percent = so_far as f64 / total_blocks as f64;
            progress.borrow_mut().update(percent, format!("Checking block {}/{}", so_far, total_blocks));
            progress.borrow_mut().display();
        }

        progress.borrow_mut().clear();
        if !block_list.is_empty() {
            eprintln!("ERROR: Undiscovered blocks in table of contents: {}", block_list.len());
            process::exit(1);
        }
        eprintln!("Passed validation: all blocks are present and correct.");
    } else {
        // look, there's no point in scanning the rest of the file if we don't care. just go to bed.
        process::exit(0)
    }

    bottle_reader.close()
}

fn read_compressed_archive<R: Read>(
    mut bottle_reader: BottleReader<R>,
    dest_path: PathBuf,
    orig_size: Option<u64>,
    mode: Mode,
    check: bool,
    indent_size: usize,
) -> BottleResult<R> {
    if bottle_reader.bottle_cap.bottle_type == BottleType::Compressed {
        if mode == Mode::Dump {
            for line in indent(bottle_reader.bottle_cap.dump(), indent_size) { info!("{}", line); }
        }
        let compressed_reader = CompressedBottleReader::new(bottle_reader)?;
        if mode != Mode::Dump {
            verbose!("Bitbottle compressed with {:?}", compressed_reader.algorithm);
        }

        let mut bottle_reader = BottleReader::new(compressed_reader)?;
        let reader = match mode {
            Mode::Unpack => extract_file_archive(bottle_reader, dest_path, orig_size),
            Mode::Info | Mode::CheckOnly => info_file_archive(bottle_reader, orig_size, mode == Mode::Info, check),
            Mode::Dump => {
                dump_file_archive(&mut bottle_reader, indent_size + 4)?;
                bottle_reader.close()
            },
        };
        reader?.close()
    } else {
        match mode {
            Mode::Unpack => extract_file_archive(bottle_reader, dest_path, orig_size),
            Mode::Info | Mode::CheckOnly => info_file_archive(bottle_reader, orig_size, mode == Mode::Info, check),
            Mode::Dump => {
                dump_file_archive(&mut bottle_reader, indent_size)?;
                bottle_reader.close()
            }
        }
    }
}

fn read_encrypted_archive<K: BottleSecretKey>(
    reader: Box<dyn Read>,
    dest_path: PathBuf,
    orig_size: Option<u64>,
    encryption_key: EncryptionKey,
    secret_key: Option<K>,
    mode: Mode,
    check: bool,
) -> BottleResult<Box<dyn Read>> {
    let mut bottle_reader = BottleReader::new(reader)?;

    if bottle_reader.bottle_cap.bottle_type == BottleType::Encrypted {
        if mode == Mode::Dump {
            for line in bottle_reader.bottle_cap.dump() { info!("{}", line); }
        }

        let info = EncryptedBottleReader::unpack_info(&mut bottle_reader)?;
        let key_count = info.public_encrypted_keys.len();
        if mode == Mode::Dump {
            // dump any nested headers
            for encrypted_key in info.public_encrypted_keys.iter() {
                info!("{}", indent_line("Data stream:".to_string(), 4));
                for line in indent(encrypted_key.header.dump(), 8) { info!("{}", line); }
            }
        } else {
            info!("Bitbottle encrypted with {:?}{}{}",
                info.algorithm,
                if info.argon.is_some() { ", password" } else { "" },
                if key_count > 0 {
                    format!(", {} public key{} ({:?})",
                        key_count, if key_count > 1 { "s" } else { "" }, info.public_key_algorithm)
                } else {
                    String::new()
                },
            );

            verbose!("    Block size: {}", to_binary_si((1 << info.block_size_bits) as f64));
            if let Some(argon) = info.argon {
                verbose!("    Password encryption: ARGON2ID (time={}, mem={}, par={})",
                    argon.time_cost, to_binary_si((1 << argon.memory_cost_bits) as f64), argon.parallelism);
            }
            for encrypted_key in &info.public_encrypted_keys {
                verbose!("    Encrypted for: {} ({})", pad_truncate(encrypted_key.public_key.name(), 24),
                    hex::encode(encrypted_key.public_key.as_bytes()));
            }
        }

        let reader = match secret_key {
            Some(sk) => EncryptedBottleReader::build_with_info(bottle_reader, info, encryption_key, &[ sk ]),
            None => EncryptedBottleReader::new_with_info(bottle_reader, info, encryption_key),
        }?;
        let bottle_reader = BottleReader::new(reader)?;
        let bottle_reader = read_compressed_archive(bottle_reader, dest_path, orig_size, mode, check, 4)?;
        bottle_reader.close()
    } else {
        read_compressed_archive(bottle_reader, dest_path, orig_size, mode, check, 0)
    }
}

// load the secret key, asking for a password if necessary
fn read_secret_key_file(filename: &str) -> BottleResult<Ed25519SecretKey> {
    match Ed25519SecretKey::from_ssh_file(filename, None) {
        Err(BottleError::SshPasswordRequired) => (),
        rv => { return rv; },
    };

    let password = prompt_password_stderr(&format!("Password for {}: ", filename)).unwrap_or_else(|e| {
        eprintln!();
        eprintln!("ERROR: {}", e);
        process::exit(1);
    });
    eprintln!();
    Ed25519SecretKey::from_ssh_file(filename, Some(&password))
}


pub fn main() {
    let args = App::new("unbottle")
        .setting(AppSettings::DeriveDisplayOrder)
        .setting(AppSettings::UnifiedHelpMessage)
        .version("0.1")
        .author("Robey Pointer <robey@lag.net>")
        .about(ABOUT)
        .args_from_usage(ARGS)
        .get_matches();

    if args.is_present("verbose") {
        unsafe { VERBOSE = true; }
    }
    if args.is_present("quiet") {
        unsafe { QUIET = true; }
    }

    let check = args.is_present("check");
    let mut mode = if check { Mode::CheckOnly } else { Mode::Unpack };
    if args.is_present("info") {
        mode = Mode::Info;
    }
    if args.is_present("dump") {
        mode = Mode::Dump;
    }

    let mut dest_path = args.value_of("dest").map(|s| s.to_string());

    // decrypt?
    let password = if args.is_present("password") {
        let password = prompt_password_stderr("Password: ").map_err(|e| {
            eprintln!();
            eprintln!("ERROR: {}", e);
            process::exit(1);
        }).ok();
        eprintln!();
        password
    } else {
        None
    };
    let sk_filename = args.value_of("secret");
    let encryption_key = password.map(EncryptionKey::Password).unwrap_or(EncryptionKey::Generate);

    // load the secret key?
    let sk = sk_filename.map(|filename| {
        read_secret_key_file(filename).map(|sk| {
            verbose!("Decrypting with key: {}", sk.name());
            sk
        })
    }).transpose().unwrap_or_else(|e| {
        eprintln!("ERROR: Unable to read secret key file: {}", e);
        process::exit(1);
    });

    let mut orig_size: Option<u64> = None;
    let reader: Box<dyn Read> = match args.value_of("FILE") {
        Some(filename) => {
            let path = PathBuf::from(filename);
            if dest_path.is_none() {
                dest_path = Some(path.file_stem().map(|s| s.to_str()).flatten().unwrap().to_string());
                if mode == Mode::Unpack {
                    verbose!("No destination path given; using: ./{}/", dest_path.as_ref().unwrap());
                }
            }
            orig_size = path.metadata().map(|m| m.len()).ok();
            Box::new(File::open(filename).unwrap_or_else(|e| {
                eprintln!("ERROR: Unable to read file {}: {}", filename, e);
                process::exit(1);
            }))
        },
        None => {
            Box::new(std::io::stdin())
        },
    };
    if dest_path.is_none() {
        eprintln!("ERROR: Destination path (-d) required");
        process::exit(1);
    }
    let dest_path = PathBuf::from(dest_path.unwrap());

    read_encrypted_archive(reader, dest_path, orig_size, encryption_key, sk, mode, check)
        .map(|_| ())
        .unwrap_or_else(|e| {
            eprintln!("ERROR: {}", e);
        });
}
