use clap::App;
use std::path::PathBuf;
use bitbottle::{FileScanner, HashType, ScanState};
use bitbottle::cli::{ProgressLine, to_binary_si};

const ARGS: &str = "
-v --verbose                'list each file's metadata at the end'
--blake2                    'use blake2 hashing instead of sha-256'
--blake3                    'use blake3 hashing instead of sha-256'
<PATH>...                   'path(s) to scan'
";


pub fn main() {
    let args = App::new("buzscan")
        .version("1.0")
        .author("Robey Pointer <robey@lag.net>")
        .about(
"Scans and hashes a list of files and folders (recursively) to determine
how much of the data is unique and how much is duplicated."
        )
        .args_from_usage(ARGS)
        .get_matches();

    let mut hash_type = HashType::SHA256;
    if args.is_present("blake2") {
        hash_type = HashType::BLAKE2;
    }
    if args.is_present("blake3") {
        hash_type = HashType::BLAKE2;
    }

    let progress = ProgressLine::new().bobble();
    let mut file_scanner = FileScanner::new(hash_type, 18, 20, 22, 10, vec![0u8; 65536], |state, file_list, size| {
        let mut progress = progress.borrow_mut();
        let file_count = file_list.plain_files().count();
        match state {
            ScanState::FileList(_atlas) => {
                progress.show_bar = false;
                progress.update(0f64, format!("Scanning files: {:>8} files, disk space: {:>5}",
                    file_count, to_binary_si(size as f64)));
            },
            ScanState::Chunking => {
                progress.show_bar = true;
                let percent = size as f64 / file_list.total_size() as f64;
                let unique_size = file_list.blocks.values().fold(0, |sum, block| sum + block.size);
                let total_size = file_list.total_size();
                progress.update(percent, format!("{:>8} files, {:>5}/{:>5}, {:>5} unique",
                    file_count, to_binary_si(size as f64), to_binary_si(total_size as f64),
                    to_binary_si(unique_size as f64)));
                if size == total_size {
                    progress.force_update();
                }
            }
        }
        progress.display();
    });

    file_scanner.scan_paths(&args.values_of("PATH").unwrap().map(PathBuf::from).collect::<Vec<_>>()).unwrap();
    let file_list = file_scanner.build_block_list(hash_type).unwrap();

    if args.is_present("v") {
        for atlas in &file_list.files {
            let atlas = atlas.borrow();
            println!("{:>5} {:>3o} {:>8} {:>8} {:>64} {}",
                if atlas.is_folder { "".to_string() } else { to_binary_si(atlas.size as f64) },
                atlas.perms & 0x1ff,
                atlas.user,
                atlas.group,
                if atlas.is_folder { "".to_string() } else { hex::encode(atlas.contents.hash) },
                atlas.path.as_path().display()
            );
        }
    }
}
