use clap::{App, AppSettings};
use rpassword::prompt_password_stderr;
use std::cell::RefCell;
use std::env;
use std::fs::File;
use std::io::Write;
use std::path::{Path, PathBuf};
use std::process;
use std::rc::Rc;
use bitbottle::*;
use bitbottle::cli::*;

const ABOUT: &str =
    "Build a file archive, with optional compression and encryption.";

const ARGS: &str = "\
-C --directory [PATH]       'change (cd) to this folder (directory) before parsing the paths to write into the archive'

-r --pub [FILE]...          'encrypt the archive, and add these ssh-ed25519 public key files to the list of recipients'
-p --password               'encrypt the archive, and allow a password to decrypt'
--aes                       'if encrypting, use aes-128-gcm to encrypt, instead of xchacha20-poly1305'

-z --snappy                 'compress with snappy (fast)'
-y --lzma2                  'compress with lzma2 (intense)'
--no-compress               'don't compress, even if one of the compression options in present'

--sha256                    'use SHA-256 (slow) instead of blake-3 for file and block hashes'
--blake2                    'use blake-2 instead of blake-3 for file and block hashes'

-v --verbose                'describe what's happening as the bitbottle is written'
-q --quiet                  'don't display progress bars or explanatory information'

-o [FILE]                   'write archive to file instead of stdout'
<PATH>...                   'files and folders to write into the archive'
";

const AFTER_HELP: &str = "\
Simplest behavior is to generate an uncompressed, unencrypted archive from
the files and folders listed on the command line, and write the archive to
STDOUT, or a file using '-o':

    bitbottle -o my_code.bb src/

Add '--snappy' (fast, average compression) or '--lzma2' (slow, extreme
compression) to compress the archive.

Add '--pub' to encrypt the archive for one or more SSH keys, using the public
keys to encrypt. Each corresponding private key will be able to independently
decrypt the archive. Add '--password' to use a password instead.

Add '--sha256' or '--blake2' to change the hash function used to split files
into blocks and verify file contents. The default is blake3, which is the
fastest on most platforms.

For example, to use extreme compression and encrypt for one recipient:

    bitbottle -o my_code.bb --lzma2 --pub ./tests/data/test-key.pub src/
";

static mut VERBOSE: bool = false;
static mut QUIET: bool = false;

macro_rules! verbose {
    ($($arg:tt)*) => ({
        if unsafe { VERBOSE } {
            eprintln!($($arg)*);
        }
    })
}

macro_rules! info {
    ($($arg:tt)*) => ({
        if !unsafe { QUIET } {
            eprintln!($($arg)*);
        }
    })
}

fn is_verbose() -> bool {
    unsafe { VERBOSE }
}

fn is_quiet() -> bool {
    unsafe { QUIET }
}


fn write_archive_pretty<W: Write>(
    mut writer: W,
    paths: Vec<PathBuf>,
    options: WriteArchiveOptions,
    progress: Rc<RefCell<ProgressLine>>,
    bytes_written: Rc<RefCell<u64>>,
) -> BottleResult<()> {
    write_archive(&mut writer, &paths, options, |state, file_list| {
        let mut progress = progress.borrow_mut();
        match state {
            WriteArchiveState::FileList { atlas } => {
                progress.show_bar = false;
                if is_verbose() {
                    progress.clear();
                    verbose!("{}", display_atlas_filename(&atlas.borrow()));
                    progress.force_update();
                }

                progress.update(0f64, format!("Finding files: {:>8} files, {:>5}",
                    file_list.total_file_count(), to_binary_si(file_list.total_size() as f64)));
            },
            WriteArchiveState::FileScan { size } => {
                if !progress.show_bar {
                    // new state!
                    progress.clear();
                    info!("Creating archive: {} files, {} bytes",
                        file_list.total_file_count(), to_binary_si(file_list.total_size() as f64));
                    progress.show_bar = true;
                }

                let total_size = file_list.total_size();
                let percent = size as f64 / total_size as f64;
                progress.update(percent, format!("[1/3] Scanning: {:>5}/{:>5}, {:>5} unique",
                    to_binary_si(size as f64), to_binary_si(total_size as f64),
                    to_binary_si(file_list.total_block_size() as f64)));
                if size == total_size {
                    progress.clear();
                    verbose!("Scanned unique blocks: {} blocks, {} bytes",
                        file_list.blocks.len(), to_binary_si(file_list.total_block_size() as f64));
                }
            },
            WriteArchiveState::SymlinkIgnoredWarning { symlink } => {
                info!("*** WARNING: Ignored symlink {:?} -> {:?}", symlink.source, symlink.target);
            },
            WriteArchiveState::TableOfContents { file_count } => {
                progress.show_bar = true;
                let percent = file_count as f64 / file_list.total_file_count() as f64;
                progress.update(percent, format!("[2/3] Writing TOC: {:>8}/{:>8} files",
                    file_count, file_list.total_file_count()));
            },
            WriteArchiveState::BlockData { block_count: _, size } => {
                progress.show_bar = true;
                let total_size = file_list.total_block_size();
                let percent = size as f64 / total_size as f64;
                progress.update(percent, format!("[3/3] Writing blocks: {:>5}/{:>5} -> {:>5}",
                    to_binary_si(size as f64), to_binary_si(total_size as f64),
                    to_binary_si(*bytes_written.borrow() as f64)));
                if size == total_size {
                    progress.force_update();
                }
            },
        }
        progress.display();
    })
}

fn create_archive(
    writer: Box<dyn Write>,
    paths: Vec<PathBuf>,
    password: Option<&str>,
    pk_filenames: Option<Vec<String>>,
    encryption: EncryptionAlgorithm,
    compression: Option<CompressionAlgorithm>,
    hash_type: HashType,
) -> BottleResult<()> {
    // load any public keys
    let pks: BottleResult<Vec<Ed25519PublicKey>> = pk_filenames.iter().flatten().map(|filename| {
        Ed25519PublicKey::from_ssh_file(filename).map(|pk| {
            verbose!("Encrypting for {} ({})", pad_truncate(pk.name(), 16), hex::encode(pk.as_bytes()));
            pk
        })
    }).collect();
    let pks = pks.unwrap_or_else(|e| {
        eprintln!("ERROR: Unable to read public key file: {}", e);
        process::exit(1);
    });

    let mut options = EncryptedBottleWriterOptions::new();
    if let Some(password) = password {
        options.key = EncryptionKey::Password(password.to_string());
    }
    options.algorithm = encryption;

    let progress = ProgressLine::new().bobble();
    progress.borrow_mut().show_bar = false;
    progress.borrow_mut().show_ever = !is_quiet();
    let bytes_written = Rc::new(RefCell::new(0u64));

    // the archive is built from a file list, then compressed, then optionally
    // encrypted, and finally the output bytes are counted so we can show
    // status. but we have to specify these layers inside out:
    // counting, encryption, compression, archive.

    let mut writer = CountingWriter::new(writer, |byte_count, _done| {
        bytes_written.replace(byte_count);
    });
    let archive_options = WriteArchiveOptions { hash_type, ..WriteArchiveOptions::default() };

    if !pks.is_empty() || password.is_some() {
        let mut writer = if pks.is_empty() {
            EncryptedBottleWriter::new(writer, options)?
        } else {
            EncryptedBottleWriter::for_recipients(writer, options, pks.as_slice())?
        };
        if let Some(compression) = compression {
            let mut writer = CompressedBottleWriter::new(writer, compression)?;
            write_archive_pretty(
                &mut writer, paths, archive_options, progress.clone(), bytes_written.clone()
            )?;
            writer.close()?.close()?;
        } else {
            write_archive_pretty(
                &mut writer, paths, archive_options, progress.clone(), bytes_written.clone()
            )?;
            writer.close()?;
        }
    } else if let Some(compression) = compression {
        let mut writer = CompressedBottleWriter::new(writer, compression)?;
        write_archive_pretty(
            &mut writer, paths, archive_options, progress.clone(), bytes_written.clone()
        )?;
        writer.close()?;
    } else {
        write_archive_pretty(
            &mut writer, paths, archive_options, progress.clone(), bytes_written.clone()
        )?;
    }

    progress.borrow_mut().clear();
    verbose!("Wrote {} bytes.", to_binary_si(*bytes_written.borrow() as f64));
    Ok(())
}



pub fn main() {
    let args = App::new("bitbottle")
        .setting(AppSettings::DeriveDisplayOrder)
        .setting(AppSettings::UnifiedHelpMessage)
        .version("0.1")
        .author("Robey Pointer <robey@lag.net>")
        .about(ABOUT)
        .args_from_usage(ARGS)
        .after_help(AFTER_HELP)
        .get_matches();

    if args.is_present("verbose") {
        unsafe { VERBOSE = true; }
    }
    if args.is_present("quiet") {
        unsafe { QUIET = true; }
    }

    let paths: Vec<PathBuf> = args.values_of("PATH").unwrap().map(PathBuf::from).collect();

    // encrypt?
    let password = if args.is_present("password") {
        prompt_password_stderr("Password: ").map_err(|e| {
            eprintln!("ERROR: {}", e);
            process::exit(1);
        }).ok()
    } else {
        None
    };
    let pk_filenames = args.values_of_lossy("pub");
    let encryption = if args.is_present("aes") {
        EncryptionAlgorithm::AES_128_GCM
    } else {
        EncryptionAlgorithm::XCHACHA20_POLY1305
    };

    let mut compression = if args.is_present("snappy") {
        Some(CompressionAlgorithm::SNAPPY)
    } else if args.is_present("lzma2") {
        Some(CompressionAlgorithm::LZMA2)
    } else {
        None
    };
    if args.is_present("no-compress") {
        compression = None;
    }

    // change file/block hash?
    let mut hash_type = HashType::BLAKE3;
    if args.is_present("blake2") {
        hash_type = HashType::BLAKE2;
    }
    if args.is_present("sha256") {
        hash_type = HashType::SHA256;
    }

    let writer: Box<dyn Write> = match args.value_of("o") {
        Some(filename) => Box::new(File::create(filename).unwrap_or_else(|e| {
            eprintln!("ERROR: Unable to write to file {}: {}", filename, e);
            process::exit(1);
        })),
        None => Box::new(std::io::stdout()),
    };

    if let Some(folder) = args.value_of("directory") {
        env::set_current_dir(Path::new(folder)).unwrap_or_else(|e| {
            eprintln!("ERROR: Unable to change directory to {}: {}", folder, e);
            process::exit(1);
        });
    }

    create_archive(
        writer,
        paths,
        password.as_deref(),
        pk_filenames,
        encryption,
        compression,
        hash_type,
    ).unwrap_or_else(|e| {
        eprintln!("ERROR: {}", e);
    });
}
