use lzma::LzmaError;
use std::error::Error;
use std::fmt;
use std::path::PathBuf;
use crate::bottle_cap::BottleType;

/// std::io::Error is too annoying
#[derive(Debug)]
pub enum BottleError {
    IoError(std::io::Error),
    FileError(String, std::io::ErrorKind),
    BadMagic,
    UnknownVersion,
    UnknownBottleType,
    HeaderTooLarge,
    BadCrc { expected: u32, got: u32 },
    CorruptStream,
    InvalidBottleState,
    WrongBottleType { expected: BottleType, got: BottleType },
    WrongStreamType,
    UnexpectedStream,
    MissingHeader,
    NoStream,  // internal error

    // signatures:
    NotAnEd25519Key,
    SshEncodingError,
    InvalidSshFile(String),
    UnsupportedSshFileEncryption(String, String),
    SshPasswordRequired,
    DryocBoxError,

    // compressed bottles:
    UnknownCompression,
    CompressionError,  // probably an internal error in the compressor
    Lzma2Error(LzmaError),

    // encrypted bottles:
    NoRandom,
    UnknownEncryption,
    BadKeySize,
    RequiresKey,
    NotPasswordEncrypted,
    Argon2Error(argonautica::Error),  // probably an internal error
    CorruptPublicKey,
    NoMatchingPublicKey(Vec<String>),
    CipherError,

    // file bottles:
    IncompleteFileArchive,
    UnknownHashType,
    InvalidAddPath(PathBuf),
    BadPath,
    DuplicatePaths(PathBuf, PathBuf),
}

impl BottleError {
    pub fn to_io_error(self) -> std::io::Error {
        std::io::Error::new(std::io::ErrorKind::Other, Box::new(self))
    }
}

impl Error for BottleError {

}

impl From<std::io::Error> for BottleError {
    fn from(error: std::io::Error) -> BottleError {
        BottleError::IoError(error)
    }
}

impl fmt::Display for BottleError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::IoError(e) => {
                if let Some(inner) = e.get_ref() {
                    if e.kind() == std::io::ErrorKind::Other {
                        write!(f, "{}", inner)?
                    } else {
                        write!(f, "I/O error: {:?}: {}", e.kind(), inner)?
                    }
                } else {
                    write!(f, "I/O error: {:?}", e.kind())?
                }
            },
            Self::FileError(filename, e) => write!(f, "{}: {:?}", filename, e)?,
            Self::BadMagic => write!(f, "Bad magic number")?,
            Self::UnknownVersion => write!(f, "Unknown bottle version")?,
            Self::UnknownBottleType => write!(f, "Unknown bottle type")?,
            Self::HeaderTooLarge => write!(f, "Bottle header too large")?,
            Self::BadCrc { expected, got } => write!(f, "Bad CRC32C: got {:x}, expected {:x}", got, expected)?,
            Self::CorruptStream => write!(f, "Corrupt stream (bad signal)")?,
            Self::InvalidBottleState => write!(f, "Internal error: previous stream was not closed")?,
            Self::WrongBottleType { expected, got } => {
                write!(f, "Wrong bottle type: got {:?}, expected {:?}", got, expected)?
            },
            Self::WrongStreamType => write!(f, "Corrupt bottle: incorrect stream type(s) for bottle")?,
            Self::UnexpectedStream => write!(f, "Extra stream in bottle")?,
            Self::MissingHeader => write!(f, "Bottle is missing a mandatory header")?,
            Self::NoStream => write!(f, "No stream has been opened in this bottle")?,
            Self::NotAnEd25519Key => write!(f, "Not an Ed25519 key")?,
            Self::SshEncodingError => write!(f, "Error decoding ssh key format")?,
            Self::InvalidSshFile(filename) => write!(f, "Not an ssh key file: {}", filename)?,
            Self::UnsupportedSshFileEncryption(filename, kind) => {
                write!(f, "Unsupported encryption ({}) in ssh file: {}", kind, filename)?
            },
            Self::SshPasswordRequired => write!(f, "Password required for this SSH key file")?,
            Self::DryocBoxError => write!(f, "Error encrypting message with Ed25519 key material")?,
            Self::UnknownCompression => write!(f, "Unknown compression")?,
            Self::CompressionError => write!(f, "Error compressing data")?,
            Self::Lzma2Error(e) => write!(f, "LZMA error: {}", e)?,
            Self::NoRandom => write!(f, "No CPRNG random bytes available")?,
            Self::UnknownEncryption => write!(f, "Unknown encryption")?,
            Self::BadKeySize => write!(f, "Bad key size for encryption algorithm")?,
            Self::RequiresKey => write!(f, "No key or password provided for encrypted bottle")?,
            Self::NotPasswordEncrypted => {
                write!(f, "Password provided but encrypted bottle is not password protected")?
            },
            Self::Argon2Error(e) => write!(f, "Argon2 key generation error: {}", e)?,
            Self::CorruptPublicKey => write!(f, "Public key in encrypted bottle is corrupted")?,
            Self::NoMatchingPublicKey(names) => {
                write!(f, "No public key that can decode this bottle (encrypted for: {})", names.join(", "))?
            },
            Self::CipherError => write!(f, "Encryption engine failure (wrong password or key?)")?,
            Self::IncompleteFileArchive => write!(f, "File archive is missing some blocks (corrupted)")?,
            Self::UnknownHashType => write!(f, "Unknown hash type for file blocking")?,
            Self::InvalidAddPath(path) => {
                write!(f, "Invalid path for adding to archive (contains . or ..): {:?}", path)?
            },
            Self::BadPath => write!(f, "Invalid path (failed to parse: OS error?)")?,
            Self::DuplicatePaths(path1, path2) => {
                write!(f, "Paths would resolve to the same relative path in the archive: {:?}, {:?}", path1, path2)?
            },
        };
        Ok(())
    }
}

pub type BottleResult<T> = Result<T, BottleError>;
