use num_enum::TryFromPrimitive;
use std::fmt;
use std::io::{Read, Write};
use crate::bottle_cap::{BottleCap, BottleType};
use crate::bottle_error::{BottleError, BottleResult};
use crate::framed::{FramingWriter, UnframingReader};
use crate::header::Header;

// frame headers 40, 80, c0 are reserved, so use them for out-of-band signalling.
#[repr(u8)]
#[derive(TryFromPrimitive)]
enum Signal {
    Data = 0x40,
    Bottle = 0x80,
    End = 0xc0,
}

enum WriterState<W: Write> {
    Idle(W),
    Data(FramingWriter<W>),
    Bottle(W),
}

/// Generate a bitbottle and write it into a `Write` downstream.
pub struct BottleWriter<W: Write> {
    bottle_cap: BottleCap,
    block_bits: usize,
    state: Option<WriterState<W>>,
}

impl<W: Write> BottleWriter<W> {
    /// Start a new bottle and write the bottle cap & header to a `Write`
    /// output stream. Frames in the stream will buffer a block size of
    /// `2 ** block_bits`.
    pub fn new(
        mut output: W,
        bottle_type: BottleType,
        header: Header,
        block_bits: usize,
    ) -> BottleResult<BottleWriter<W>> {
        let bottle_cap = BottleCap::new(bottle_type, header);
        bottle_cap.write(&mut output)?;
        let state = Some(WriterState::Idle(output));
        Ok(BottleWriter { bottle_cap, block_bits, state })
    }

    pub fn write_bottle(&mut self) -> BottleResult<()> {
        if let Some(WriterState::Idle(mut output)) = self.state.take() {
            output.write_all(&[ Signal::Bottle as u8 ])?;
            self.state = Some(WriterState::Bottle(output));
            Ok(())
        } else {
            Err(BottleError::InvalidBottleState)
        }
    }

    pub fn write_data_stream(&mut self) -> BottleResult<()> {
        if let Some(WriterState::Idle(mut output)) = self.state.take() {
            output.write_all(&[ Signal::Data as u8 ])?;
            self.state = Some(WriterState::Data(FramingWriter::new(output, self.block_bits)));
            Ok(())
        } else {
            Err(BottleError::InvalidBottleState)
        }
    }

    pub fn close_stream(&mut self) -> BottleResult<()> {
        if let Some(state) = self.state.take() {
            match state {
                WriterState::Bottle(output) => {
                    self.state = Some(WriterState::Idle(output));
                    Ok(())
                },
                WriterState::Data(framing_writer) => {
                    let output = framing_writer.close()?;
                    self.state = Some(WriterState::Idle(output));
                    Ok(())
                },
                _ => Err(BottleError::InvalidBottleState),
            }
        } else {
            Err(BottleError::InvalidBottleState)
        }
    }

    pub fn write_reader_data_stream(&mut self, reader: &mut dyn Read) -> BottleResult<()> {
        self.write_data_stream()?;
        std::io::copy(reader, self)?;
        self.close_stream()?;
        Ok(())
    }

    pub fn close(mut self) -> BottleResult<W> {
        if let Some(WriterState::Idle(mut output)) = self.state.take() {
            output.write_all(&[ Signal::End as u8 ])?;
            output.flush()?;
            Ok(output)
        } else {
            Err(BottleError::InvalidBottleState)
        }
    }
}

impl<W: Write> Write for BottleWriter<W> {
    fn write(&mut self, data: &[u8]) -> std::io::Result<usize> {
        match self.state {
            None | Some(WriterState::Idle(_)) => {
                Err(BottleError::InvalidBottleState.to_io_error())
            },
            Some(WriterState::Bottle(ref mut output)) => {
                output.write(data)
            },
            Some(WriterState::Data(ref mut framing_writer)) => {
                framing_writer.write(data)
            },
        }
    }

    fn flush(&mut self) -> std::io::Result<()> {
        Ok(())
    }
}

impl<W: Write> fmt::Debug for BottleWriter<W> {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "BottleWriter(type={:02x}, {:?})", self.bottle_cap.bottle_type as u8, self.bottle_cap.header)
    }
}


enum ReaderState<R: Read> {
    Idle(R),
    Data(UnframingReader<R>),
    Bottle(Box<BottleReader<R>>),
    Done(R),
}

impl<R: Read> fmt::Debug for ReaderState<R> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            ReaderState::Idle(_) => write!(f, "IDLE"),
            ReaderState::Data(_) => write!(f, "DATA"),
            ReaderState::Bottle(_) => write!(f, "BOTTLE"),
            ReaderState::Done(_) => write!(f, "DONE"),
        }
    }
}

#[derive(Clone, Copy, Debug, PartialEq)]
pub enum BottleStream {
    End,
    Data,
    Bottle,
}

pub struct BottleReader<R: Read> {
    pub bottle_cap: BottleCap,
    state: Option<ReaderState<R>>,
}

/// Parse a bitbottle from a `Read` byte stream.
impl<R: Read> BottleReader<R> {
    pub fn new(mut reader: R) -> BottleResult<BottleReader<R>> {
        let bottle_cap = BottleCap::read(&mut reader)?;
        let state = Some(ReaderState::Idle(reader));
        Ok(BottleReader { bottle_cap, state })
    }

    pub fn expect_type(&self, bottle_type: BottleType) -> BottleResult<()> {
        if self.bottle_cap.bottle_type != bottle_type {
            return Err(BottleError::WrongBottleType { expected: bottle_type, got: self.bottle_cap.bottle_type });
        }
        Ok(())
    }

    /// Read the next nested stream within this bottle. On success, it returns
    /// a `BottleStream` indicating whether we've reached the end (`End`),
    /// or the next stream is raw data (`Data`) or a nested bottle (`Bottle`).
    pub fn next_stream(&mut self) -> BottleResult<BottleStream> {
        if let Some(ReaderState::Idle(mut reader)) = self.state.take() {
            let mut buffer = [0u8; 1];
            reader.read_exact(&mut buffer)?;
            let signal: Signal = buffer[0].try_into().map_err(|_| BottleError::CorruptStream)?;
            match signal {
                Signal::Data => {
                    self.state = Some(ReaderState::Data(UnframingReader::new(reader)));
                    Ok(BottleStream::Data)
                },
                Signal::Bottle => {
                    let r = BottleReader::new(reader)?;
                    self.state = Some(ReaderState::Bottle(Box::new(r)));
                    Ok(BottleStream::Bottle)
                },
                Signal::End => {
                    self.state = Some(ReaderState::Done(reader));
                    Ok(BottleStream::End)
                },
            }
        } else {
            Err(BottleError::InvalidBottleState)
        }
    }

    pub fn expect_next_stream(&mut self, stream_type: BottleStream) -> BottleResult<()> {
        if self.next_stream()? != stream_type {
            return Err(BottleError::WrongStreamType);
        }
        Ok(())
    }

    pub fn expect_end(&mut self) -> BottleResult<()> {
        if self.next_stream()? != BottleStream::End {
            return Err(BottleError::UnexpectedStream);
        }
        Ok(())
    }

    pub fn expect_end_and_close(mut self) -> BottleResult<R> {
        self.expect_end()?;
        self.close()
    }

    pub fn close_stream(&mut self) -> BottleResult<()> {
        if let Some(state) = self.state.take() {
            match state {
                ReaderState::Data(unframing_reader) => {
                    let reader = unframing_reader.close();
                    self.state = Some(ReaderState::Idle(reader));
                    Ok(())
                },
                ReaderState::Bottle(bottle_reader) => {
                    let reader = bottle_reader.close()?;
                    self.state = Some(ReaderState::Idle(reader));
                    Ok(())
                },
                _ => Err(BottleError::InvalidBottleState),
            }
        } else {
            Err(BottleError::InvalidBottleState)
        }
    }

    pub fn close(mut self) -> BottleResult<R> {
        if let Some(ReaderState::Done(reader)) = self.state.take() {
            Ok(reader)
        } else {
            Err(BottleError::InvalidBottleState)
        }
    }

    // if we're in a data stream, provide a reference
    pub fn data_stream(&mut self) -> BottleResult<&mut dyn Read> {
        if let Some(ReaderState::Data(ref mut unframing_reader)) = self.state {
            Ok(unframing_reader)
        } else {
            Err(BottleError::InvalidBottleState)
        }
    }

    // if we're in a bottle stream, provide a reference
    pub fn bottle_reader(&mut self) -> BottleResult<&mut BottleReader<R>> {
        if let Some(ReaderState::Bottle(ref mut bottle_reader)) = self.state {
            Ok(bottle_reader)
        } else {
            Err(BottleError::InvalidBottleState)
        }
    }
}


// ----- tests

#[cfg(test)]
mod test {
    use hex::{decode, encode};
    use crate::bottle_cap::BottleType;
    use super::{BottleReader, BottleStream, BottleWriter, Header};

    const MAGIC: &str = "f09f8dbc";
    const TEST_NO_HEADER: &str = "000f00006b11c604";
    const TEST_HEADER_150: &str = "000f0200214155310096";

    #[test]
    fn write_data() {
        let mut buffer = Vec::new();
        let mut b1 = BottleWriter::new(&mut buffer, BottleType::Test, Header::new(), 20).unwrap();
        let mut data: &[u8] = &[ 0xff ];
        b1.write_reader_data_stream(&mut data).unwrap();
        b1.close().unwrap();
        assert_eq!(encode(buffer), format!("{}{}4001ffc0", MAGIC, TEST_NO_HEADER));
    }

    #[test]
    fn write_bottle() {
        let mut buffer = Vec::new();
        let mut b1 = BottleWriter::new(Box::new(&mut buffer), BottleType::Test, Header::new(), 20).unwrap();
        {
            // let inner drop and release its hold on b1 at the end of this context block:
            b1.write_bottle().unwrap();
            let mut h = Header::new();
            h.add_int(0, 150).unwrap();
            let mut b2 = BottleWriter::new(Box::new(&mut b1), BottleType::Test, h, 20).unwrap();
            let mut data: &[u8] = &[ 0xff ];
            b2.write_reader_data_stream(&mut data).unwrap();
            b2.close().unwrap();
            b1.close_stream().unwrap();
        }
        b1.close().unwrap();
        assert_eq!(encode(&buffer), format!("{}{}80{}{}4001ffc0c0", MAGIC, TEST_NO_HEADER, MAGIC, TEST_HEADER_150));
    }

    #[test]
    fn read_data() {
        let buffer = &decode(format!("{}{}4001ffc0", MAGIC, TEST_NO_HEADER)).unwrap()[..];
        let mut b1 = BottleReader::new(buffer).unwrap();
        assert_eq!(b1.bottle_cap.bottle_type, BottleType::Test);
        assert_eq!(format!("{:?}", b1.bottle_cap.header), "Header()");

        assert_eq!(b1.next_stream().unwrap(), BottleStream::Data);
        {
            let reader = b1.data_stream().unwrap();
            let mut buffer = Vec::new();
            reader.read_to_end(&mut buffer).unwrap();
            assert_eq!(encode(buffer), "ff");
            b1.close_stream().unwrap();
        }
        assert_eq!(b1.next_stream().unwrap(), BottleStream::End);
    }

    #[test]
    fn read_bottle() {
        let hex = format!("{}{}80{}{}4001ffc0c0", MAGIC, TEST_NO_HEADER, MAGIC, TEST_HEADER_150);
        let buffer = &decode(hex).unwrap()[..];
        let mut b1 = BottleReader::new(buffer).unwrap();
        assert_eq!(b1.bottle_cap.bottle_type, BottleType::Test);
        assert_eq!(format!("{:?}", b1.bottle_cap.header), "Header()");

        assert_eq!(b1.next_stream().unwrap(), BottleStream::Bottle);
        {
            let b2 = b1.bottle_reader().unwrap();
            assert_eq!(b2.bottle_cap.bottle_type, BottleType::Test);
            assert_eq!(format!("{:?}", b2.bottle_cap.header), "Header(U8(0)=150)");

            // we need to go deeper.
            assert_eq!(b2.next_stream().unwrap(), BottleStream::Data);
            {
                let reader = b2.data_stream().unwrap();
                let mut buffer = Vec::new();
                reader.read_to_end(&mut buffer).unwrap();
                assert_eq!(encode(buffer), "ff");
                b2.close_stream().unwrap();
            }
            assert_eq!(b2.next_stream().unwrap(), BottleStream::End);
            b1.close_stream().unwrap();
        }
        assert_eq!(b1.next_stream().unwrap(), BottleStream::End);
    }
}
