use std::collections::{HashMap, HashSet};
use std::fs;
use std::io::{Read, Seek, SeekFrom, Write};
use std::os::unix::prelude::{FileExt, PermissionsExt};
use std::path::{Component, Path, PathBuf};
use crate::FileAtlasRef;
use crate::hashing::HashType;
use crate::bottle::BottleReader;
use crate::bottle_error::{BottleError, BottleResult};
use crate::file_atlas::FileAtlas;
use crate::file_bottle::{FileListBottleReader, FileListBottleWriter};
use crate::file_list::{Block, BlockHash, FileList, Symlink};
use crate::file_scanner::{FileScanner, ScanState};


#[derive(Clone, Debug, PartialEq, Eq)]
pub enum WriteArchiveState {
    /// Building the list of files to scan, by traversing recursively into
    /// the given paths. `FileList.files.len()` will be growing, and `atlas`
    /// will contain each new file or folder as it's unearthed.
    FileList { atlas: FileAtlasRef },

    /// Warning that a symlink is invalid and not being added to the archive.
    SymlinkIgnoredWarning { symlink: Symlink },

    /// Scanning each file to break it into blocks, compute the SHA-256 of
    /// each block, and the SHA-256 of the overall file. `size` is the bytes
    /// processed so far, out of `FileList.total_size()`.
    FileScan { size: u64 },

    /// Writing the table of contents (filenames, metadata, and block maps)
    /// into the archive. `file_count` is the number of files completed so
    /// far, out of `FileList.files.len()`.
    TableOfContents { file_count: usize },

    /// Writing block data to the archive. `block_count` is the number of
    /// blocks so far, out of `FileList.blocks.len()`, and `size` is the
    /// bytes so far, out of `FileList.total_size()`.
    BlockData { block_count: usize, size: u64 },
}

#[derive(Clone, Debug, PartialEq, Eq)]
pub enum ReadArchiveState {
    /// Reading the table of contents (filenames, metadata, and block maps)
    /// from the archive. `file_count` is the number of files completed so
    /// far, and `total` is the total reported by the archive header.
    /// `hash_type` is the hash algorithm used for blocking.
    TableOfContents { file_count: usize, total: usize, hash_type: HashType },

    /// If we encounter a path with illegal segments (`.`, `..`, or an
    /// absolute path), we emit this warning, weth the bad path and our
    /// attempt to correct it. We will use the corrected path. You are free
    /// to ignore this warning.
    BadPathWarning { bad_path: PathBuf, new_path: PathBuf },

    /// Warning that the archive contains an invalid symlink that will be
    /// ignored.
    SymlinkIgnoredWarning { symlink: Symlink },

    /// Reading blocks from the archive and writing them to disk.
    /// `blocks_read` may be less than `blocks_written` because a single
    /// block may appear in multiple places.
    BlockData { blocks_read: usize, blocks_written: usize, bytes_read: u64, bytes_written: u64 },
}


pub struct WriteArchiveOptions {
    pub hash_type: HashType,
    pub min_bits: u8,
    pub pref_bits: u8,
    pub max_bits: u8,
    pub window_bits: u8,
}

impl WriteArchiveOptions {
    // good defaults
    pub fn new() -> WriteArchiveOptions {
        WriteArchiveOptions {
            hash_type: HashType::SHA256,
            min_bits: 18,
            pref_bits: 20,
            max_bits: 22,
            window_bits: 10,
        }
    }
}

impl Default for WriteArchiveOptions {
    fn default() -> Self {
        Self::new()
    }
}



pub fn write_archive<W: Write, Callback: FnMut (WriteArchiveState, &FileList)>(
    writer: W,
    paths: &[PathBuf],
    options: WriteArchiveOptions,
    mut updater: Callback,
) -> BottleResult<()> {
    // first, build a file & block list
    let mut file_scanner = FileScanner::new(
        options.hash_type,
        options.min_bits,
        options.pref_bits,
        options.max_bits,
        options.window_bits,
        vec![0u8; 0x1_0000],
        |scan_state, file_list, size| {
            match scan_state {
                ScanState::FileList(atlas) => {
                    updater(WriteArchiveState::FileList { atlas }, file_list)
                },
                ScanState::Chunking => {
                    updater(WriteArchiveState::FileScan { size }, file_list);
                },
            }
        },
    );

    file_scanner.scan_paths(paths)?;
    file_scanner.build_block_list(options.hash_type)?;
    let mut file_list: FileList = file_scanner.into();

    for symlink in file_list.drop_stray_symlinks() {
        updater(WriteArchiveState::SymlinkIgnoredWarning { symlink: symlink.clone() }, &file_list);
    }

    // write the table of contents.
    let mut seen_blocks: HashSet<BlockHash> = HashSet::new();
    let mut block_bytes_written: u64 = 0;
    let mut bottle_writer = FileListBottleWriter::new(writer, options.hash_type, &file_list, |file_count| {
        updater(WriteArchiveState::TableOfContents { file_count }, &file_list);
    })?;

    // write the individual blocks, whenever we come across one we haven't written yet.
    for atlas in &file_list.files {
        let mut open_file: Option<fs::File> = None;
        let mut seek: u64 = 0;
        let atlas = atlas.borrow();
        for block in &atlas.contents.blocks {
            if !seen_blocks.contains(&block.hash) {
                // read this block from the file (opening if necessary)
                let mut f = open_file.take().map(Ok).unwrap_or_else(|| fs::File::open(&atlas.path))?;
                f.seek(SeekFrom::Start(seek))?;
                let mut buffer = vec![0u8; block.size];
                f.read_exact(&mut buffer)?;

                // just another block in the wall
                bottle_writer.add_block(block, &buffer)?;
                seen_blocks.insert(block.hash);
                block_bytes_written += block.size as u64;
                open_file = Some(f);

                updater(
                    WriteArchiveState::BlockData { block_count: seen_blocks.len(), size: block_bytes_written },
                    &file_list,
                );
            }
            seek += block.size as u64;
        }
    }

    // bye!
    bottle_writer.close()?;
    Ok(())
}


pub struct ReadArchiveOptions {
    file_filter: Box<dyn FnMut (&FileAtlas) -> bool>,
}

impl ReadArchiveOptions {
    // good defaults
    pub fn new() -> ReadArchiveOptions {
        ReadArchiveOptions {
            file_filter: Box::new(|_| true),
        }
    }
}

impl Default for ReadArchiveOptions {
    fn default() -> Self {
        Self::new()
    }
}


/// Read the archive's file list, but don't expand the files.
pub fn read_archive<R: Read, Callback: FnMut (ReadArchiveState, &FileList)>(
    bottle_reader: BottleReader<R>,
    updater: &mut Callback,
) -> BottleResult<FileListBottleReader<R>> {
    // read the table of contents.
    let bottle_reader = FileListBottleReader::new(bottle_reader, |file_count, total, hash_type, file_list| {
        // check for bad path that must be corrected
        {
            let mut atlas = file_list.files.last().unwrap().borrow_mut();
            let mut components = atlas.normalized_path.components();
            if !components.all(|c| matches!(c, Component::Normal(_))) {
                let bad_path = atlas.normalized_path.clone();
                let new_path: PathBuf = components.filter(|c| matches!(c, Component::Normal(_))).collect();
                atlas.normalized_path = new_path.clone();
                updater(ReadArchiveState::BadPathWarning { bad_path, new_path }, file_list);
            }
        }

        updater(ReadArchiveState::TableOfContents { file_count, total, hash_type }, file_list);
    })?;
    Ok(bottle_reader)
}


pub fn expand_archive<R: Read, Callback: FnMut (ReadArchiveState, &FileList)>(
    bottle_reader: BottleReader<R>,
    mut options: ReadArchiveOptions,
    dest_path: &Path,
    mut updater: Callback,
) -> BottleResult<R> {
    fs::create_dir_all(dest_path)?;

    let mut bottle_reader = read_archive(bottle_reader, &mut updater)?;

    // for safety:
    for symlink in bottle_reader.file_list.drop_stray_symlinks() {
        updater(ReadArchiveState::SymlinkIgnoredWarning { symlink: symlink.clone() }, &bottle_reader.file_list);
    }

    let mut blocks_read = 0;
    let mut blocks_written = 0;
    let mut bytes_read = 0u64;
    let mut bytes_written = 0u64;

    // cache open files, and which blocks are missing
    let mut file_cache: HashMap<PathBuf, (fs::File, HashSet<Block>)> = HashMap::new();

    // first, create folders
    for atlas in bottle_reader.file_list.files.iter().filter(|atlas| atlas.borrow().is_folder) {
        let atlas = atlas.borrow();
        let path = dest_path.join(&atlas.normalized_path);
        fs::create_dir_all(&path)?;
        fs::set_permissions(&path, PermissionsExt::from_mode(atlas.perms))?;
    }

    while let Some((block, data)) = bottle_reader.next_block()? {
        blocks_read += 1;
        bytes_read += data.len() as u64;

        // would be nice to flatten() the Option<Vec<_>>, but nobody knows how.
        if let Some(atlas_list) = bottle_reader.file_list.file_map.get(&block.hash) {
            for atlas in atlas_list {
                let atlas = atlas.borrow();
                if !(options.file_filter)(&atlas) { continue; }
                let path = dest_path.join(&atlas.normalized_path);

                // find (or create) the file handle, and set of remaining blocks for this file
                if !file_cache.contains_key(&path) {
                    // in case we had to create this path by defanging:
                    if let Some(p) = path.parent() {
                        fs::create_dir_all(p)?;
                    }

                    let file = fs::File::create(&path)?;
                    file.set_len(atlas.size)?;
                    file.set_permissions(PermissionsExt::from_mode(atlas.perms))?;
                    let block_set: HashSet<Block> = atlas.contents.blocks.iter().cloned().collect();
                    file_cache.insert(path.clone(), (file, block_set));
                }
                // fine to assert since we just added it:
                let (file, block_set) = file_cache.get_mut(&path).unwrap();

                // write this block wherever it was wronged!
                for (offset, _block) in atlas.contents.offsets_of(&block.hash) {
                    file.write_all_at(&data, offset)?;
                    blocks_written += 1;
                    bytes_written += data.len() as u64;
                    updater(
                        ReadArchiveState::BlockData { blocks_read, blocks_written, bytes_read, bytes_written },
                        &bottle_reader.file_list
                    );
                }

                // close the file if that was the last block it needed.
                block_set.remove(&block);
                if block_set.is_empty() {
                    file.sync_all()?;
                    file_cache.remove(&path);
                }
            }
        }
    }

    // create any zero-length files that have no block data
    for atlas in bottle_reader.file_list.files.iter().filter(|atlas| {
        let atlas = atlas.borrow();
        !atlas.is_folder && atlas.symlink_target.is_none() && atlas.size == 0
    }) {
        let atlas = atlas.borrow();
        let path = dest_path.join(&atlas.normalized_path);

        // in case we had to create this path by defanging:
        if let Some(p) = path.parent() {
            fs::create_dir_all(p)?;
        }

        fs::File::create(&path)?.set_len(atlas.size)?;
    }

    // create symlinks
    for atlas in bottle_reader.file_list.files.iter().filter(|atlas| atlas.borrow().symlink_target.is_some()) {
        let atlas = atlas.borrow();
        let path = dest_path.join(&atlas.normalized_path);
        std::os::unix::fs::symlink(atlas.symlink_target.as_ref().unwrap(), path)?;
    }

    if !file_cache.is_empty() {
        // this really can't happen unless someone made a logic error
        return Err(BottleError::IncompleteFileArchive);
    }

    bottle_reader.close()
}
