//! bit-struct

#![no_std]

use core::marker::PhantomData;
use core::ops::{BitAnd, BitOrAssign, BitXorAssign, Shl, ShlAssign, Shr, ShrAssign};
use num_traits::{Bounded, Num};

/// A struct which allows for getting/setting a given property
pub struct GetSet<'a, P, T, const START: usize, const STOP: usize> {
    /// The referenced bitfield type.
    parent: &'a mut P,
    /// The type in the get/set operations
    _phantom: PhantomData<T>,
}

impl<
        'a,
        P: Num + Bounded + ShlAssign<usize> + ShrAssign<usize>,
        T,
        const START: usize,
        const STOP: usize,
    > GetSet<'a, P, T, START, STOP>
{
    /// Create a new [GetSet]. This should be called from methods generated by the [bit_struct]
    /// macro
    pub fn new(parent: &'a mut P) -> Self {
        Self {
            parent,
            _phantom: PhantomData::default(),
        }
    }

    /// Get a mask of STOP-START + 1 length. This doesn't use the shift left and subtract one
    /// trick because of the special case where (0b1 << (STOP - START + 1)) - 1 will cause an
    /// overflow
    fn mask(&self) -> P {
        let num_bits = core::mem::size_of::<P>() * 8;
        let mut max_value = P::max_value();
        let keep_bits = STOP - START + 1;

        max_value >>= num_bits - keep_bits;
        max_value
    }
}

impl<
        'a,
        P: Num
            + Shl<usize, Output = P>
            + Shr<usize, Output = P>
            + ShlAssign<usize>
            + ShrAssign<usize>
            + Bounded
            + BitAnd<Output = P>
            + Copy,
        T: TryFrom<P>,
        const START: usize,
        const STOP: usize,
    > GetSet<'a, P, T, START, STOP>
{
    /// Get the property. Returns an error it does not exist.
    pub fn get(&self) -> Result<T, T::Error> {
        let parent = *self.parent;
        let mask = self.mask();
        let section = (parent >> START) & mask;
        T::try_from(section)
    }
}

impl<
        'a,
        P: Num
            + Shl<usize, Output = P>
            + Copy
            + BitOrAssign
            + BitXorAssign
            + BitAnd<Output = P>
            + ShlAssign<usize>
            + ShrAssign<usize>
            + Bounded,
        T: Into<P>,
        const START: usize,
        const STOP: usize,
    > GetSet<'a, P, T, START, STOP>
{
    /// Set the property
    pub fn set(&mut self, value: T) {
        let mask = self.mask();
        let mask_shifted = mask << START;

        // zero out parent
        *self.parent |= mask_shifted;
        *self.parent ^= mask_shifted;

        let to_set = value.into() & mask;
        *self.parent |= to_set << START;
    }
}

/// Create a bit struct. Look at tests folder to see examples.
/// ```
/// bit_struct::enums!(
///     /// Mode
///     Mode { Zero, One }
/// );
///
/// bit_struct::bit_struct!(
///     struct Abc(u16){
///         mode(15,15): Mode,
///         count(1,5): u8,
///     }
///
///     struct FullCount(u16){
///         count(0,15): u16,
///     }
///
///     struct TooManyBits(u16) {
///         count(0,15): u8
///     }
///
/// );
/// ```
#[macro_export]
macro_rules! bit_struct {
    (
        $(
        $struct_vis: vis struct $name: ident ($kind: ty) {
        $($field: ident($start: literal, $end: literal): $actual: ty),* $(,)?
        }
        )*
    ) => {
        $(
        $struct_vis struct $name($kind);

        impl $name {
            $(
            pub fn $field(&mut self) -> bit_struct::GetSet<'_, $kind, $actual, $start, $end>{
                bit_struct::GetSet::new(&mut self.0)
            }
            )*
        }
        )*
    };
}

/// Create enums which have convenient TryFrom/From implementations. This makes using them with
/// the [bit_struct] macro much easier.
/// ```
/// bit_struct::enums!(
///     /// Mode
///     Mode { Zero, One }
/// );
///
/// bit_struct::bit_struct!(
///     struct Abc(u16){
///         mode(15,15): Mode,
///         count(1,5): u8,
///     }
///
///     struct FullCount(u16){
///         count(0,15): u16,
///     }
///
///     struct TooManyBits(u16) {
///         count(0,15): u8
///     }
///
/// );
/// ```
#[macro_export]
macro_rules! enums {
    (
        $(
            $(#[doc = $struct_doc:expr])*
            $(#[derive($($struct_der:ident),+)])?
            $enum_vis: vis $name: ident {
                $(#[doc = $fst_field_doc:expr])*
                $fst_field: ident
                $(,
                    $(#[doc = $field_doc:expr])*
                    $field: ident
                )* $(,)?
            }
        )*
    ) => {
        $(

        #[repr(u8)]
        $(#[doc = $struct_doc])*
        $(#[derive($($struct_der),+)])?
        #[derive(Copy, Clone, Debug, PartialOrd, PartialEq)]
        $enum_vis enum $name {
            $(#[doc = $fst_field_doc])*
            $fst_field,
            $(
                $(#[doc = $field_doc])*
                $field
            ),*
        }

        impl TryFrom<u8> for $name {

            type Error = ();

            fn try_from(value: u8) -> Result<$name, Self::Error> {
                let variants = [$name::$fst_field, $($name::$field),*];
                if (value as usize) < variants.len() {
                    Ok(variants[value as usize])
                } else {
                    Err(())
                }
            }
        }

        impl TryFrom<u16> for $name {

            type Error = ();

            fn try_from(value: u16) -> Result<$name, Self::Error> {
                $name::try_from(value as u8)
            }
        }

        impl TryFrom<u32> for $name {

            type Error = ();

            fn try_from(value: u32) -> Result<$name, Self::Error> {
                $name::try_from(value as u8)
            }
        }

        impl TryFrom<u64> for $name {

            type Error = ();

            fn try_from(value: u64) -> Result<$name, Self::Error> {
                $name::try_from(value as u8)
            }
        }

        impl TryFrom<u128> for $name {

            type Error = ();

            fn try_from(value: u128) -> Result<$name, Self::Error> {
                $name::try_from(value as u8)
            }
        }

        impl From<$name> for u8 {
            fn from(value: $name) -> u8 {
                value as u8
            }
        }

        impl From<$name> for u16 {
            fn from(value: $name) -> u16 {
                (value as u8) as u16
            }
        }

        impl From<$name> for u32 {
            fn from(value: $name) -> u32 {
                (value as u8) as u32
            }
        }
        impl From<$name> for u64 {
            fn from(value: $name) -> u64 {
                (value as u8) as u64
            }
        }

        impl From<$name> for u128 {
            fn from(value: $name) -> u128 {
                (value as u8) as u128
            }
        }

        )*

    };
}
