// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

//! Handle (azimuth, elevation) coordinates.
//!
//! Most of this was blatently stolen (with permission) from [Chris Jordan](https://github.com/cjordan)

use mwalib::MWA_LATITUDE_RADIANS;
use std::f64::consts::FRAC_PI_2;

use super::hadec::HADec;

/// A struct containing an Azimuth and Elevation. All units are in radians.
#[derive(Clone, Debug)]
pub struct AzEl {
    /// Hour angle \[radians\]
    pub az: f64,
    /// Declination \[radians\]
    pub el: f64,
}

impl AzEl {
    /// Make a new `AzEl` struct from values in radians.
    pub fn new(az_rad: f64, el_rad: f64) -> Self {
        Self {
            az: az_rad,
            el: el_rad,
        }
    }

    /// Make a new `AzEl` struct from values in degrees.
    pub fn new_degrees(az_deg: f64, el_deg: f64) -> Self {
        Self::new(az_deg.to_radians(), el_deg.to_radians())
    }

    /// Get the zenith angle in radians.
    pub fn za(&self) -> f64 {
        FRAC_PI_2 - self.el
    }

    /// Convert the horizon coordinates to equatorial coordinates (Hour Angle
    /// and Declination), given the local latitude on Earth.
    ///
    /// Uses ERFA.
    pub fn to_hadec(&self, latitude_rad: f64) -> HADec {
        let mut ha = 0.0;
        let mut dec = 0.0;
        unsafe { erfa_sys::eraAe2hd(self.az, self.el, latitude_rad, &mut ha, &mut dec) }
        HADec { ha, dec }
    }

    /// Convert the horizon coordinates to equatorial coordinates (Hour Angle
    /// and Declination) for the MWA's location.
    pub fn to_hadec_mwa(&self) -> HADec {
        Self::to_hadec(self, MWA_LATITUDE_RADIANS)
    }
}

impl std::fmt::Display for AzEl {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        write!(
            f,
            "({:.4}°, {:.4}°)",
            self.az.to_degrees(),
            self.el.to_degrees()
        )
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use approx::*;

    #[test]
    fn to_hadec() {
        let ae = AzEl::new_degrees(45.0, 30.0);
        let hd = ae.to_hadec(-0.497600);
        assert_abs_diff_eq!(hd.ha, -0.6968754873551053, epsilon = 1e-10);
        assert_abs_diff_eq!(hd.dec, 0.3041176697804004, epsilon = 1e-10);
    }

    #[test]
    fn to_hadec2() {
        let ae = AzEl::new(0.261700, 0.785400);
        let hd = ae.to_hadec(-0.897600);
        assert_abs_diff_eq!(hd.ha, -0.185499449332533, epsilon = 1e-10);
        assert_abs_diff_eq!(hd.dec, -0.12732312479328656, epsilon = 1e-10);
    }

    #[test]
    fn test_za() {
        let ae = AzEl::new(0.261700, 0.785400);
        let za = ae.za();
        assert_abs_diff_eq!(za, 0.7853963268, epsilon = 1e-10);
    }

    #[test]
    fn test_to_hadec_mwa() {
        let ae = AzEl::new(0.5, 0.5);
        let hd_mwa = ae.to_hadec_mwa();
        let hd = ae.to_hadec(mwalib::MWA_LATITUDE_RADIANS);
        assert_eq!(format!("{}", &hd_mwa), format!("{}", &hd));
        assert_abs_diff_eq!(hd.ha, hd_mwa.ha, epsilon = 1e-10);
        assert_abs_diff_eq!(hd.dec, hd_mwa.dec, epsilon = 1e-10);
    }
}
